<?php

namespace Drupal\browsersync\Hook;

use Drupal\browsersync\BrowsersyncHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Hook implementations for Browsersync.
 */
class BrowsersyncHooks {

  /**
   * BrowsersyncHooks constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The 'config.factory' service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user account.
   * @param \Drupal\browsersync\BrowsersyncHelper $browsersyncHelper
   *   The 'browsersync.helper' service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected AccountProxyInterface $currentUser,
    protected BrowsersyncHelper $browsersyncHelper,
  ) {}

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme($existing, $type, $theme, $path): array {
    return [
      'browsersync_snippet' => [
        'variables' => [
          'host' => BrowsersyncHelper::BROWSERSYNC_DEFAULT_HOST,
          'port' => BrowsersyncHelper::BROWSERSYNC_DEFAULT_PORT,
        ],
      ],
    ];
  }

  /**
   * Implements hook_css_alter().
   *
   * Browsersync does not work with CSS import so we need to force Drupal to
   * embed CSS files as <link> elements.
   *
   * @link https://github.com/shakyShane/browser-sync/issues/10
   */
  #[Hook('css_alter')]
  public function cssAlter(&$css): void {
    $browsersync_enabled = $this->browsersyncHelper
      ->browsersyncGetSetting('enabled');

    $system_css_preprocess = $this->configFactory
      ->get('system.performance')
      ->get('css.preprocess');

    if ($browsersync_enabled && !$system_css_preprocess) {
      foreach ($css as $key => $value) {
        // Skip core files.
        if (strpos($value['data'], 'core/') !== 0) {
          $css[$key]['preprocess'] = FALSE;
        }
      }
    }
  }

  /**
   * Implements hook_page_bottom().
   *
   * Adds the Browsersync snippet to the bottom of the page.
   */
  #[Hook('page_bottom')]
  public function pageBottom(array &$page_bottom): void {
    $browsersync_enabled = $this->browsersyncHelper->browsersyncGetSetting('enabled');

    if ($browsersync_enabled && $this->currentUser->hasPermission('use browsersync')) {
      $page_bottom['browsersync'] = [
        '#theme' => 'browsersync_snippet',
        '#weight' => 100,
      ];
      foreach (['host', 'port'] as $setting) {
        if ($value = $this->browsersyncHelper->browsersyncGetSetting($setting)) {
          $page_bottom['browsersync']['#' . $setting] = $value;
        }
      }
    }
  }

}
