<?php

namespace Drupal\bsky\Plugin\Action;

use Drupal\bsky\PostService;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Add Facets to a BlueSky Postobject
 *
 * @Action(
 *   id = "bsky_add_facets",
 *   label = @Translation("BlueSky: Add Facets"),
 *   description = @Translation("Add facets to a Post object. Facets create links for handles, hashtags, and URLs.")
 * )
 */
class AddFacets extends ConfigurableActionBase {

  /**
   * The Drupal PostService service.
   */
  protected PostService $postService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->postService = $container->get('bsky.post_service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'post' => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['post'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Post'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['post'],
      '#description' => $this->t('A previously created Post object.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['post'] = $form_state->getValue('post');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $token = $this->tokenService;
    $post = unserialize($token->getTokenData($this->configuration['post']));
    $post = $this->postService->addFacets($post);
    $token->addTokenData($this->configuration['post'], serialize($post));
  }
}
