<?php

namespace Drupal\bsky\Plugin\Action;

use Drupal\bsky\PostService;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Add an image to a BlueSky post object.
 *
 * @Action(
 *   id = "bsky_add_image",
 *   label = @Translation("BlueSky: Add Image"),
 *   description = @Translation("Add an image to a Post object.")
 * )
 */
class AddImage extends ConfigurableActionBase {

  /**
   * The Drupal PostService service.
   */
  protected PostService $postService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->postService = $container->get('bsky.post_service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'post' => '',
      'image' => '',
      'alt_text' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['post'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Post'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['post'],
      '#description' => $this->t('A previously created Post object.'),
    ];
    $form['image'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Image'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['image'],
      '#description' => $this->t('Path to an image to upload to BlueSky, attached to the Post.'),
    ];
    $form['alt_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Alt Text'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['alt_text'],
      '#description' => $this->t('Alt text to attach to the image.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['post'] = $form_state->getValue('post');
    $this->configuration['image'] = $form_state->getValue('image');
    $this->configuration['alt_text'] = $form_state->getValue('alt_text');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $token = $this->tokenService;
    $post = unserialize($token->getTokenData($this->configuration['post']));
    $image = $token->getOrReplace($this->configuration['image']);
    $alt_text = $token->getOrReplace($this->configuration['alt_text']);
    $post = $this->postService->addImage($post, $image, $alt_text);
    $token->addTokenData($this->configuration['post'], serialize($post));
  }
}
