<?php

namespace Drupal\bsky\Plugin\Action;

use Drupal\bsky\PostService;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Create a BlueSky Post object.
 *
 * @Action(
 *   id = "bsky_create_post",
 *   label = @Translation("BlueSky: Create Post"),
 *   description = @Translation("Create a Post object to send to BlueSky.")
 * )
 */
class CreatePost extends ConfigurableActionBase {

  /**
   * The Drupal PostService service.
   */
  protected PostService $postService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->postService = $container->get('bsky.post_service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'message' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['message'],
      '#description' => $this->t('The message that will be displayed on the post.'),
    ];
    $form['token_name'] = [
      '#required' => TRUE,
      '#type' => 'textfield',
      '#title' => $this->t('Name of response token'),
      '#default_value' => $this->configuration['token_name'],
      '#description' => $this->t('The response value will be loaded into this specified token. You can access [token_name:body] raw response, [token_name:json] when return is valid json, [token_name:headers], [token_name:status], [token_name:client_error] > 0 when has problem.',),
      '#eca_token_reference' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['message'] = $form_state->getValue('message');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $token = $this->tokenService;
    $message = $this->configuration['message'];
    $message = $token->getOrReplace($message);
    $post = $this->postService->createPost($message);
    $token->addTokenData($this->configuration['token_name'], serialize($post));
  }
}
