<?php

namespace Drupal\bsky\Plugin\Action;

use Drupal\bsky\PostService;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Create a BlueSky Post object.
 *
 * @Action(
 *   id = "bsky_send_post",
 *   label = @Translation("BlueSky: Send Post"),
 *   description = @Translation("Send a Post object to BlueSky.")
 * )
 */
class SendPost extends ConfigurableActionBase {

  /**
   * The Drupal PostService service.
   */
  protected PostService $postService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->postService = $container->get('bsky.post_service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'post' => '',
        'token_name' => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['post'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Post'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['post'],
      '#description' => $this->t('The Post to send to BlueSky.'),
    ];
    $form['token_name'] = [
      '#required' => TRUE,
      '#type' => 'textfield',
      '#title' => $this->t('Name of response token'),
      '#default_value' => $this->configuration['token_name'],
      '#description' => $this->t('The response value will be loaded into this specified token.',),
      '#eca_token_reference' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['post'] = $form_state->getValue('post');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $post = unserialize($this->tokenService->getTokenData($this->configuration['post']));
    $response = $this->postService->sendPost($post);
    $this->tokenService->addTokenData($this->configuration['token_name'], (array) $response);
  }
}
