<?php

declare(strict_types=1);

namespace Drupal\bsky;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use potibm\Bluesky\BlueskyApi;
use potibm\Bluesky\BlueskyPostService;
use potibm\Bluesky\Feed\Post;
use potibm\Bluesky\Response\RecordResponse;

/**
 * Main Drupal service wrapping Bluesky library.
 */
final class PostService implements PostServiceInterface {

  /**
   * @var \Drupal\Core\Config\ImmutableConfig Module configuration.
   */
  protected $configuration;

  /**
   * @var BlueskyApi API service library.
   */
  protected $api;

  /**
   * @var BlueskyPostService Library postService.
   */
  protected $postService;

  /**
   * @var bool True if there are credentials found.
   */
  protected $isConfigured;

  public function __construct(
    ConfigFactoryInterface $configFactory,
    KeyRepositoryInterface $keyRepository,
  ) {
    $this->configuration = $configFactory->get('bsky.settings');
    $handle = $this->configuration->get('handle');
    $app_key_name = $this->configuration->get('app_key');
    if (empty($handle) || empty($app_key_name)) {
      $this->isConfigured = FALSE;
      return;
    }
    try {
      $app_key = $keyRepository->getKey($app_key_name);
    }
    catch (\Exception $e) {
      $this->isConfigured = FALSE;
      return;
    }
    $this->api = new BlueskyApi($handle, $app_key->getKeyValue());
    $this->postService = new BlueskyPostService($this->api);
    $this->isConfigured = TRUE;
  }

  /**
   * @inheritDoc
   */
  public function createPost(string $message): Post {
    if (!$this->isConfigured) {
      throw new InvalidPluginDefinitionException('bsky', 'Credentials not found for Bsky service.');
    }
    $post = Post::create($message);
    return $post;
  }

  /**
   * @inheritDoc
   */
  public function addImage(Post $post, string $path, string $alt_text): Post {
    $post = $this->postService->addImage($post, $path, $alt_text);
    return $post;
  }

  /**
   * @inheritDoc
   */
  public function addCard(Post $post, string $url, string $title, string $description, string $imagePath = NULL): Post {
    $post = $this->postService->addWebsiteCard($post, $url, $title, $description, $imagePath);
    return $post;
  }

  /**
   * @inheritDoc
   */
  public function addFacets(Post $post): Post {
    $post = $this->postService->addFacetsFromMentionsAndLinksAndTags($post);
    return $post;
  }

  /**
   * @inheritDoc
   */
  public function sendPost(Post $post): RecordResponse {
    return $this->api->createRecord($post);
  }
}
