<?php

namespace Drupal\bulk_image_regeneration\Form;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\image\Entity\ImageStyle;

/**
 * Provides a form for bulk image regeneration.
 */
class BulkImageRegenerationForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bulk_image_regeneration_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['description'] = [
      '#markup' => $this->t('<p>This tool will regenerate image derivatives for existing images. Select which image styles to regenerate and configure processing options.</p>'),
    ];

    // Get all available image styles
    $image_styles = ImageStyle::loadMultiple();
    $style_options = [];
    foreach ($image_styles as $style) {
      $style_options[$style->id()] = $style->label();
    }

    $form['image_styles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Image Styles to Regenerate'),
      '#description' => $this->t('Select which image styles should be regenerated. Processing all 37 styles at once for 1000+ images can cause performance issues. Select only the styles you need.'),
      '#options' => $style_options,
      '#required' => TRUE,
    ];

    // File extensions are now handled automatically by content field selection
    // No need for manual extension input since content fields are image fields

    // Specific files field removed - content fields provide better targeting

    // Get available image fields from config
    $image_fields = $this->getAvailableImageFieldsFromConfig();

    if (!empty($image_fields)) {
      $form['content_fields'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Content Fields (Priority: Medium)'),
        '#description' => $this->t('Select specific image fields to regenerate images used in those fields. Takes priority over extension scanning but not over specific files.'),
        '#options' => $image_fields,
      ];
    }

    $form['batch_size'] = [
      '#type' => 'select',
      '#title' => $this->t('Batch Size'),
      '#description' => $this->t('Number of images to process per batch operation.'),
      '#options' => [
        5 => '5',
        10 => '10',
        25 => '25',
        50 => '50',
        100 => '100',
      ],
      '#default_value' => 25,
    ];

    $form['dry_run'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Dry Run'),
      '#description' => $this->t('Check this to preview what would be processed without actually regenerating images.'),
      '#default_value' => FALSE,
    ];

    $form['handle_orphaned_files'] = [
      '#type' => 'select',
      '#title' => $this->t('Handle Orphaned Files'),
      '#description' => $this->t('What to do with files that exist on disk but are missing database records.'),
      '#options' => [
        'skip' => $this->t('Skip orphaned files'),
        'create_entity' => $this->t('Create file entities for orphaned files'),
        'delete' => $this->t('Delete orphaned files from disk'),
      ],
      '#default_value' => 'skip',
      '#states' => [
        'visible' => [
          ':input[name="dry_run"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Start Regeneration'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $content_fields = $form_state->getValue('content_fields', []);

    // Require content fields to be selected
    if (empty($content_fields)) {
      $form_state->setError($form['content_fields'], $this->t('Please select at least one content field to process.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $selected_styles = array_filter($form_state->getValue('image_styles'));
    $content_fields = array_filter($form_state->getValue('content_fields', []));
    $batch_size = $form_state->getValue('batch_size');
    $dry_run = $form_state->getValue('dry_run');
    $handle_orphaned_files = $form_state->getValue('handle_orphaned_files');

    if (empty($selected_styles)) {
      $this->messenger()->addError($this->t('Please select at least one image style to regenerate.'));
      return;
    }

    // Get files to process and collect diagnostics for reporting.
    $field_diagnostics = [];
    $files = $this->getFilesFromContentFields($content_fields, $field_diagnostics);

    if (empty($files)) {
      $this->messenger()->addWarning($this->t('No image files found to process.'));
      return;
    }

    $total_images = count($files);

    // Aggregate diagnostics to help operators understand skipped references.
    $aggregate = [
      'total_references' => 0,
      'missing_entities' => 0,
      'missing_file_entities' => 0,
      'temporary_file_entities' => 0,
    ];
    foreach ($field_diagnostics as $stats) {
      $aggregate['total_references'] += $stats['total_references'];
      $aggregate['missing_entities'] += $stats['missing_entities']['count'];
      $aggregate['missing_file_entities'] += $stats['missing_file_entities']['count'];
      $aggregate['temporary_file_entities'] += $stats['temporary_file_entities']['count'];
    }
    $unresolved_references = max(0, $aggregate['total_references'] - $total_images);

    if ($aggregate['total_references'] > 0) {
      if ($unresolved_references > 0) {
        $this->messenger()->addWarning($this->t('Found @resolved usable images out of @references stored references. @skipped references will be skipped (missing profiles: @missing_profiles, missing file entities: @missing_files, temporary files: @temporary_files).', [
          '@resolved' => $total_images,
          '@references' => $aggregate['total_references'],
          '@skipped' => $unresolved_references,
          '@missing_profiles' => $aggregate['missing_entities'],
          '@missing_files' => $aggregate['missing_file_entities'],
          '@temporary_files' => $aggregate['temporary_file_entities'],
        ]));
      }
      else {
        $this->messenger()->addStatus($this->t('All @count referenced images are available for processing.', [
          '@count' => $total_images,
        ]));
      }
    }

    $total_operations = $total_images * count($selected_styles);
    $this->messenger()->addMessage($this->t('Starting regeneration of @count styles for @files images from selected content fields (@total total operations). Processing one style at a time for better performance.',
      ['@count' => count($selected_styles), '@files' => $total_images, '@total' => $total_operations]));

    // Log the start of regeneration
    \Drupal::logger('bulk_image_regeneration')->info('Starting bulk image regeneration. Content fields: @fields, Image styles: @styles, Files: @files, Total operations: @total',
      [
        '@fields' => implode(', ', $content_fields),
        '@styles' => implode(', ', $selected_styles),
        '@files' => $total_images,
        '@total' => $total_operations,
      ]);

    // Prepare batch operation
    $batch_builder = new BatchBuilder();
    $batch_builder->setTitle($this->t('Bulk Image Regeneration'));
    $batch_builder->setInitMessage($this->t('Starting regeneration of @count styles for @files files...', [
      '@count' => count($selected_styles),
      '@files' => count($files)
    ]));
    $batch_builder->setProgressMessage($this->t('Processing chunk @current of @total. Image counts are shown above.'));
    $batch_builder->setErrorMessage($this->t('An error occurred during processing.'));

    // Process one style at a time for better performance
    foreach ($selected_styles as $style_id) {
      \Drupal::logger('bulk_image_regeneration')->info('Starting regeneration for style: @style', ['@style' => $style_id]);

      // Split files into batches for this style
      $batches = array_chunk($files, $batch_size, TRUE);

      foreach ($batches as $batch_files) {
        $batch_builder->addOperation(
          [static::class, 'processBatch'],
          [$batch_files, [$style_id], $dry_run, $total_images, $aggregate['total_references']]
        );
      }
    }

    $batch_builder->setFinishCallback([static::class, 'finishBatch']);

    batch_set($batch_builder->toArray());
  }

  /**
   * Get available image fields from config.
   */
  protected function getAvailableImageFieldsFromConfig(): array {
    $fields = [];

    try {
      // Query config table for field configurations
      $query = \Drupal::database()->select('config', 'c')
        ->fields('c', ['name', 'data'])
        ->condition('c.name', 'field.field.%', 'LIKE')
        ->condition('c.collection', '');

      $results = $query->execute();

      foreach ($results as $result) {
        $config_data = unserialize($result->data);

        // Check if this is an image field
        if (isset($config_data['field_type']) && $config_data['field_type'] === 'image') {
          // Parse the field name to get entity type, bundle, and field name
          // Format: field.field.{entity_type}.{bundle}.{field_name}
          $parts = explode('.', $result->name);
          if (count($parts) >= 5) {
            $entity_type = $parts[2];
            $bundle = $parts[3];
            $field_name = $parts[4];

            $label = isset($config_data['label']) ? $config_data['label'] : $field_name;
            $key = $entity_type . '.' . $bundle . '.' . $field_name;

            // Create a readable label
            $readable_label = ucfirst($entity_type) . ': ' . $bundle . ' - ' . $label . ' (' . $field_name . ')';
            $fields[$key] = $readable_label;
          }
        }
      }

      // Sort fields alphabetically
      asort($fields);

    } catch (\Exception $e) {
      \Drupal::logger('bulk_image_regeneration')->error('Error loading image fields from config: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    return $fields;
  }

  /**
   * Get files from selected content fields.
   */
  protected function getFilesFromContentFields(array $selected_fields, ?array &$diagnostics = NULL): array {
    $files = [];
    $collect_diagnostics = is_array($diagnostics);

    foreach ($selected_fields as $field_key) {
      // Parse the field key: entity_type.bundle.field_name
      $parts = explode('.', $field_key);
      if (count($parts) === 3) {
        $entity_type = $parts[0];
        $bundle = $parts[1];
        $field_name = $parts[2];

        $field_table = $entity_type . '__' . $field_name;
        $target_column = $field_name . '_target_id';

        if ($collect_diagnostics && !isset($diagnostics[$field_key])) {
          $diagnostics[$field_key] = [
            'total_references' => 0,
            'resolved_fids' => [],
            'missing_entities' => ['count' => 0, 'sample' => []],
            'missing_file_entities' => ['count' => 0, 'sample' => []],
            'temporary_file_entities' => ['count' => 0, 'sample' => []],
          ];

          try {
            $diagnostics[$field_key]['total_references'] = (int) \Drupal::database()
              ->select($field_table, 'f')
              ->condition('bundle', $bundle)
              ->countQuery()
              ->execute()
              ->fetchField();
          } catch (\Exception $e) {
            \Drupal::logger('bulk_image_regeneration')->warning('Unable to count references for @field due to: @error', [
              '@field' => $field_table,
              '@error' => $e->getMessage(),
            ]);
          }
        }

        try {
          $entity_storage = \Drupal::entityTypeManager()->getStorage($entity_type);
          $file_storage = \Drupal::entityTypeManager()->getStorage('file');
          $entity_cache = [];
          $file_cache = [];

          $records = \Drupal::database()
            ->select($field_table, 'f')
            ->fields('f', ['entity_id', $target_column])
            ->condition('bundle', $bundle)
            ->execute();

          foreach ($records as $record) {
            $entity_id = (int) $record->entity_id;
            $fid = $record->{$target_column} !== NULL ? (int) $record->{$target_column} : NULL;

            if (!$fid) {
              if ($collect_diagnostics) {
                $diagnostics[$field_key]['missing_file_entities']['count']++;
                if (count($diagnostics[$field_key]['missing_file_entities']['sample']) < 5) {
                  $diagnostics[$field_key]['missing_file_entities']['sample'][] = NULL;
                }
              }
              continue;
            }

            if (!array_key_exists($entity_id, $entity_cache)) {
              $entity_cache[$entity_id] = $entity_storage->load($entity_id);
            }

            if (!$entity_cache[$entity_id]) {
              if ($collect_diagnostics) {
                $diagnostics[$field_key]['missing_entities']['count']++;
                if (count($diagnostics[$field_key]['missing_entities']['sample']) < 5) {
                  $diagnostics[$field_key]['missing_entities']['sample'][] = $entity_id;
                }
              }
              continue;
            }

            if (!array_key_exists($fid, $file_cache)) {
              $file_cache[$fid] = $file_storage->load($fid);
            }

            $file = $file_cache[$fid];
            if (!$file) {
              if ($collect_diagnostics) {
                $diagnostics[$field_key]['missing_file_entities']['count']++;
                if (count($diagnostics[$field_key]['missing_file_entities']['sample']) < 5) {
                  $diagnostics[$field_key]['missing_file_entities']['sample'][] = $fid;
                }
              }
              continue;
            }

            if (!$file->isPermanent()) {
              if ($collect_diagnostics) {
                $diagnostics[$field_key]['temporary_file_entities']['count']++;
                if (count($diagnostics[$field_key]['temporary_file_entities']['sample']) < 5) {
                  $diagnostics[$field_key]['temporary_file_entities']['sample'][] = $fid;
                }
              }
              continue;
            }

            if ($collect_diagnostics) {
              $diagnostics[$field_key]['resolved_fids'][$fid] = TRUE;
            }

            $files[$fid] = $file->getFileUri();
          }
        } catch (\Exception $e) {
          \Drupal::logger('bulk_image_regeneration')->warning('Error processing field @field: @error', [
            '@field' => $field_key,
            '@error' => $e->getMessage(),
          ]);
        }
        if ($collect_diagnostics && isset($diagnostics[$field_key]['resolved_fids'])) {
          $diagnostics[$field_key]['resolved_count'] = count($diagnostics[$field_key]['resolved_fids']);
          unset($diagnostics[$field_key]['resolved_fids']);
        }
      }
    }

    return $files;
  }


  /**
   * Process a batch of files.
   */
  public static function processBatch(array $files, array $selected_styles, bool $dry_run, int $total_images, int $total_references, array &$context): void {
    $logger = \Drupal::logger('bulk_image_regeneration');

    if (!isset($context['results']['processed'])) {
      $context['results']['processed'] = 0;
      $context['results']['errors'] = 0;
      $context['results']['skipped'] = 0;
      $context['results']['orphaned_files'] = 0;
      $context['results']['relocated_files'] = 0;
      $context['results']['missing_files'] = 0;
      $context['results']['total_images'] = $total_images;
      $context['results']['total_references'] = $total_references;
      $context['results']['processed_unique_fids'] = [];

      $logger->info('Starting bulk image regeneration batch. Files: @files, Styles: @styles, Dry run: @dry',
        [
          '@files' => count($files),
          '@styles' => implode(', ', $selected_styles),
          '@dry' => $dry_run ? 'YES' : 'NO',
        ]);
    }

    foreach ($files as $fid => $uri) {
      try {
        // Load the file
        $file = \Drupal::entityTypeManager()->getStorage('file')->load($fid);
        if (!$file) {
          // Check if the file exists on disk even if the entity is missing
          $real_path = \Drupal::service('file_system')->realpath($uri);
          $file_exists_on_disk = $real_path && file_exists($real_path);

          if ($file_exists_on_disk) {
            \Drupal::logger('bulk_image_regeneration')->warning('Orphaned file found: FID @fid exists on disk but missing database record. URI: @uri, Path: @path', [
              '@fid' => $fid,
              '@uri' => $uri,
              '@path' => $real_path,
            ]);
            $context['results']['orphaned_files']++;

            // For now, skip orphaned files but count them separately
            $context['results']['skipped']++;
            continue;
          } else {
            \Drupal::logger('bulk_image_regeneration')->error('File entity missing and file not found on disk: FID @fid, URI: @uri', [
              '@fid' => $fid,
              '@uri' => $uri,
            ]);
            $context['results']['errors']++;
            continue;
          }
        }

        // Check if the file exists at the expected location (handles storage route changes)
        $real_path = \Drupal::service('file_system')->realpath($uri);
        if (!$real_path || !file_exists($real_path)) {
          // File entity exists but file is missing from expected location
          \Drupal::logger('bulk_image_regeneration')->warning('File entity exists but file missing from disk for FID: @fid, URI: @uri, Expected path: @path', [
            '@fid' => $fid,
            '@uri' => $uri,
            '@path' => $real_path ?: 'N/A',
          ]);

          // Try to find the file in alternative locations
          $found_alternative = static::findFileInAlternativeLocations($uri, $real_path);
          if ($found_alternative) {
            \Drupal::logger('bulk_image_regeneration')->warning('File relocated: FID @fid expected at @expected but found at @alternative (storage route may have changed)', [
              '@fid' => $fid,
              '@expected' => $real_path ?: $uri,
              '@alternative' => $found_alternative,
            ]);

            // Update the URI to the correct location if we found it
            if ($dry_run) {
              $context['results']['relocated_files'] = ($context['results']['relocated_files'] ?? 0) + 1;
            } else {
              // In real mode, we could update the file entity URI here
              // But for now, just log and skip
              $context['results']['relocated_files'] = ($context['results']['relocated_files'] ?? 0) + 1;
            }
          } else {
            \Drupal::logger('bulk_image_regeneration')->error('File completely missing: FID @fid not found at expected location @expected and no alternatives found', [
              '@fid' => $fid,
              '@expected' => $real_path ?: $uri,
            ]);
            $context['results']['missing_files'] = ($context['results']['missing_files'] ?? 0) + 1;
          }

          $context['results']['skipped']++;
          continue;
        }

        // Get applicable image styles
        $image_styles = ImageStyle::loadMultiple();
        $styles_to_process = array_intersect_key($image_styles, array_flip($selected_styles));

        foreach ($styles_to_process as $style) {
          // Skip styles with no effects (like 'civicrm')
          if (empty($style->getEffects()->getConfiguration())) {
            continue;
          }

          if ($dry_run) {
            // Just count what would be processed
            $context['results']['processed']++;
          } else {
            try {
              // Actually generate the derivative
              $style->createDerivative($uri, $style->buildUri($uri));
              $context['results']['processed']++;
            } catch (\Exception $e) {
              \Drupal::logger('bulk_image_regeneration')->warning('Derivative creation failed: FID @fid, URI @uri, Style @style - @error', [
                '@fid' => $fid,
                '@uri' => $uri,
                '@style' => $style->id(),
                '@error' => $e->getMessage(),
              ]);
              // Don't count as error for individual style failures - too noisy
              $context['results']['skipped']++;
            }
          }
        }

        // Track unique image progress for user-facing messaging.
        if (!isset($context['results']['processed_unique_fids'][$fid])) {
          $context['results']['processed_unique_fids'][$fid] = TRUE;
        }

        if (!empty($context['results']['total_images'])) {
          $unique_processed = count($context['results']['processed_unique_fids']);
          $context['message'] = \Drupal::translation()->formatPlural(
            $unique_processed,
            'Processed 1 of @total images (from @references references).',
            'Processed @count of @total images (from @references references).',
            [
              '@total' => $context['results']['total_images'],
              '@references' => $context['results']['total_references'],
            ]
          );
        }

      } catch (\Exception $e) {
        \Drupal::logger('bulk_image_regeneration')->error('Error processing file @uri: @error', [
          '@uri' => $uri,
          '@error' => $e->getMessage(),
        ]);
        $context['results']['errors']++;
      }
    }
  }

  /**
   * Try to find a file in alternative locations when storage routes have changed.
   */
  protected static function findFileInAlternativeLocations(string $original_uri, string|false $expected_real_path): string|false {
    // Parse the original URI to understand the structure
    $parsed_uri = parse_url($original_uri);
    if (!$parsed_uri || !isset($parsed_uri['path'])) {
      return FALSE;
    }

    $path_parts = explode('/', trim($parsed_uri['path'], '/'));
    if (count($path_parts) < 2) {
      return FALSE;
    }

    $scheme = $parsed_uri['scheme'] ?? 'public';
    $relative_path = implode('/', array_slice($path_parts, 1)); // Remove scheme from path

    // Common alternative locations to check
    $alternative_locations = [];

    // Try different schemes
    if ($scheme === 'public') {
      $alternative_locations[] = 'private://' . $relative_path;
    } elseif ($scheme === 'private') {
      $alternative_locations[] = 'public://' . $relative_path;
    }

    // Try common subdirectories or parent directories
    $alternative_locations[] = $scheme . '://files/' . $relative_path;
    $alternative_locations[] = $scheme . '://sites/default/files/' . $relative_path;

    // Try without the first directory level (in case directory structure changed)
    if (count($path_parts) > 2) {
      $alternative_locations[] = $scheme . '://' . implode('/', array_slice($path_parts, 2));
    }

    // Try with common migration patterns
    if (strpos($relative_path, 'profile_assets/') === 0) {
      // Files that moved from profile_assets to somewhere else
      $alternative_locations[] = $scheme . '://' . str_replace('profile_assets/', '', $relative_path);
      $alternative_locations[] = $scheme . '://images/' . str_replace('profile_assets/', '', $relative_path);
      $alternative_locations[] = $scheme . '://pictures/' . str_replace('profile_assets/', '', $relative_path);
    }

    // Check each alternative location
    foreach ($alternative_locations as $alt_uri) {
      $alt_real_path = \Drupal::service('file_system')->realpath($alt_uri);
      if ($alt_real_path && file_exists($alt_real_path)) {
        return $alt_uri; // Return the working URI
      }
    }

    return FALSE; // File not found in any alternative location
  }

  /**
   * Finish batch processing.
   */
  public static function finishBatch(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();
    $logger = \Drupal::logger('bulk_image_regeneration');

    // Log comprehensive results to watchdog
    $log_context = [
      'processed' => $results['processed'] ?? 0,
      'errors' => $results['errors'] ?? 0,
      'skipped' => $results['skipped'] ?? 0,
      'orphaned_files' => $results['orphaned_files'] ?? 0,
      'relocated_files' => $results['relocated_files'] ?? 0,
      'missing_files' => $results['missing_files'] ?? 0,
      'total_operations' => ($results['processed'] ?? 0) + ($results['skipped'] ?? 0),
      'success' => $success,
    ];

    if ($success) {
      $logger->info('Bulk image regeneration completed successfully. Processed: @processed, Errors: @errors, Skipped: @skipped, Orphaned: @orphaned, Relocated: @relocated, Missing: @missing',
        $log_context);

      $messenger->addMessage(\Drupal::translation()->formatPlural(
        $results['processed'] ?? 0,
        'Processed 1 image.',
        'Processed @count images.'
      ));

      if (($results['errors'] ?? 0) > 0) {
        $logger->warning('Bulk image regeneration encountered @errors errors during processing.', ['@errors' => $results['errors']]);
        $messenger->addWarning(\Drupal::translation()->formatPlural(
          $results['errors'] ?? 0,
          '1 error occurred during processing.',
          '@count errors occurred during processing.'
        ));
      }

      if (($results['skipped'] ?? 0) > 0) {
        $logger->info('Bulk image regeneration skipped @skipped images.', ['@skipped' => $results['skipped']]);
        $messenger->addStatus(\Drupal::translation()->formatPlural(
          $results['skipped'] ?? 0,
          '1 image was skipped.',
          '@count images were skipped.'
        ));
      }

      if (isset($results['orphaned_files']) && $results['orphaned_files'] > 0) {
        $logger->warning('Bulk image regeneration found @orphaned orphaned files (exist on disk but missing database records).',
          ['@orphaned' => $results['orphaned_files']]);
        $messenger->addWarning(\Drupal::translation()->formatPlural(
          $results['orphaned_files'],
          '1 orphaned file found (exists on disk but missing database record).',
          '@count orphaned files found (exist on disk but missing database records).'
        ) . ' ' . \Drupal::translation()->translate('Consider running database cleanup or file entity repair.'));
      }

      if (isset($results['relocated_files']) && $results['relocated_files'] > 0) {
        $logger->warning('Bulk image regeneration found @relocated files in alternative locations (storage routes may have changed).',
          ['@relocated' => $results['relocated_files']]);
        $messenger->addWarning(\Drupal::translation()->formatPlural(
          $results['relocated_files'],
          '1 file found in alternative location (storage route may have changed).',
          '@count files found in alternative locations (storage routes may have changed).'
        ) . ' ' . \Drupal::translation()->translate('Consider updating file URIs or running file migration.'));
      }

      if (isset($results['missing_files']) && $results['missing_files'] > 0) {
        $logger->error('Bulk image regeneration found @missing completely missing files from disk.',
          ['@missing' => $results['missing_files']]);
        $messenger->addError(\Drupal::translation()->formatPlural(
          $results['missing_files'],
          '1 file is completely missing from disk.',
          '@count files are completely missing from disk.'
        ) . ' ' . \Drupal::translation()->translate('These files cannot be processed and may need to be restored from backups.'));
      }
    } else {
      $logger->error('Bulk image regeneration batch processing failed.', $log_context);
      $messenger->addError(t('Batch processing failed.'));
    }
  }

}
