<?php

namespace Drupal\bulk_term_delete\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\Entity\Term;

class BulkTermDeleteForm extends FormBase {

  public function getFormId() {
    return 'bulk_term_delete_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $vocabulary = NULL) {
    $terms = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->loadTree($vocabulary);

    if (empty($terms)) {
      $form['message'] = [
        '#markup' => $this->t('No terms found in this vocabulary.'),
      ];
      return $form;
    }

    $header = [
      'select' => $this->t('Select'),
      'name' => $this->t('Term name'),
      'tid' => $this->t('ID'),
    ];

    $options = [];
    foreach ($terms as $term) {
      $options[$term->tid] = [
        'name' => $term->name,
        'tid' => $term->tid,
      ];
    }

    $form['terms_table'] = [
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $options,
      '#empty' => $this->t('No terms available.'),
      '#multiple' => TRUE,
      '#default_value' => [],
    ];

    $form['actions']['delete'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete selected terms'),
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected = array_filter($form_state->getValue('terms_table'));

    if (empty($selected)) {
      $this->messenger()->addWarning($this->t('No terms selected.'));
      return;
    }

    $deleted = 0;
    foreach ($selected as $tid) {
      $term = Term::load($tid);
      if ($term) {
        $term->delete();
        $deleted++;
      }
    }
    $this->messenger()->addMessage($this->t('@count term(s) deleted.', ['@count' => $deleted]));

    // Redirect back to taxonomy overview.
    $vocabulary = \Drupal::routeMatch()->getParameter('vocabulary');
    if ($vocabulary) {
      $form_state->setRedirect('entity.taxonomy_vocabulary.overview_form', ['taxonomy_vocabulary' => $vocabulary]);
    }
  }
}
