<?php

namespace Drupal\bulk_term_delete\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;

class BulkTermDeleteForm extends FormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs the BulkTermDeleteForm.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, RouteMatchInterface $route_match) {
    $this->entityTypeManager = $entity_type_manager;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bulk_term_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, string $vocabulary = NULL): array {

    $terms = $this->entityTypeManager
      ->getStorage('taxonomy_term')
      ->loadTree($vocabulary);

    if (empty($terms)) {
      $form['message'] = [
        '#markup' => $this->t('No terms found in this vocabulary.'),
      ];
      return $form;
    }

    $header = [
      'name' => $this->t('Term name'),
      'tid' => $this->t('ID'),
    ];

    $options = [];
    foreach ($terms as $term) {
      $options[$term->tid] = [
        'name' => $term->name,
        'tid' => $term->tid,
      ];
    }

    $form['terms_table'] = [
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $options,
      '#empty' => $this->t('No terms available.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['delete'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete selected terms'),
      '#button_type' => 'danger',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */

  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected = array_filter($form_state->getValue('terms_table'));

    if (empty($selected)) {
      $this->messenger()->addWarning($this->t('No terms selected.'));
      return;
    }

    $storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $deleted = 0;

    foreach ($selected as $tid) {
      /** @var \Drupal\taxonomy\Entity\Term|null $term */
      $term = $storage->load($tid);
      if ($term) {
        $term->delete();
        $deleted++;
      }
    }

    $this->messenger()->addStatus(
      $this->t('@count term(s) deleted.', ['@count' => $deleted])
    );

    // Redirect back to taxonomy overview.
    $vocabulary = $this->routeMatch->getParameter('vocabulary');
    if ($vocabulary) {
      $form_state->setRedirect(
        'entity.taxonomy_vocabulary.overview_form',
        ['taxonomy_vocabulary' => $vocabulary]
      );
    }
  }
}
