<?php

namespace Drupal\Tests\rabbitmq\Unit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\rabbitmq\ConnectionFactory;
use Drupal\rabbitmq\Queue\Queue;
use Drupal\Tests\UnitTestCase;
use PhpAmqpLib\Channel\AMQPChannel;
use PhpAmqpLib\Connection\AbstractConnection;
use PhpAmqpLib\Message\AMQPMessage;
use Psr\Log\LoggerInterface;

/**
 * Tests the queue class.
 *
 * @group rabbitmq
 *
 * @coversDefaultClass \Drupal\rabbitmq\Queue\Queue
 */
class QueueTest extends UnitTestCase {

  /**
   * Mock AMPQChannel.
   *
   * @var \PhpAmqpLib\Channel\AMQPChannel|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $channelMock;

  /**
   * Mock Connection.
   *
   * @var \PhpAmqpLib\Connection\AbstractConnection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $connectionMock;

  /**
   * Mock connection factory.
   *
   * @var \Drupal\rabbitmq\ConnectionFactory|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $connectionFactoryMock;

  /**
   * Mock module_handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandlerMock;

  /**
   * Logger service mock.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject|\Psr\Log\LoggerInterface
   */
  protected $loggerMock;

  /**
   * Mock queue config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configMock;

  /**
   * Mock UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $uuidMock;

  /**
   * Mock time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $timeMock;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // RabbitMQ Connection/Channel basic mocking.
    $this->channelMock = $this->createMock(AMQPChannel::class);
    $this->connectionMock = $this->createMock(AbstractConnection::class);
    $this->connectionMock->method('getConnection')->willReturnSelf();
    $this->connectionMock->method('channel')->willReturn($this->channelMock);
    $this->connectionFactoryMock = $this->createMock(ConnectionFactory::class);
    $this->connectionFactoryMock->method('getConnection')->willReturn($this->connectionMock);

    // Additional services required by the Queue class.
    $this->moduleHandlerMock = $this->createMock(ModuleHandlerInterface::class);
    $this->loggerMock = $this->createMock(LoggerInterface::class);
    $this->configMock = $this->createMock(ImmutableConfig::class);

    $this->uuidMock = $this->createMock(UuidInterface::class);
    $this->uuidMock->method('generate')->willReturn('00643728-8a9e-48bb-b337-db14ad1f3f2c');

    $this->timeMock = $this->createMock(TimeInterface::class);
    $this->timeMock->method('getCurrentTime')->willReturn(1600000000);

  }

  /**
   * Test that createItem handles item_id.
   *
   * @covers ::createItem
   */
  public function testCreateItemId() {
    $data = 'foo';

    $submitted_message = NULL;

    $this->channelMock
      ->expects($this->once())
      ->method('basic_publish')
      ->with(
        self::callback(
          function ($message) use (&$submitted_message): bool {
            $submitted_message = $message;
            return TRUE;
          }
        )
      );

    $queue = $this->getQueue();

    $item_id = $queue->createItem($data);
    $this->assertEquals('00643728-8a9e-48bb-b337-db14ad1f3f2c', $submitted_message->get('message_id'), "Message has message_id header");
    $this->assertEquals('00643728-8a9e-48bb-b337-db14ad1f3f2c', $item_id, "createItem returns correct item_id");
  }

  /**
   * Test that claimItem returns message created timestamp.
   *
   * @param ?string $return_value
   *   Response to return for $msg->get('message_id')
   * @param string $expected_id
   *   Expected item_id value.
   *
   * @dataProvider itemIdDataProvider
   *
   * @covers ::claimItem
   */
  public function testItemIdFromClaimItem(?string $return_value, string $expected_id) {
    $options['message_id'] = $return_value;
    assert(json_encode('foo') !== FALSE);
    $message = new AMQPMessage(json_encode('foo'), $options);
    $message->setDeliveryTag(1);
    $this->channelMock->method('basic_get')->willReturn($message);

    $queue = $this->getQueue();

    $item = $queue->claimItem();
    $this->assertNotFalse($item);
    $this->assertEquals($expected_id, $item->item_id);
  }

  /**
   * DataProvider for testItemIdFromClaimItem().
   *
   * @return array
   *   Array of test data.
   */
  public static function itemIdDataProvider(): array {
    return [
      'No message_id set' => [
        NULL,
        '00643728-8a9e-48bb-b337-db14ad1f3f2c',
      ],
      'With message_id' => [
        '38e14be6-6e21-4992-9eec-c1624f5f783b',
        '38e14be6-6e21-4992-9eec-c1624f5f783b',
      ],
    ];
  }

  /**
   * Test that createItem returns message created timestamp.
   *
   * @covers ::createItem
   */
  public function testCreateItemTimestamp() {
    $submitted_message = NULL;
    $data = 'foo';

    $this->channelMock
      ->expects($this->once())
      ->method('basic_publish')
      ->with(
        self::callback(
          function ($message) use (&$submitted_message): bool {
            $submitted_message = $message;
            return TRUE;
          }
        )
      );

    $queue = $this->getQueue();

    $queue->createItem($data);
    $this->assertEquals(1600000000, $submitted_message->get('timestamp'), "Message has timestamp");
  }

  /**
   * Test that claimItem returns message created timestamp.
   *
   * @param ?int $return_value
   *   Value to set for message timestamp.
   * @param int $expected_timestamp
   *   Expected created timestamp value.
   *
   * @dataProvider timestampDataProvider
   *
   * @covers ::claimItem
   */
  public function testClaimItemTimestamp(?int $return_value, int $expected_timestamp) {
    $options['timestamp'] = $return_value;
    assert(json_encode('foo') !== FALSE);
    $message = new AMQPMessage(json_encode('foo'), $options);
    $message->setDeliveryTag(1);
    $this->channelMock->method('basic_get')->willReturn($message);

    $queue = $this->getQueue();

    $item = $queue->claimItem();
    $this->assertNotFalse($item);
    $this->assertEquals($expected_timestamp, $item->created);
  }

  /**
   * DataProvider for testClaimItemTimestamp().
   *
   * @return array
   *   Array of test data.
   */
  public static function timestampDataProvider(): array {
    return [
      'No timestamp set' => [
        NULL,
        0,
      ],
      'With timestamp' => [
        1600000000,
        1600000000,
      ],
    ];
  }

  /**
   * Helper to obtain a Queue using the mocked services.
   *
   * @return \Drupal\rabbitmq\Queue\Queue
   *   A queue using the mocks previously configured.
   */
  protected function getQueue(): Queue {
    return new Queue(
      'unittest_queue',
      $this->connectionFactoryMock,
      $this->moduleHandlerMock,
      $this->loggerMock,
      $this->configMock,
      $this->uuidMock,
      $this->timeMock
    );
  }

}
