<?php

namespace Drupal\bunny;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\Core\Site\Settings;
use Drupal\bunny\Queue\QueueFactory;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Modifies the language manager service.
 */
class BunnyServiceProvider extends ServiceProviderBase {

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container): void {
    $credentials = Settings::get('bunny_credentials');
    if (empty($credentials)) {
      // Fallback to the RabbitMQ configuration.
      $credentials = Settings::get('rabbitmq_credentials');
    }
    if (!empty($credentials)) {
      foreach ($credentials as $key => $values) {
        $connectionFactoryServiceId = 'bunny.connection.factory.' . $key;
        $connectionFactory = new Definition(ConnectionFactory::class, [
          new Reference('settings'),
          $key,
        ]);
        $connectionFactory->setPublic(TRUE);
        $container->setDefinition($connectionFactoryServiceId, $connectionFactory);

        $queueFactory = new Definition(QueueFactory::class, [
          new Reference($connectionFactoryServiceId),
          new Reference('module_handler'),
          new Reference('logger.channel.bunny'),
          new Reference('config.factory'),
          new Reference('uuid'),
          new Reference('datetime.time'),
          new Reference('bunny.serializers'),
        ]);
        $queueFactory->setPublic(TRUE);
        $queueFactory->addTag('queue_factory');
        $container->setDefinition('queue.bunny.' . $key, $queueFactory);

        // Holdover to allow configuration migration.
        $deprecatedQueueFactory = new Definition(QueueFactory::class, [
          new Reference($connectionFactoryServiceId),
          new Reference('module_handler'),
          new Reference('logger.channel.bunny'),
          new Reference('config.factory'),
          new Reference('uuid'),
          new Reference('datetime.time'),
          new Reference('bunny.serializers'),
        ]);
        $deprecatedQueueFactory->setPublic(TRUE);
        $deprecatedQueueFactory->addTag('queue_factory');
        $deprecatedQueueFactory->setDeprecated('drupal/bunny', '2.0.0', '%service_id% is deprecated, use queue.bunny.* services instead. Will be removed from bunny 3.x.');
        $container->setDefinition('queue.rabbitmq.' . $key, $deprecatedQueueFactory);
      }
    }
  }

}
