<?php

declare(strict_types=1);

namespace Drupal\Tests\bunny\Functional;

use Drupal\bunny\Queue\Queue;
use Drupal\Tests\BrowserTestBase;
use PHPUnit\Framework\Attributes\Group;

/**
 * Basic E2E Test of module installed and processing messages.
 *
 * @group bunny
 */
#[Group('bunny')]
final class BasicE2ETest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['bunny'];

  /**
   * The queue name for this test run.
   */
  protected string $queueName;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $settings['settings']['bunny_credentials']['default'] = [
      'host' => (object) [
        'value' => 'rabbitmq',
        'required' => TRUE,
      ],
      'port' => (object) [
        'value' => '5672',
        'required' => TRUE,
      ],
      'vhost' => (object) [
        'value' => '/',
        'required' => TRUE,
      ],
      'username' => (object) [
        'value' => 'guest',
        'required' => TRUE,
      ],
      'password' => (object) [
        'value' => 'guest',
        'required' => TRUE,
      ],
    ];

    // Set Bunny as the default queue handler.
    $settings['settings']['queue_default'] = (object) [
      'value' => 'queue.bunny.default',
      'required' => TRUE,
    ];

    $this->writeSettings($settings);

    // BunnyServiceProvider requires a container rebuild to process the
    // settings configured above.
    /** @var \Drupal\Core\DrupalKernelInterface $kernel */
    $kernel = \Drupal::service('kernel');
    $kernel->invalidateContainer();
    $kernel->rebuildContainer();

    $time = $this->container->get('datetime.time')->getCurrentTime();
    $this->queueName = 'test-' . date('c', $time) . '-' . $this->randomMachineName(8);

    $config = $this->config('bunny.config');
    $queues = $config->get('queues');
    assert(is_array($queues));
    $queues[$this->queueName] = [
      'passive' => FALSE,
      'durable' => TRUE,
      'exclusive' => FALSE,
      'auto_delete' => FALSE,
      'name' => 'test.queue',
      'nowait' => FALSE,
      'routing_keys' => [],
    ];
    $config->set('queues', $queues)->save();
  }

  /**
   * Test callback.
   */
  public function testBasicInstallAndFunction(): void {
    /** @var \Drupal\Core\Queue\QueueInterface $queue */
    $queue = \Drupal::service('queue')->get($this->queueName);
    $this->assertInstanceOf(Queue::class, $queue);
    $queue->createItem('test');
    $item = $queue->claimItem();
    $this->assertNotFalse($item);
    $this->assertObjectHasProperty('data', $item);
    $this->assertSame('test', $item->data);
  }

  /**
   * {@inheritdoc}
   */
  public function tearDown(): void {
    /** @var \Drupal\bunny\Queue\Queue $queue */
    $queue = \Drupal::service('queue')->get($this->queueName);
    $queue->getChannel()->queue_delete($this->queueName);
    parent::tearDown();
  }

}
