<?php

namespace Drupal\Tests\bunny\Kernel;

use Drupal\Core\Site\Settings;
use Drupal\KernelTests\KernelTestBase;
use Drupal\bunny\Queue\QueueBase;
use Drupal\bunny\Queue\QueueFactory;

use PhpAmqpLib\Channel\AMQPChannel;
use PhpAmqpLib\Connection\AMQPStreamConnection;

/**
 * Class BunnyTestBase is a base class for Bunny tests.
 */
abstract class BunnyTestBase extends KernelTestBase {
  const MODULE = 'bunny';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', QueueBase::MODULE];

  /**
   * Server factory.
   *
   * @var \Drupal\bunny\ConnectionFactory
   */
  protected $connectionFactory;

  /**
   * The name requested for the temporary queue created during tests.
   *
   * @var string
   */
  protected $queueName;

  /**
   * The routing key, actually equal to the queue name, but not necessarily so.
   *
   * @var string
   */
  protected $routingKey;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $settings = Settings::getAll();
    $settings['bunny_credentials']['default'] = [
      'host' => 'rabbitmq',
      'port' => 5672,
      'vhost' => '/',
      'username' => 'guest',
      'password' => 'guest',
    ];
    // Set Bunny as the default queue handler.
    $settings['queue_default'] = 'queue.bunny.default';
    new Settings($settings);

    // BunnyServiceProvider requires a container rebuild to process the
    // settings configured above.
    /** @var \Drupal\Core\DrupalKernelInterface $kernel */
    $kernel = $this->container->get('kernel');
    $kernel->invalidateContainer();
    $kernel->rebuildContainer();

    $this->installConfig([QueueBase::MODULE]);
    $time = $this->container->get('datetime.time')->getCurrentTime();
    $this->routingKey = $this->queueName = 'test-' . date('c', $time) . '-' . $this->randomMachineName(8);

    $this->connectionFactory = $this->container->get('bunny.connection.factory');

    $config = $this->config('bunny.config');
    $queues = $config->get('queues');
    $queues[] = [
      'passive' => FALSE,
      'durable' => TRUE,
      'exclusive' => FALSE,
      'auto_delete' => FALSE,
      'name' => $this->queueName,
      'nowait' => FALSE,
      'routing_keys' => [],
    ];
    $config->set('queues', $queues)->save();
  }

  /**
   * {@inheritdoc}
   */
  public function tearDown(): void {
    $connection = $this->connectionFactory->getConnection();

    $channel = $connection->channel();
    if ($channel instanceof AMQPChannel) {
      $channel->close();
    }

    $connection->close();

    parent::tearDown();
  }

  /**
   * Initialize a server and free channel.
   *
   * @param string $name
   *   Queue name.
   *
   * @return array[]
   *   - \AMQPChannel: A channel to the default queue.
   *   - string: the queue name.
   */
  protected function initChannel(string $name = QueueFactory::DEFAULT_QUEUE_NAME): array {
    $connection = $this->connectionFactory->getConnection();
    $this->assertTrue($connection instanceof AMQPStreamConnection, 'Default connections is an AMQP Connection');

    $channel = $connection->channel();
    $this->assertTrue($channel instanceof AMQPChannel, 'Default connection provides channels');

    [$actualName] = $channel->queue_declare(
      $name,
      FALSE,
      TRUE,
      FALSE,
      FALSE
    );
    $this->assertEquals($name, $actualName, 'Queue declaration succeeded');

    return [$channel, $actualName];
  }

}
