<?php

declare(strict_types=1);

namespace Drupal\Tests\bunny\Unit\Serialization;

use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\bunny\Exception\InvalidSerializerException;
use Drupal\bunny\Serialization\Json;
use Drupal\bunny\Serialization\PhpSerialize;
use Drupal\bunny\Serialization\SerializerCollector;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the Json serializer.
 *
 * @group bunny
 * @covers \Drupal\bunny\Serialization\SerializerCollector
 */
class SerializerCollectorTest extends UnitTestCase {

  /**
   * Obtain the serializer under test.
   *
   * @return \Drupal\bunny\Serialization\SerializerCollector
   *   The serializer under test
   */
  public function getFixture(): SerializerCollector {
    $class_resolver_mock = $this->createMock(ClassResolverInterface::class);
    $class_resolver_mock->method('getInstanceFromDefinition')->willReturnMap(
      [
        ['bunny.serialization.phpserialize', new PhpSerialize()],
        ['bunny.serialization.json', new Json()],
        ['not_a_serializer', new \stdClass()],
      ]
    );

    $list_of_services = [
      'bunny.serialization.phpserialize',
      'bunny.serialization.json',
      'not_a_serializer',
    ];
    return new SerializerCollector($class_resolver_mock, $list_of_services);
  }

  /**
   * Tes the getSerializer() method.
   *
   * @param non-empty-string $name
   *   ServiceId of the serializer to request.
   * @param class-string $expected_class
   *   Expected class of the serializer.
   * @param callable|null $setup
   *   Optional callable to make changes to the test. Provided $this as its
   *   first parameter.
   *
   * @dataProvider providerGetSerializer
   */
  public function testGetSerializer(string $name, string $expected_class, ?callable $setup = NULL): void {
    $service = $this->getFixture();
    if ($setup) {
      $setup($this);
    }
    $serializer = $service->getSerializer($name);
    $this->assertInstanceOf($expected_class, $serializer);

  }

  /**
   * DataProvider for testGetSerializer()
   *
   * @return \Generator
   *   The test data.
   */
  public static function providerGetSerializer(): \Generator {

    yield 'Serializer php' => [
      'name' => 'bunny.serialization.phpserialize',
      'expected_class' => PhpSerialize::class,
    ];

    yield 'Serializer json' => [
      'name' => 'bunny.serialization.json',
      'expected_class' => Json::class,
    ];

    yield 'Serializer Does Not Exist' => [
      'name' => 'does_not_exist',
      'expected_class' => "\DoesNotExistClass",
      'setup' => function (self $context) {
        $context->expectException(InvalidSerializerException::class);
        $context->expectExceptionMessage('Unknown Serializer');
      },
    ];

    yield 'Class does not implement Serialization interface' => [
      'name' => 'not_a_serializer',
      'expected_class' => "\stdClass",
      'setup' => function (self $context) {
        $context->expectException(InvalidSerializerException::class);
        $context->expectExceptionMessage('Does not implement SerializationInterface');
      },
    ];

  }

}
