<?php

namespace Drupal\bureauworks_tmgmt;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\SourcePreviewInterface;
use Drupal\tmgmt\TranslatorPluginUiBase;

/**
 * Bureau translator UI.
 */
class BureauTranslatorUi extends TranslatorPluginUiBase {
  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\tmgmt\TranslatorInterface $translator */
    $translator = $form_state->getFormObject()->getEntity();

    $form['end_point_api'] = [
      '#type' => 'textfield',
      '#title' => t('Api End Point'),
      '#default_value' => $translator->getSetting('end_point_api'),
    ];
    $form['accesskey'] = [
      '#type' => 'textfield',
      '#title' => t('Access Key'),
      '#default_value' => $translator->getSetting('accesskey'),
    ];
    $form['secretAccesskey'] = [
      '#type' => 'textfield',
      '#title' => t('Secret Key'),
      '#default_value' => $translator->getSetting('secretAccesskey'),
    ];
    $form['orgUnitUUID'] = [
      '#type' => 'textfield',
      '#title' => t('Org Unit UUID'),
      '#default_value' => $translator->getSetting('orgUnitUUID'),
      '#description' => t('The Project Organizational Unit UUID. You can check Organization Units UUID by analyzing your URLs in Bureau Works.'),
    ];
    $form['contactUUID'] = [
      '#type' => 'textfield',
      '#title' => t('Contact UUID'),
      '#default_value' => $translator->getSetting('contactUUID'),
      '#description' => t('The Project Contact UUID. You can check Contact UUID by analyzing your URLs in Bureau Works.'),
    ];
    $form['workflows'] = [
      '#type' => 'textfield',
      '#title' => t('Workflows'),
      '#default_value' => $translator->getSetting('workflows'),
      '#description' => t('Workflows, separated by comma. E.g.: TRANSLATION,REVIEW,REVIEW_2'),
    ];
    $form['autoSaveTranslationsForWorkflows'] = [
      '#type' => 'textfield',
      '#title' => t('Auto Save Translations for Workflows'),
      '#default_value' => $translator->getSetting('autoSaveTranslationsForWorkflows'),
      '#description' => t('Automatically save fetched content for specified workflow stages, provided the content has been delivered in Bureau Works. Enter a comma-separated list of workflow stages E.g.: TRANSLATION,REVIEW,REVIEW_2'),
    ];
    $form['autoAcceptableWorkflows'] = [
      '#type' => 'textfield',
      '#title' => t('Auto Acceptable Workflows'),
      '#default_value' => $translator->getSetting('autoAcceptableWorkflows'),
      '#description' => t('Automatically accept fetched content for specified workflow stages, provided the content has been delivered in Bureau Works. Enter a comma-separated list of workflow stages E.g.: TRANSLATION,REVIEW,REVIEW_2'),
    ];
    $form['shouldSendPreviewUrl'] = [
      '#type' => 'checkbox',
      '#title' => t('Send preview URLs'),
      '#default_value' => $translator->getSetting('shouldSendPreviewUrl') ?: 0,
      '#description' => t('If checked, content URLs will be sent with the translation job to enable preview in Bureau Works.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function checkoutInfo(JobInterface $job) {
    $form = [];

    if ($job->isActive()) {
      $form['actions']['pull'] = [
        '#type' => 'submit',
        '#value' => $this->t('Fetch translations'),
        '#submit' => [[$this, 'submitFetchTranslations']],
        '#weight' => -10,
      ];

      $remote_mappings = $job->getRemoteMappings();
      $remote_mapping = reset($remote_mappings);
      // Display simulate operations in case translator uses sandbox API key.
      if ($job->getTranslator()->getSetting('use_sandbox') && $remote_mapping) {
        $form['actions']['simulate_complete'] = [
          '#type' => 'submit',
          '#value' => $this->t('Simulate complete'),
          '#submit' => [[$this, 'submitSimulateOrderComplete']],
        ];

        // Check whether source plugin of the first item supports preview mode.
        $form['actions']['simulate_preview'] = [
          '#type' => 'submit',
          '#value' => $this->t('Simulate preview'),
          '#submit' => [[$this, 'submitSimulateTranslationPreview']],
          '#access' => $remote_mapping->getJobItem()->getSourcePlugin() instanceof SourcePreviewInterface,
        ];
      }
    }

    return $form;
  }

  /**
   * Submit callback to fetch translations from Bureau.
   */
  public function submitFetchTranslations(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\tmgmt\Entity\Job $job */
    $job = $form_state->getFormObject()->getEntity();

    /** @var \Drupal\bureauworks_tmgmt\Plugin\tmgmt\Translator\BureauTranslator $translator_plugin */
    $translator_plugin = $job->getTranslator()->getPlugin();
    $translator_plugin->fetchTranslations($job);
  }

  /**
   * Submit callback to simulate completed orders from Bureau.
   */
  public function submitSimulateOrderComplete(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\tmgmt\Entity\Job $job */
    $job = $form_state->getFormObject()->getEntity();

    /** @var \Drupal\bureauworks_tmgmt\Plugin\tmgmt\Translator\BureauTranslator $translator_plugin */
    $translator_plugin = $job->getTranslator()->getPlugin();
    $translator_plugin->simulateCompleteOrder($job);
  }

  /**
   * Submit callback to simulate translation previews from Bureau.
   */
  public function submitSimulateTranslationPreview(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\tmgmt\Entity\Job $job */
    $job = $form_state->getFormObject()->getEntity();
    /** @var \Drupal\bureauworks_tmgmt\Plugin\tmgmt\Translator\BureauTranslator $translator_plugin */
    $translator_plugin = $job->getTranslator()->getPlugin();
    $translator_plugin->simulateTranslationPreview($job);
  }

  /**
   * {@inheritdoc}
   */
  public function checkoutSettingsForm(array $form, FormStateInterface $form_state, JobInterface $job) {
    $settings['name'] = [
      '#type' => 'textfield',
      '#title' => t('Name'),
      '#default_value' => $job->getSetting('name'),
      '#description' => t('Set the name for the Bureau order. By default, a job label will be used.'),
    ];
    $settings['comment'] = [
      '#type' => 'textfield',
      '#title' => t('Comment'),
      '#default_value' => $job->getSetting('comment'),
      '#description' => t('Set the comment for the Bureau order.'),
    ];
    if (!$job->isContinuous()) {
      $settings['duedate'] = [
        '#type' => 'date',
        '#title' => t('Due Date'),
        '#default_value' => $job->getSetting('duedate'),
        '#description' => t('The deadline for providing a translation.'),
      ];
    }

    return $settings;
  }

}
