<?php

namespace Drupal\bureauworks_tmgmt\Helper;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\Cache;

/**
 * Helper class for interacting with Drupal's cache API.
 */
class BureauCacheHelper {

  /**
   * Sets a value in the cache with optional tags.
   *
   * @param string $key
   *   The cache key.
   * @param mixed $value
   *   The value to cache.
   * @param int $ttl
   *   (optional) Time to live in seconds. Defaults to 3600 (1 hour).
   * @param array $tags
   *   (optional) Cache tags. Defaults to an empty array.
   *
   * @return mixed
   *   The cached value.
   */
  public static function set(string $key, $value, $ttl = 3600, array $tags = []) {
    $cache = \Drupal::cache();

    $expire = $ttl === CacheBackendInterface::CACHE_PERMANENT ? CacheBackendInterface::CACHE_PERMANENT : time() + $ttl;

    if (!empty($tags)) {
      $cache->set($key, $value, $expire, $tags);
    } else {
      $cache->set($key, $value, $expire);
    }

    \Drupal::logger('bureau_api')->debug('Cached key: [@key] with value: [@value]', ['@key' => $key ?? '', '@value' => print_r($value ?? '', TRUE)]);
    return $value;
  }

    /**
     * Retrieves a value from the cache.
     *
     * @param string $key
     *   The cache key.
     *
     * @return mixed|null
     *   The cached value, or NULL if not found.
     */
  public static function get(string $key) {
    $cache = \Drupal::cache();
    $cached = $cache->get($key);
    return $cached ? $cached->data : NULL;
  }

    /**
     * Deletes a value from the cache.
     *
     * @param string $key
     *   The cache key to delete.
     */
  public static function delete(string $key) {
    $cache = \Drupal::cache();
    $cache->delete($key);
    \Drupal::logger('bureau_api')->debug('Removed key @key from cache.', ['@key' => $key ?? '']);
  }

  /**
   * Invalidates cache entries by their tags.
   *
   * @param array $tags
   *   An array of cache tags to invalidate.
   */
  public static function invalidateTags(array $tags) {
    Cache::invalidateTags($tags);
    \Drupal::logger('bureau_api')->debug('Invalidated cache tags: @tags', ['@tags' => implode(', ', $tags)]);
  }

}
