<?php

namespace Drupal\bureauworks_tmgmt\Helper;

use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Static helper for queue gatekeeping logic.
 */
class QueueGateKeeper {

  public const NODES_TO_REQUEUE_CACHE_KEY = 'bureau_api.nodes_to_requeue';
  public const QUEUE_ID = 'bureau_api.job_queue';
  private const CACHE_TAG_PREFIX = 'bw_node:';
  private const BASE_KEY_PREFIX = 'bureau_api.';

  public static function addNodeToQueue($node_id) {
    if (empty($node_id)) {
        \Drupal::logger('bureau_api')->debug('Node ID is empty or not provided.');
        return;
    }

    $node = \Drupal::entityTypeManager()->getStorage('node')->load($node_id);
    if (!$node) {
        \Drupal::logger('bureau_api')->warning('Node not found: @id. Cannot queue it!', ['@id' => $node_id]);
        return;
    }

    $source_language = $node->language()->getId();
    $language_codes = self::getLanguageCodes();

    if (empty($language_codes) || !is_array($language_codes)) {
        \Drupal::logger('bureau_api')->warning('No language codes available. Cannot queue node ID: @id', ['@id' => $node_id]);
        return;
    }

    foreach ($language_codes as $target_language) {
      if ($target_language === $source_language) {
          continue;
      }

      self::queue($node_id, $target_language);
    }
  }

  private static function getLanguageCodes() {
    $cache_key = self::BASE_KEY_PREFIX . 'available_language_codes';

    $cacheHit = BureauCacheHelper::get($cache_key);
    if (is_array($cacheHit) && !empty($cacheHit)) {
      \Drupal::logger('bureau_api')->debug('Fetched available language codes from cache: @codes', ['@codes' => implode(', ', $cacheHit ?? [])]);
      return $cacheHit;
    }

    $languages = \Drupal::languageManager()->getLanguages();
    $language_codes = array_keys($languages);
    BureauCacheHelper::set($cache_key, $language_codes, 1200, []); // Cache for 20 minutes
    return $language_codes;
  }

  private static function queue(int $node_id, string $language_code) {
      $cache_key = self::buildNodeCacheKey($node_id, $language_code);

      if (BureauCacheHelper::get($cache_key)) {
          \Drupal::logger('bureau_api')->warning('It seems that node ID @id with language @lang has already been queued. Skipping addition to queue..', ['@id' => $node_id, '@lang' => $language_code]);
          return;
      }

      $queue = \Drupal::queue(QueueGateKeeper::QUEUE_ID);
      $queue->createItem([
          'node_id' => $node_id,
          'target_language' => $language_code,
          'timestamp' => round(microtime(true) * 1000),
      ]);

      self::addNodeToQueueMirror($node_id, $language_code);
      \Drupal::logger('bureau_api')->notice('Queued node for processing: [' . ((string) $node_id) . '] : [' . $language_code . ']');
  }

  private static function addNodeToQueueMirror(int $node_id, string $language_code): void {
    $cache_key = self::buildNodeCacheKey($node_id, $language_code);
    $cache_tags = self::buildNodeCacheTags($node_id, $language_code);
    BureauCacheHelper::set($cache_key, $language_code, CacheBackendInterface::CACHE_PERMANENT, $cache_tags);
    \Drupal::logger('bureau_api')->debug('Added node [@id] with lang [@lang] to queue mirror.', ['@id' => $node_id, '@lang' => $language_code]);
  }

  private static function buildNodeCacheKey(string $node_id, string $language_code): string {
    $suffix = 'node_' . $node_id . '_' . $language_code;
    return self::BASE_KEY_PREFIX . $suffix;
  }

  private static function buildNodeCacheTags(string $node_id, string $language_code): array {
    $node_tag = self::CACHE_TAG_PREFIX . $node_id;
    $node_tag_lang = $node_tag . '_' . $language_code;
    return [$node_tag_lang];
  }

  public static function invalidateNodeInQueueMirror(int $node_id, string $language_code) {
    $cache_tags = self::buildNodeCacheTags($node_id, $language_code);
    BureauCacheHelper::invalidateTags($cache_tags);
  }

  public static function requeueCachedNodes() {
    $cache = \Drupal::cache();
    $cache_key = self::NODES_TO_REQUEUE_CACHE_KEY;
    $cacheHit = $cache->get($cache_key);

    // If no cached data is found, log and return.
    if (!$cacheHit || !is_array($cacheHit->data)) {
       \Drupal::logger('bureau_api')->debug('No nodes found to requeue. Key: @key', ['@key' => $cache_key]);
       return;
    }

    // Log cached node IDs for debugging.
    $node_ids = array_keys($cacheHit->data);
    \Drupal::logger('bureau_api')->debug('Re-queueing nodes from cache: @ids', [
    '@ids' => implode(', ', $node_ids ?? []),
    ]);
    
    $data = $cacheHit->data;
    foreach (array_keys($data) as $key) {
        $parts = explode('|', $key);
        $node_id = (int) $parts[0];
        $language_code = $parts[1];

        self::queue($node_id, $language_code);
    }

    // Clear cache
    BureauCacheHelper::delete($cache_key);
  }

  public static function cacheForFutureRequeue(int $node_id, string $language_code) {
    $cache_key = self::NODES_TO_REQUEUE_CACHE_KEY;
    $nodes = BureauCacheHelper::get($cache_key);
    if (!is_array($nodes)) {
      $nodes = [];
    }

    $unique_key = self::buildNodeEntryKey($node_id, $language_code);

    $nodes[$unique_key] = $node_id;
    BureauCacheHelper::set($cache_key, $nodes, CacheBackendInterface::CACHE_PERMANENT);
    \Drupal::logger('bureau_api')->debug('Cached node ID [@id] with language [@lang] for future re-queueing.', ['@id' => $node_id ?? '', '@lang' => $language_code ?? '']);
  }

  private static function buildNodeEntryKey(int $node_id, string $language_code): string {
    return $node_id . '|' . $language_code;
  }

  public static function preventFutureRequeue(int $node_id, string $language_code) {
    $cache_key = self::NODES_TO_REQUEUE_CACHE_KEY;
    $unique_key = self::buildNodeEntryKey($node_id, $language_code);

    $nodes = BureauCacheHelper::get($cache_key);

    if (is_array($nodes) && isset($nodes[$unique_key])) {
      unset($nodes[$unique_key]);
      BureauCacheHelper::set($cache_key, $nodes, CacheBackendInterface::CACHE_PERMANENT);
      \Drupal::logger('bureau_api')->debug('Removed node ID [@id] with language [@lang] to prevent future re-queue.', ['@id' => $node_id ?? '', '@lang' => $language_code ?? '']);
    }
  }

  public static function cacheUpdatedEntity(int $entity_id, string $entity_type, string $langcode, string $origin = 'unknown') {
    $cache_key = self::buildEntityUpdateKey($entity_id, $entity_type, $langcode);
    BureauCacheHelper::set($cache_key, $origin, 180, []); // Cache for 3 minutes
    \Drupal::logger('bureau_api')->debug('Cached updated entity for [@key] with value [@value]', ['@key' => $cache_key, '@value' => $origin ?? '']);
  }

  public static function buildEntityUpdateKey(int $entity_id, string $entity_type, string $langcode): string {
    return self::BASE_KEY_PREFIX . 'entity_update_' . $entity_type . '_' . $entity_id . '_' . $langcode;
  }

  public static function getUpdatedEntityFromCache(int $entity_id, string $entity_type, string $langcode) {
    $cache_key = self::buildEntityUpdateKey($entity_id, $entity_type, $langcode);
    return BureauCacheHelper::get($cache_key);
  }

  public static function evictUpdatedEntityFromCache(int $entity_id, string $entity_type, string $langcode) {
    $cache_key = self::buildEntityUpdateKey($entity_id, $entity_type, $langcode);
    BureauCacheHelper::delete($cache_key);
    \Drupal::logger('bureau_api')->debug('Evicted updated entity cache for [@key]', ['@key' => $cache_key]);
  }

}
