<?php

namespace Drupal\bureauworks_tmgmt\Plugin\QueueWorker;

use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\tmgmt\Entity\Job;
use Drupal\tmgmt\Entity\JobItem;
use Drupal\tmgmt\JobItemInterface;
use Drupal\node\Entity\Node;
use Drupal\tmgmt\Entity\Translator;
use Drupal\bureauworks_tmgmt\Helper\QueueGateKeeper;

/**
 * Processes queued TMGMT jobs for Bureau Works.
 *
 * @QueueWorker(
 *   id = "bureau_api.job_queue",
 *   title = @Translation("Bureau API Job Queue Worker"),
 *   cron = {"time" = 30}
 * )
 */
class BureauApiJobQueueWorker extends QueueWorkerBase {

  public function processItem($data) {
    $node_id = null;
    $target_language = null;

    try {
      if (!isset($data['node_id']) || !isset($data['target_language']) || !isset($data['timestamp'])) {
        \Drupal::logger('bureau_api')->warning('Queue item missing node_id or target_language or timestamp. Skipping...');
        return;
      }

      // Load the node entity.
      $node = \Drupal::entityTypeManager()->getStorage('node')->load($data['node_id']);
      if (!$node) {
        \Drupal::logger('bureau_api')->error('Node not found: ' . ($data['node_id'] ?? ''));
        return;
      }

      $source_language = $node->language()->getId();
      if ($target_language === $source_language) {
        return;
      }

      $node_id = $node->id();
      $target_language = $data['target_language'];
      $timestamp = (int) $data['timestamp'];
      \Drupal::logger('bureau_api')->info('Processing queued node: [@nid] : [@lang]', ['@nid' => (string) $node_id, '@lang' => $target_language]);

      $added_job_item = $this->addNodeToTranslationJob($node, $target_language, $timestamp);
      if (!$added_job_item) {
        \Drupal::logger('bureau_api')->warning('No job item was added for node [' . ((string) $node->id()) . '] with source language [' . $source_language . '] and target language [' . $target_language . ']. This may indicate that there could be an ongoing job item for this node or no Jobs were found.');
      }

    } catch (\Throwable $e) {
      \Drupal::logger('bureau_api')->error('Exception while processing queued node: @error', ['@error' => $e->getMessage()]);
    } finally {
      if ($node_id !== null) {
        QueueGateKeeper::invalidateNodeInQueueMirror($node_id, $target_language);
        \Drupal::logger('bureau_api')->info('Finished processing queued node: [@nid] | language: [@lang]', ['@nid' => (string) $node_id, '@lang' => $target_language]);
      }
    }
  }

  /**
   * Adds a job item to a translation job if not already ongoing for the node.
   */
  function addNodeToTranslationJob(Node $node, string $target_langcode, int $timestamp): ?JobItem {
    $logger = \Drupal::logger('bureau_api');

    $node_id = $node->id();
    $source_langcode = $node->language()->getId();
    $jobs = $this->getContinuousJobsForLanguagePair($source_langcode, $target_langcode);

    if (empty($jobs)) {
      $logger->warning("No continuous jobs found for Node ID: [{$node_id}] [{$source_langcode}] -> [{$target_langcode}] Skipping...");
      return NULL;
    }

    // Find if a job item for this node already exists.
    $existing_job_item = $this->findOngoingJobItemForNode($jobs, $node_id, $target_langcode);

    if ($existing_job_item) {
      $logger->warning("Node [{$node_id}] already has an ongoing job item in job [{$existing_job_item->getJobId()}] with target language [{$target_langcode}]...");

      $shouldDiscard = $this->shouldDiscardQueueItem($existing_job_item->id(), $timestamp);
      if ($shouldDiscard) {
        $logger->debug("The existing job item [{$existing_job_item->id()}] for node [{$node_id}] and language [{$target_langcode}] has a more recent change time than the queue timestamp. Not re-queuing.");
        QueueGateKeeper::preventFutureRequeue($node_id, $target_langcode);
      } else {
        QueueGateKeeper::cacheForFutureRequeue($node_id, $target_langcode);
      }

      return NULL;
    }

    // Default to first job in the list.
    $job = reset($jobs);
    if (!$job) {
      $logger->error("No valid job found for node " . ((string) $node_id));
      return NULL;
    }

    // Create a new job item
    $job_item = JobItem::create([
      'tjid'           => $job->id(),
      'plugin'         => 'content',
      'item_type'      => 'node',
      'item_id'        => $node_id,
      'target_language'=> $target_langcode,
    ]);
    $job_item->save();

    $this->requestJobItemTranslation($job_item);
    QueueGateKeeper::preventFutureRequeue($node_id, $target_langcode);

    $logger->info("Created a new job item for the node [" . ((string) $node_id) . "] in job [" . ((string) $job->id()) . "] for the source language [" . ((string) $source_langcode) . "] and target language [" . ((string) $target_langcode) . "].");
    return $job_item;
  }

    /**
   * Get all continuous jobs for a given language pair.
   */
  function getContinuousJobsForLanguagePair(string $source_langcode, string $target_langcode): array {
    $job_storage = \Drupal::entityTypeManager()->getStorage('tmgmt_job');

    return $job_storage->loadByProperties([
      'source_language' => $source_langcode,
      'target_language' => $target_langcode,
      'job_type' => 'continuous',
      'translator' => 'bwx'
    ]);
  }

    /**
   * Finds the first job item in the given jobs for the given node and target language.
   *
   * @param \Drupal\tmgmt\Entity\Job[] $jobs
   * @param int $node_id
   * @param string $target_langcode
   *
   * @return \Drupal\tmgmt\Entity\JobItem|null
   */
  function findOngoingJobItemForNode(array $jobs, int $node_id, string $target_langcode): ?JobItem {

    
    // Sort jobs by created time descending (newest first)
    usort($jobs, function($a, $b) {
      return $b->getCreatedTime() <=> $a->getCreatedTime();
    });

    foreach ($jobs as $job) {
      $items = $job->getItems();
      $items = array_filter($items, function (JobItem $item) use ($node_id) {
        return (int) $item->get('item_id')->value === $node_id && $item->get('item_type')->value === 'node';
      });

      // Filter for ongoing items (not accepted, aborted, or inactive).
      $items = array_filter($items, function (JobItem $item) {
        return !$item->isAccepted() && !$item->isAborted() && !$item->isInactive();
      });

      if (!empty($items)) {
        \Drupal::logger('bureau_api')->debug('Found ongoing job item for node @nid: and target language @lang', [
          '@nid' => (string) $node_id,
          '@lang' => $target_langcode,
        ]);
        return reset($items);
      }
    }

    return NULL;
  }

  private function shouldDiscardQueueItem(int $job_item_id, int $queueTimestamp): bool {
    $job_item = JobItem::load($job_item_id);
    if (!$job_item) {
      \Drupal::logger('bureau_api')->error('Job item not found: [@id]. Cannot determine if queue item should be discarded.', ['@id' => $job_item_id]);
      return false;
    }

    $creationTimestamp = $this->inferJobItemCreationTimestamp($job_item_id);
    $lastUpdateTimestamp = $job_item->getChangedTime() * 1000; // Convert to milliseconds

    $hasNewerCreation = $creationTimestamp !== null && $creationTimestamp >= $queueTimestamp;
    $hasNewerUpdate = ($lastUpdateTimestamp >= $queueTimestamp) && $job_item->isActive();

    return $hasNewerCreation || $hasNewerUpdate;
  }

  private function inferJobItemCreationTimestamp(int $job_item_id) {
    if (empty($job_item_id) || !is_int($job_item_id)) {
      \Drupal::logger('bureau_api')->error('Invalid job item ID provided for inferring creation timestamp.');
      return NULL;
    }

    $job_item = JobItem::load($job_item_id);
    if (!$job_item) {
      \Drupal::logger('bureau_api')->error('Job item not found: [@id]. Cannot infer creation timestamp.', ['@id' => $job_item_id]);
      return NULL;
    }

    $remote_mappings = $job_item->getRemoteMappings();
    if (empty($remote_mappings)) {
      \Drupal::logger('bureau_api')->error('No remote mappings found for job item [@job_item_id].', ['@job_item_id' => $job_item->id()]);
      return NULL;
    }

    // Find the mapping that has 'project_created_at' set.
    foreach ($remote_mappings as $mapping) {
      $created_at = $mapping->getRemoteData('project_created_at');
      if (!empty($created_at)) {
        return $created_at;
      }
    }
    \Drupal::logger('bureau_api')->error('No remote mapping with project_created_at found for job item [@job_item_id].', ['@job_item_id' => $job_item->id()]);
    return NULL;
  }

  private function requestJobItemTranslation(JobItemInterface $job_item) {
    $job = $job_item->getJob();
    $translator = $job->getTranslator();

    if ($translator->getPlugin() && method_exists($translator->getPlugin(), 'requestJobItemTranslation')) {
      $translator->getPlugin()->requestJobItemTranslation($job_item);
    } else {
      \Drupal::logger('bureau_api')->error('Translator plugin does not support requestJobItemTranslation method.');
    }
  }

}