<?php

namespace Drupal\bureauworks_tmgmt\Plugin\tmgmt\Translator;

use Drupal\Core\StreamWrapper\PublicStream;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\tmgmt\ContinuousTranslatorInterface;
use Drupal\tmgmt\Entity\RemoteMapping;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt\JobItemInterface;
use Drupal\tmgmt\TMGMTException;
use Drupal\tmgmt\Translator\AvailableResult;
use Drupal\tmgmt\TranslatorInterface;
use Drupal\tmgmt\TranslatorPluginBase;
use Drupal\tmgmt\Entity\JobItem;
use Drupal\node\NodeInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Entity\EntityViewBuilderInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\tmgmt_file\Format\FormatManager;
use Drupal\tmgmt_file\Plugin\tmgmt_file\Format\Xliff;
use GuzzleHttp\ClientInterface;
use ZipArchive;
use GuzzleHttp\Exception\BadResponseException;
use \Drupal\Core\File\FileSystemInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\bureauworks_tmgmt\Helper\QueueGateKeeper;

/**
 * bureau translation plugin controller.
 *
 * @TranslatorPlugin(
 *   id = "bwx",
 *   label = @Translation("Bureau Works"),
 *   description = @Translation("Expert Translation and Localization Services by Bureau Works"),
 *   logo = "icons/bureau-icon.png",
 *   ui = "Drupal\bureauworks_tmgmt\BureauTranslatorUi",
 * )
 */
class BureauTranslator extends TranslatorPluginBase implements ContainerFactoryPluginInterface, ContinuousTranslatorInterface
{
  use StringTranslationTrait;

  /**
   * Guzzle HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $client;

  /**
   * The format manager.
   *
   * @var \Drupal\tmgmt_file\Format\FormatManager
   */
  protected $formatManager;

  /**
   * List of supported languages by Bureau.
   *
   * @var string[]
   */
  protected $supportedRemoteLanguages = [];

  /**
   * Constructs an Bureau Translator object.
   *
   * @param \GuzzleHttp\ClientInterface $client
   *   The Guzzle HTTP client.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\tmgmt_file\Format\FormatManager $format_manager
   *   The TMGMT file format manager.
   */
  public function __construct(ClientInterface $client, array $configuration, $plugin_id, array $plugin_definition, FormatManager $format_manager)
  {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->client = $client;
    $this->formatManager = $format_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
  {
    return new static(
      $container->get('http_client'),
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.tmgmt_file.format')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultSettings()
  {
    $defaults = parent::defaultSettings();
    $defaults['export_format'] = 'xlf';
    // Enable CDATA for content encoding in File translator.
    $defaults['xliff_cdata'] = TRUE;
    $defaults['xliff_processing'] = FALSE;
    return $defaults;
  }

  /**
   * {@inheritdoc}
   */
  public function requestTranslation(JobInterface $job)
  {
    $this->requestJobItemsTranslation($job->getItems());
    if (!$job->isRejected()) {
      $job->submitted('Job has been successfully submitted for translation.');
    }
  }

  public function requestJobItemTranslation($job_item)
  {
    if (!$job_item instanceof JobItemInterface) {
      throw new TMGMTException('Invalid job item provided for translation request.');
    }

    $job_items = [$job_item];
    $this->requestJobItemsTranslation($job_items);
  }

  /**
   * Executes a request against Bureau API.
   *
   * @param \Drupal\tmgmt\TranslatorInterface $translator
   *   The translator.
   * @param string $path
   *   Resource path.
   * @param array $parameters
   *   (optional) Parameters to send to Bureau service.
   * @param string $method
   *   (optional) HTTP method (GET, POST...). Defaults to GET.
   *
   * @return array
   *   Response array from Bureau.
   *
   * @throws \Drupal\tmgmt\TMGMTException
   */

  protected function doRequest(TranslatorInterface $translator, $path, array $parameters = [], $method = 'GET', $requesttype = NULL)
  {

    $api_endpoint_url = $translator->getSetting('end_point_api') . '/api/v3';

    $login_url = $api_endpoint_url . '/' . 'auth';

    $parameters_login = [
      'accessKey' => $translator->getSetting('accesskey'),
      'secret' => $translator->getSetting('secretAccesskey'),
    ];
    $parameters_login = json_encode($parameters_login);
    try {
      $options['headers']['Content-Type'] = 'application/json';
      $options['body'] = $parameters_login;

      $response = $this->client->request('POST', $login_url, $options);
    } catch (BadResponseException $e) {
      $response = $e->getResponse();
      throw new TMGMTException('Unable to connect to Bureau service due to following error: @error', ['@error' => $response->getReasonPhrase()], $response->getStatusCode());
    }

    $loginheader = $response->getHeaders();
    $Authtoken = $loginheader['X-AUTH-TOKEN'][0];
    $Apiurl = $api_endpoint_url . '/' . $path;

    try {
      // Add the authorization token.
      $options['headers']['X-AUTH-TOKEN'] = $Authtoken;
      
      if ($method == 'GET') {
        $options['query'] = $parameters;
      } else {
        if ($requesttype == 'fileupload') {
          $data = [
            'headers' => [
              'X-AUTH-TOKEN' => $Authtoken
            ],
            'multipart' => [$parameters],
          ];
          $response = $this->client->request($method, $Apiurl, $data);
        } else {
          $options['headers']['Content-Type'] = 'application/json';
          if (!empty($parameters)) {
            $parameters = json_encode($parameters);
            $options['body'] = $parameters;
          }
        }
      }

      // Make a request.
      if ($requesttype != 'fileupload') {
        $response = $this->client->request($method, $Apiurl, $options);
      }
    } catch (BadResponseException $e) {
      $response = $e->getResponse();
      throw new TMGMTException('Unable to connect to Bureau service due to following error: @error', ['@error' => $response->getReasonPhrase()], $response->getStatusCode());
    }

    $body = $response->getBody();
    if ($requesttype == 'download') {
      $received_data = $body;
    } else {
      $received_data = json_decode($body, TRUE);
    }
    
    return $received_data;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedTargetLanguages(TranslatorInterface $translator, $source_language)
  {
    $target_languages = [];
    $language_pairs = $this->getLanguagePairs($translator, $source_language);

    // Parse languages.
    foreach ($language_pairs as $language_pair) {
      $target_language = $language_pair['code'];
      $target_languages[$target_language] = $target_language;
    }

    return $target_languages;
  }
  public function getSupportedSourceLangCode(TranslatorInterface $translator)
  {
    $target_languages = [];
    $language_pairs = $this->getLanguagePairs($translator);

    // Parse languages.
    foreach ($language_pairs as $language_pair) {
      $target_language = substr($language_pair['code'], 0, 2);
      $target_languages[$target_language] = $language_pair['code'];
    }

    return $target_languages;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedRemoteLanguages(TranslatorInterface $translator)
  {
    try {
      $supported_languages = $this->doRequest($translator, 'language');
      // Parse languages.
      foreach ($supported_languages as $language) {
        $this->supportedRemoteLanguages[$language['code']] = $language['code'];
      }
      return $this->supportedRemoteLanguages;
    } catch (\Exception $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedLanguagePairs(TranslatorInterface $translator)
  {
    $language_pairs = [];

    try {
      $supported_language_pairs = $this->getLanguagePairs($translator);
      // Build a mapping of source and target language pairs.
      foreach ($supported_language_pairs as $language) {
        $language_pairs[] = [
          'source_language' => $language['code'],
          'target_language' => $language['code'],
        ];
      }
    } catch (\Exception $e) {
      return [];
    }

    return $language_pairs;
  }

  /**
   * Gets the available language pairs.
   */
  public function getLanguagePairs(TranslatorInterface $translator, $source_language = '')
  {
    $options = [];
    try {
      return $this->doRequest($translator, 'language', $options);
    } catch (TMGMTException $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function checkAvailable(TranslatorInterface $translator)
  {
    if ($translator->getSetting('end_point_api')) {
      return AvailableResult::yes();
    }
    return AvailableResult::no($this->t('@translator is not available. Make sure it is properly <a href=:configured>configured</a>.', [
      '@translator' => $translator->label(),
      ':configured' => $translator->toUrl()->toString(),
    ]));
  }

  /**
   * Creates an Project.
   */
  public function createProject(TranslatorInterface $translator, $name, $comment = NULL, $duedate = NULL, $source, $target)
  {
    $now = new \DateTime();
    $current_timestamp = $now->format('Y-m-d_H-i-s');

    $query = [
      "reference" => $name,
      "orgUnitUUID" => $translator->getSetting('orgUnitUUID'),
      "contactUUID" => $translator->getSetting('contactUUID'),
      "source" => $source,
      "tags" => [
        "drupal"
      ],
      "ciTag" => "drupal_" . $current_timestamp,
    ];

    return $this->doRequest($translator, 'project/ci', $query, 'POST');
  }

  /**
   * Sets an order into complete state via simulate call.
   */
  public function simulateOrderComplete(TranslatorInterface $translator, $order_id)
  {
    $path = '/project/' . $order_id . '/complete';
    return $this->doRequest($translator, $path, [], 'POST');
  }

  public function createProjectResource(TranslatorInterface $translator, $name, $order_id, $preview_url = NULL)
  {
    $query = [
      "name" => $name,
    ];

    if ($preview_url) {
      $query['previewUrl'] = $preview_url;
    }

    $resourceUrl =   'project/' . $order_id . '/resource';
    return $this->doRequest($translator, $resourceUrl, $query, 'POST');
  }

  public function sendResourceWorkUnits(TranslatorInterface $translator, $resource_id, $order_id, $targetids)
  {
    $workflow = $translator->getSetting('workflows');
    $workflow_arr = explode (",", str_replace(" ","", $workflow)); // replace spaces for nothing and then split comma

    $query[] = [
      "projectResourceUuid" => $resource_id,
      "workflows" => $workflow_arr,
      "targetLocales" => [
        $targetids
      ]
    ];
    
    $workunitsUrl =  'project/' . $order_id . '/work-unit?bulk=true';

    return $this->doRequest($translator, $workunitsUrl, $query, 'POST');
  }

  /**
   * Uploads a file, and attaches it to the given order.
   */
  public function sendSourceFile(TranslatorInterface $translator, $source, $target, $order_id, $file_name, $xliff_content, $resource_id)
  {
    $options = [
      'name' => 'file',
      'contents' => $xliff_content,
      'filename' => $file_name,
    ];
    $posturl = 'project/' . $order_id . '/resource/' . $resource_id . '/content';
    return $this->doRequest($translator, $posturl, $options, 'PUT', 'fileupload');
  }

  /**
   * Submits the order.
   */
  public function submitOrder(TranslatorInterface $translator, $order_id)
  {
    $posturl = 'project/' . $order_id . '/status';
    $options = [
      "newStatus" => "PENDING"
    ];
    return $this->doRequest($translator, $posturl, $options, 'POST');
  }

  /**
   * Approve the order.
   */
  public function approveProject(TranslatorInterface $translator, $order_id)
  {
    $posturl = 'project/' . $order_id . '/status';
    $options = [
      "newStatus" => "APPROVED"
    ];
    return $this->doRequest($translator, $posturl, $options, 'POST');
  }

  /**
   * Cancel the order.
   */
  public function cancelProject(TranslatorInterface $translator, $order_id)
  {
    $posturl = 'project/' . $order_id . '/status';
    $options = [
      "newStatus" => "CANCELLED"
    ];
    return $this->doRequest($translator, $posturl, $options, 'POST');
  }

  /**
   * Retrieves the translated file.
   */
  public function getFile(TranslatorInterface $translator, $order_id)
  {
    $filepath = 'project/' . $order_id . '/download';
    return $this->doRequest($translator, $filepath, [], 'GET', 'download');
  }

  public function requestFilesAsync(TranslatorInterface $translator, $project_uuid)
  {
    $posturl = 'project/' . $project_uuid . '/download';
    $options = [
      "retrieveLatestDeliveredWorkflow" => true
    ];
    return $this->doRequest($translator, $posturl, $options, 'POST');
  }

  public function getFilesRequestStatus(TranslatorInterface $translator, $project_uuid, $request_id)
  {
    $geturl = 'project/' . $project_uuid . '/download/' . $request_id . '/status';
    return $this->doRequest($translator, $geturl, [], 'GET');
  }

  public function getWorkUnitStatusesByProjectResource(TranslatorInterface $translator, $order_id, $options)
  {
    $url = 'project/' . $order_id . '/work-unit/resource-status';
    return $this->doRequest($translator, $url, $options, 'GET');
  }

  /**
   * Validates translation data.
   *
   * @param \Drupal\tmgmt_file\Plugin\tmgmt_file\Format\Xliff $xliff
   *   The xliff converter.
   * @param string $translation
   *   The translation data.
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item.
   *
   * @return bool
   *   Returns TRUE if the translation is valid.
   *
   * @throws \Drupal\tmgmt\TMGMTException
   *   Throws TMGMTException if translation is not valid.
   */

  public function isValidTranslation(Xliff $xliff, $translation, JobItemInterface $job_item)
  {
    if (!$validated_job = $xliff->validateImport($translation, FALSE)) {
      throw new TMGMTException('Failed to validate translation preview, import aborted.');
    } elseif ($validated_job->id() != $job_item->getJob()->id()) {
      throw new TMGMTException('The remote translation preview (Job ID: @target_job_id) does not match the current job ID @job_id.', [
        '@target_job_job' => $validated_job->id(),
        '@job_id' => $job_item->getJob()->id(),
      ], 'error');
    }

    return TRUE;
  }
  
  public function importAndCleanup(JobItemInterface $job_item) {

    if (!$job_item instanceof JobItemInterface) {
      throw new TMGMTException('Invalid job item provided for translation import.');
    }

    $importedTranslationPath = $this->importTranslation($job_item);

    if ($importedTranslationPath) {
      $this->cleanupExtractedFiles($importedTranslationPath);
    }
  }

  /**
   * Fetches translations for job items of a given job.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   A job containing job items that translations will be fetched for.
   */
  public function fetchTranslations(JobInterface $job)
  {

    $total_fetched = 0;
    $job_items = $job->getItems();
    $importedTranslationPaths = [];

    foreach ($job_items as $job_item) {

      try {
        $importedTranslationPath = $this->importTranslation($job_item);
      } catch (TMGMTException $tmgmt_exception) {
        \Drupal::logger('bureau_api')->error('TMGMTException: @message', ['@message' => $tmgmt_exception->getMessage()]);
        $job_item->addMessage($tmgmt_exception->getMessage());
      } catch (\Exception $e) {
        \Drupal::logger('bureau_api')->error('Exception: @message', ['@message' => $e->getMessage() ?? '']);
      }

      if ($importedTranslationPath) {
        $importedTranslationPaths[] = $importedTranslationPath;
        $total_fetched++;
      }

    }

    // Cleanup extracted files after import.
    foreach ($importedTranslationPaths as $importedTranslationPath) {
      $this->cleanupExtractedFiles($importedTranslationPath);
    }

    // Log information about fetched translations.
    if ($total_fetched > 0) {
      $job->addMessage($this->t('Fetched translations for [@total_fetched] job items.', ['@total_fetched' => $total_fetched]));
    } else {
      $job->addMessage($this->t('No translations were fetched for the job items.'));
    }
  }

  /**
   * Retrieves the translation.
   */
  public function importTranslation(JobItemInterface $job_item)
  {
    if (!$job_item instanceof JobItemInterface) {
      \Drupal::logger('bureau_api')->error('Invalid job item provided for translation import.');
      return NULL;
    }

    $remote_mapping = $this->findRemoteMappingWithOrderInfo($job_item);
    if (!$remote_mapping) {
      \Drupal::logger('bureau_api')->error('No remote mapping found for job item ID: [@jid]. Skipping import.', ['@jid' => $job_item->id()]);
      return NULL;
    }

    \Drupal::logger('bureau_api')->info('Starting import for job item ID: [@jid]', ['@jid' => $job_item->id()]);

    $xliff = $this->formatManager->createInstance('xlf');

    $file_id = $remote_mapping->getRemoteIdentifier1();
    $order_id = $remote_mapping->getRemoteIdentifier2();

    $sourceLanguageCode = $job_item->getJob()->getSourceLangcode();
    $remoteTargetLanguage = $job_item->getJob()->getRemoteTargetLanguage();
    $job_item_label = $job_item->label();
    $translator = $job_item->getTranslator();

    $file_name = $this->buildFileName($job_item_label, $sourceLanguageCode, $remoteTargetLanguage);

    $extracted_files_target_path = "public://bureau_translation/{$order_id}";
    $zip_file_target_path = $this->getOrderZipFilePath($order_id);

    $candidateFile = \Drupal::service('file_system')->realpath($zip_file_target_path);

    \Drupal::service('file_system')->prepareDirectory($extracted_files_target_path, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

    if (!file_exists($candidateFile)) {
      \Drupal::logger('bureau_api')->info('File not found, downloading now: @file', ['@file' => basename($zip_file_target_path ?? '')]);

      // $translation is a ZIP file in this case
      $translation = $this->downloadFilesAsync($translator, $order_id);

      $fileUrl = \Drupal::service('file_system')->saveData($translation, $zip_file_target_path, FileSystemInterface::EXISTS_RENAME);
      $realpath = \Drupal::service('file_system')->realpath($fileUrl);

      $ziparchive = new ZipArchive();
      if ($ziparchive->open($realpath) === TRUE) {
        $ziparchive->extractTo(\Drupal::service('file_system')->realpath($extracted_files_target_path));
        $ziparchive->close();
      }
    }

    $downloadPath = \Drupal::service('file_system')->realpath("{$extracted_files_target_path}/{$remoteTargetLanguage}/{$file_name}");
    $translation = file_get_contents($downloadPath);
    $dataxml = simplexml_load_string($translation);

    if ($dataxml === FALSE) {
      $this->messenger()->addError(t('The imported file is not a valid XML.'));
      return NULL;
    }

    if (!isset($dataxml->file['target-language']) || $remoteTargetLanguage != $dataxml->file['target-language']) {
      $translation = str_replace($dataxml->file['target-language'], $remoteTargetLanguage, $translation);
    }
    
    $validated_job = $xliff->validateImport($translation, FALSE);
    
    if (!$validated_job) {
      throw new TMGMTException('Failed to validate remote translation, import aborted.');
    } elseif ($validated_job->id() != $job_item->getJob()->id()) {
      throw new TMGMTException('The remote translation (File ID: @file_id, Job ID: @target_job_id) does not match the current job ID @job_id.', [
        '@file_id' => $target_file_id,
        '@target_job_job' => $validated_job->id(),
        '@job_id' => $job_item->getJob()->id(),
      ], 'error');
    } else {

      if ($data = $xliff->import($translation, FALSE)) {
        $job_item->getJob()->addTranslatedData($data, NULL, TMGMT_DATA_ITEM_STATE_TRANSLATED);
        $job_item->addMessage('The translation has been received.');
        $job_item->save();
        $this->onTranslationImport($job_item);
      } else {
        throw new TMGMTException('Could not process received translation data for the target file @file_id.', ['@file_id' => $target_file_id]);
      }

    }

    return $extracted_files_target_path;
  }

  public function findRemoteMappingWithOrderInfo(JobItemInterface $jobItem) {
    $remote_mappings = $jobItem->getRemoteMappings();
    if (empty($remote_mappings)) {
      return NULL;
    }

    foreach ($remote_mappings as $remote_mapping) {
      if ($remote_mapping->getRemoteIdentifier2()) {
        return $remote_mapping;
      }
    }

    return NULL;
  }

  public function downloadFilesAsync(TranslatorInterface $translator, $project_uuid)
  {
    $maxAttempts = 5; // Maximum number of attempts to poll the API in case of errors
    $attempts =  0;
    $sleep_time = 500000; // 0.5 seconds in microseconds

    $response = $this->requestFilesAsync($translator, $project_uuid);
    $status = $response['status'] ?? '';
    $request_id = $response['requestUuid'] ?? '';

    $keepPolling = TRUE;
    $maxErrors = 3; // Maximum number of consecutive errors allowed during polling
    $maxTime = 180000; // three minutes in milliseconds
    $start_time = round(microtime(TRUE) * 1000);
    $errorCount = 0;

    if (empty($request_id)) {
      \Drupal::logger('bureau_api')->error('File download request initiation failed for project UUID: @pid', ['@pid' => $project_uuid]);
      throw new TMGMTException('Could not initiate file download request. Try again later');
    }

    while ($status !== 'COMPLETED' && $keepPolling && $errorCount < $maxErrors) {

      $current_time = round(microtime(TRUE) * 1000);
      $elapsed_time = $current_time - $start_time;

      try {
        $response = $this->getFilesRequestStatus($translator, $project_uuid, $request_id);
        $status = $response['status'] ?? '';
        \Drupal::logger('bureau_api')->info('File download status for project UUID: @pid is @status. Polling.. Elapsed time: [@elapsed_time ms]', ['@elapsed_time' => $elapsed_time, '@pid' => $project_uuid, '@status' => $status]);
      } catch (\Exception $e) {
        \Drupal::logger('bureau_api')->error('Failed polling files download status for project UUID: @pid. Error: ' . $e->getMessage(), ['@pid' => $project_uuid]);
        $errorCount++;
        continue;
      }

      if ($status === 'FAILED') {
        \Drupal::logger('bureau_api')->error('File download request failed for project UUID: @pid', ['@pid' => $project_uuid]);
        throw new TMGMTException('The request failed. Try again later');
      }

      usleep($sleep_time);
      $keepPolling = $elapsed_time < $maxTime;
    }

    if ($status === 'COMPLETED') {
      return file_get_contents($response['downloadUrl']);
    }

    throw new TMGMTException('Could not download files. Try again later');
  }
  
  /**
   * Post processes the job item after translation import.
   *
   * This method checks the last delivered workflow for the provided job item and determines
   * whether the translation should be accepted or saved based on the translator's settings.
   * It logs relevant information and errors throughout the process.
   *
   * @param \TMGMT\JobItemInterface $job_item
   *   The job item for which the translation should be 'published'.
   */
  private function onTranslationImport(JobItemInterface $job_item)
  {
    if (!$job_item instanceof JobItemInterface) {
      throw new TMGMTException('Invalid job item provided for translation publish.');
    }

    $translator = $job_item->getTranslator();
    $lastDeliveredWorkflow = NULL;
    try {
      $lastDeliveredWorkflow = $this->findLastDeliveredWorkflow($job_item);
    } catch (TMGMTException $e) {
      \Drupal::logger('bureau_api')->error('Error finding last delivered workflow for job item ID [@jid]: @message. Skipping accept/save step.', ['@jid' => $job_item->id(), '@message' => $e->getMessage()]);
      return;
    }

    if (!$lastDeliveredWorkflow) {
      \Drupal::logger('bureau_api')->debug('No last delivered workflow found for job item ID [@jid]. Skipping accept/save step.', ['@jid' => $job_item->id()]);
      return;
    }
    
    // Parse the comma-separated workflows into an array.
    $autoAcceptableWorkflows = explode(',', str_replace(' ', '', $translator->getSetting('autoAcceptableWorkflows')));

    // If last delivered workflow is in auto-acceptable workflows, accept the job item.
    if (in_array($lastDeliveredWorkflow, $autoAcceptableWorkflows)) {
      
      $this->cacheUpdatedEntity($job_item, 'auto accept workflow');
      if ($job_item->acceptTranslation()) {
        \Drupal::logger('bureau_api')->info('Job item ID [@jid] has been accepted successfully.', ['@jid' => $job_item->id()]);
      } else {
        $this->evictUpdatedEntityFromCache($job_item);
      }
      return;
    }

    $autoSaveTranslationsForWorkflows = explode(',', str_replace(' ', '', $translator->getSetting('autoSaveTranslationsForWorkflows')));

    if (empty($autoSaveTranslationsForWorkflows) || !in_array($lastDeliveredWorkflow, $autoSaveTranslationsForWorkflows)) {
      \Drupal::logger('bureau_api')->debug('No saveable workflows translations defined or last delivered workflow is not auto saveable. Skipping saving step for job item ID [@jid].', ['@jid' => $job_item->id()]);
      return;
    }

    $sourcePlugin = $job_item->getSourcePlugin();
    $targetLangcode = $job_item->getJob()->getTargetLangcode();

    try {

      $this->cacheUpdatedEntity($job_item, 'auto save workflow');
      if ($sourcePlugin->saveTranslation($job_item, $targetLangcode)) {
        \Drupal::logger('bureau_api')->info('Translation for job item ID [@jid] has been saved successfully.', ['@jid' => $job_item->id()]);
        $job_item->save();
      } else {
        $this->evictUpdatedEntityFromCache($job_item);
      }

    } catch (\Exception $e) {
      \Drupal::logger('bureau_api')->error('Error saving translation for job item ID [@jid]: @message', ['@jid' => $job_item->id(), '@message' => $e->getMessage()]);
    }

  }

  private function findLastDeliveredWorkflow(JobItemInterface $job_item)
  {
    $first_mapping = $this->findRemoteMappingWithOrderInfo($job_item);
    if (!$first_mapping) {
      return NULL;
    }

    $job = $job_item->getJob();

    $order_id = $first_mapping->getRemoteIdentifier2(); // BW Project ID
    $resource_id = $first_mapping->getRemoteIdentifier1();
    $source_language = $job->getRemoteSourceLanguage();
    $target_language = $job->getRemoteTargetLanguage();

    $params = [
      'page' => 0,
      'size' => 25,
      'sourceLanguage' => $source_language,
      'targetLanguage' => $target_language,
      'projectResourceUuids' => $resource_id,
    ];

    $last_delivered_workflow = $this->inferLastDeliveredWorkflow($job->getTranslator(), $order_id, $params);
    if (!$last_delivered_workflow) {
      \Drupal::logger('bureau_api')->debug('No last delivered workflow found for job item ID [@jid].', ['@jid' => $job_item->id()]);
      return NULL;
    }
    $first_mapping->addRemoteData('last_delivered_workflow', $last_delivered_workflow);
    $first_mapping->addRemoteData('last_delivered_workflow_timestamp', round(microtime(true) * 1000));
    $first_mapping->save();

    \Drupal::logger('bureau_api')->debug('Saved last delivered workflow [@workflow] for job item ID [@jid].', [
      '@workflow' => $last_delivered_workflow ?? 'NULL',
      '@jid' => $job_item->id(),
    ]);

    return $last_delivered_workflow;
  }

  private function inferLastDeliveredWorkflow(TranslatorInterface $translator, $order_id, $params)
  {
    $work_units = [];
    
    $work_units_page = $this->getWorkUnitStatusesByProjectResource($translator, $order_id, $params);
    $work_units_chunk = $work_units_page['content'];
    $total_pages = isset($work_units_page['totalPages']) ? (int) $work_units_page['totalPages'] : 0;

    if (!empty($work_units_chunk)) {
      $work_units = array_merge($work_units, $work_units_chunk);
    }

    $next_starting_page = $params['page'] + 1;

    // Iterate over remaining pages if any
    for ($page = $next_starting_page; $page < $total_pages; $page++) {
      $params['page'] = $page;
      $work_units_page = $this->getWorkUnitStatusesByProjectResource($translator, $order_id, $params);
      if (!empty($work_units_page['content'])) {
        $work_units = array_merge($work_units, $work_units_page['content']);
      }
    }

    \Drupal::logger('bureau_api')->debug('Fetched [@count] work units from Bureau Works Project ID [@oid].', ['@count' => count($work_units), '@oid' => $order_id]);
    $last_delivered_workflow = $this->parseLastDeliveredWorkflow($work_units);
    \Drupal::logger('bureau_api')->debug('Last delivered workflow for Bureau Works Project ID [@oid] is [@workflow].', ['@oid' => $order_id, '@workflow' => $last_delivered_workflow ?? 'NULL']);

    return $last_delivered_workflow;
  }

  private function parseLastDeliveredWorkflow($work_units)
  {

    $last_delivered_workflow = NULL;
    $highest_sequence = 0;
    
    foreach ($work_units as $work_unit) {

      $is_delivered = isset($work_unit['workStatus']) && $work_unit['workStatus'] === 'DELIVERED';
      if (!$is_delivered) {
        continue;
      }

      $invalid_sequence = !isset($work_unit['sequence']) || !is_numeric($work_unit['sequence']);
      if ($invalid_sequence) {
        continue;
      }

      $sequence = (int) $work_unit['sequence'];
      if ($sequence > $highest_sequence) {
        $highest_sequence = $sequence;
        $last_delivered_workflow = $work_unit['workflow'];
      }
    }

    return $last_delivered_workflow;
  }


  private function cacheUpdatedEntity(JobItemInterface $job_item, $origin)
  {
    if (!$job_item instanceof JobItemInterface) {
      return;
    }

    try {
      $job = $job_item->getJob();
      $target_language = $job->getRemoteTargetLanguage();
      $item_type = $job_item->getItemType();
      $item_id = $job_item->getItemId();
      QueueGateKeeper::cacheUpdatedEntity($item_id, $item_type, $target_language, $origin);
    } catch (\Exception $e) {
      \Drupal::logger('bureau_api')->error('Error caching updated entity for job item ID [@jid]: @message', ['@jid' => $job_item->id(), '@message' => $e->getMessage() ?? '']);
    }

  }

  private function evictUpdatedEntityFromCache(JobItemInterface $job_item)
  {
    if (!$job_item instanceof JobItemInterface) {
      return;
    }

    try {
      $job = $job_item->getJob();
      $target_language = $job->getRemoteTargetLanguage();
      $item_type = $job_item->getItemType();
      $item_id = $job_item->getItemId();
      QueueGateKeeper::evictUpdatedEntityFromCache($item_id, $item_type, $target_language);
    } catch (\Exception $e) {
      \Drupal::logger('bureau_api')->error('Error evicting updated entity from cache for job item ID [@jid]: @message', ['@jid' => $job_item->id(), '@message' => $e->getMessage() ?? '']);
    }

  }

  private function cleanupExtractedFiles($extracted_files_target_path)
  {
    \Drupal::logger('bureau_api')->info('Attempting to delete extracted files at [@path]', ['@path' => $extracted_files_target_path ?? '']);
    try {
      $extracted_files_path = \Drupal::service('file_system')->realpath($extracted_files_target_path);
      \Drupal::service('file_system')->deleteRecursive($extracted_files_path);
      \Drupal::logger('bureau_api')->info('Deleted @path', ['@path' => $extracted_files_target_path ?? '']);
    } catch (\Exception $e) {
      \Drupal::logger('bureau_api')->error('Error deleting file: @message', ['@message' => $e->getMessage() ?? '']);
    }
  }

  /**
   * Simulates complete orders from Bureau.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The translation job.
   */
  public function simulateCompleteOrder(JobInterface $job)
  {
    $remote_mappings = $job->getRemoteMappings();
    $remote_mapping = reset($remote_mappings);

    if ($remote_mapping) {
      $order_id = $remote_mapping->getRemoteIdentifier2();

      try {
        // $order = $this->simulateOrderComplete($job->getTranslator(), $order_id);
        // if ($order) {
        $job->addMessage($this->t('The order (@order_id) has been marked as completed by using simulate order complete command.', ['@order_id' => $order_id]));
        // }
      } catch (TMGMTException $e) {
        \Drupal::messenger()->addError($e->getMessage());
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function requestJobItemsTranslation(array $job_items)
  {

    if (empty($job_items)) {
      \Drupal::logger('bureau_api')->info('All job items are already in progress. Aborting translation request.');
      return;
    }

    \Drupal::logger('bureau_api')->info('Requesting translation for [@count] job items.', ['@count' => count($job_items)]);

    /** @var \Drupal\tmgmt\Entity\Job $job */
    $first_item = reset($job_items);
    $job = $first_item->getJob();

    $created_at = round(microtime(true) * 1000);

    try {
      $source_language = $job->getRemoteSourceLanguage();
      $target_language = $job->getRemoteTargetLanguage();
      $translator = $job->getTranslator();
      $xliff = $this->formatManager->createInstance('xlf');

      $name = $this->buildProjectName($job_items, $source_language, $target_language);
      $order = $this->createProject($translator, $name, $job->getSetting('comment'), $job->getSetting('duedate'), $source_language, $target_language);
      $order_id = $order['uuid'];
      $bw_project_name = $order['name'];
      
      $job->addMessage('BW Project (@order_id) has been created.', ['@order_id' => $order_id]);

      /** @var \Drupal\tmgmt\JobItemInterface $job_item */
      foreach ($job_items as $job_item) {
        $job_item_id = $job_item->id();

        // Export content as XLIFF.
        $xliff_content = $xliff->export($job, ['tjiid' => ['value' => $job_item_id]]);
        $file_name = $this->buildFileName($job_item->label(), $job->getSourceLangcode(), $target_language);

        $preview_url = $translator->getSetting('shouldSendPreviewUrl') ? $this->buildPreviewUrl($job_item) : NULL;
        if ($preview_url) {
          \Drupal::logger('bureau_api')->debug('Preview URL for job item [@job_item_id]: @url', [
            '@job_item_id' => $job_item_id,
            '@url' => $preview_url,
          ]);
        }

        // Create a project resource.
        $file_resource = $this->createProjectResource($translator, $file_name, $order_id, $preview_url);
        $resource_id = $file_resource['uuid'];

        $file_status = $this->sendSourceFile($translator, $source_language, $target_language, $order_id, $file_name, $xliff_content, $resource_id);
        $work_units = $this->sendResourceWorkUnits($translator, $resource_id, $order_id, $target_language);

        $job_item->active();

        // Add a remote reference (file ID) to the job item, including the bw project name.
        $job_item->addRemoteMapping(NULL, $resource_id, [
          'remote_identifier_2' => $order_id,
          'remote_identifier_3' => $bw_project_name
        ]);

        $remote_mapping = $this->findRemoteMappingWithOrderInfo($job_item);
        if ($remote_mapping) {
          $remote_mapping->addRemoteData('project_created_at', $created_at);
          $remote_mapping->save();
        }
      }

      \Drupal::logger('bureau_api')->info('Successfully created Bureau Works project [@name] with [@count] job items.', [
        '@count' => count($job_items),
        '@name' => $bw_project_name,
      ]);

    } catch (TMGMTException $e) {
      if (!$job->isContinuous()) {
        $job->rejected('Job has been rejected with following error: @error', ['@error' => $e->getMessage()], 'error');
      }
    }
  }

  private function buildPreviewUrl($job_item) {
    if (!$job_item instanceof JobItemInterface) {
      return NULL;
    }

    $source_url = $job_item->getSourceUrl();
    if (!$source_url) {
      return NULL;
    }

    // Get relative path (e.g., /node/123)
    $relative_path = $source_url->toString();
    $host = \Drupal::request()->getSchemeAndHttpHost();
    return $host . $relative_path;
  }

  /**
  * {@inheritdoc}
  */
  public function abortTranslation(JobInterface $job)
  {
    if (!$job instanceof JobInterface) {
      throw new TMGMTException('Invalid job provided for abortion.');
    }

    \Drupal::logger('bureau_api')->debug('Aborting translation job [@job_id].', ['@job_id' => $job->id()]);
    $this->abortJob($job);
  }

  /**
   * Aborts a translation job.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   The translation job to abort.
   */
  private function abortJob(JobInterface $job)
  {
    // Prevent abortion of continuous jobs.
    if ($job->isContinuous()) {
      \Drupal::logger('bureau_api')->warning('Continuous job [@job_id] cannot be aborted.', ['@job_id' => $job->id()]);
      return;
    }

    if ($job->isAborted()) {
      \Drupal::logger('bureau_api')->warning('Job [@job_id] is already aborted.', ['@job_id' => $job->id()]);
      return;
    }

    \Drupal::logger('bureau_api')->debug('Aborting job [@job_id].', ['@job_id' => $job->id()]);

    $job_items = $job->getItems();
    $first_job_item = reset($job_items);

    // Get Bureau Works Project ID (order ID)
    $order_id = $this->getOrderIdFromJobItem($first_job_item);
    if (!$order_id) {
      \Drupal::logger('bureau_api')->error('No Bureau Works Project found for job item [@job_item_id].', ['@job_item_id' => $first_job_item->id()]);
      return;
    }

    // Abort all job items.
    foreach ($job_items as $job_item) {
      $job_item->abortTranslation();
      //$this->setJobItemAsAborted($job_item);
    }

    $job->setState(JobInterface::STATE_ABORTED, 'Translation job has been aborted.');
    $job->save();

    try {
      $this->cancelProject($first_job_item->getTranslator(), $order_id);
    } catch (TMGMTException $e) {
      \Drupal::logger('bureau_api')->error('Failed to cancel Bureau Works project @order_id: @error', [
        '@order_id' => $order_id,
        '@error' => $e->getMessage(),
      ]);
      return;
    }

    \Drupal::logger('bureau_api')->info('Translation job @job_id has been aborted and Bureau Works project @order_id has been cancelled.', [
      '@job_id' => $job->id(),
      '@order_id' => $order_id,
    ]);
  }

  private function getOrderIdFromJobItem(JobItemInterface $job_item) {
    if (!$job_item instanceof JobItemInterface) {
      return NULL;
    }

    $remote_mapping = $this->findRemoteMappingWithOrderInfo($job_item);
    return $remote_mapping ? $remote_mapping->getRemoteIdentifier2() : NULL;
 }

 public function getCreatedAtFromJobItem(JobItemInterface $job_item) {
    if (!$job_item instanceof JobItemInterface) {
      \Drupal::logger('bureau_api')->error('Invalid job item provided for fetching project_created_at timestamp.');
      return NULL;
    }

    $remote_mappings = $job_item->getRemoteMappings();
    if (empty($remote_mappings)) {
      \Drupal::logger('bureau_api')->error('No remote mappings found for job item [@job_item_id].', ['@job_item_id' => $job_item->id()]);
      return NULL;
    }

    foreach ($remote_mappings as $remote_mapping) {
      $timestamp = $remote_mapping->getRemoteData('project_created_at');
      if ($timestamp) {
        return $timestamp;
      }
    }

    return NULL;
 }

  /**
   * Sets a job item as aborted with a given message.
   *
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item to set as aborted.
   * @param string $message
   *   The message to log with the abortion.
   *
   * @throws \Drupal\tmgmt\TMGMTException
   *   Throws TMGMTException if the job item is invalid.
   */

 private function setJobItemAsAborted(JobItemInterface $job_item, $message = 'The translation has been aborted by the user.')
  {
    if (!$job_item instanceof JobItemInterface) {
      throw new TMGMTException('Invalid job item provided for abortion.');
    }

    if ($job_item->isAborted()) {
      \Drupal::logger('bureau_api')->warning('Job item @job_item_id is already aborted.', ['@job_item_id' => $job_item->id()]);
      return;
    }

    $job_item->setState(JobItemInterface::STATE_ABORTED, $message);
    $job_item->addMessage($message, [], 'status');
    $job_item->save();

    \Drupal::logger('bureau_api')->info('Job item @job_item_id has been aborted.', ['@job_item_id' => $job_item->id()]);
  }

  /**
   * Returns the path to the ZIP file for a given order ID.
   *
   * @param string $order_id
   *   The order ID.
   *
   * @return string
   *   The path to the ZIP file.
   */
  private function getOrderZipFilePath($order_id) {
    return 'public://bureau_translation/' . $order_id . '/' . $order_id . '.zip';
  }

  /**
   * Builds a file name for the translation file.
   *
   * @param string $label
   *   The label of the job item.
   * @param string $source_language
   *   The source language code.
   * @param string $target_language
   *   The target language code.
   *
   * @return string
   *   The constructed file name.
   */
  private function buildFileName($label, $source_language, $target_language) {
    return "{$label}_{$source_language}_{$target_language}.xlf";
  }

  /**
   * Builds the project name based on job items content titles.
   *
   * @param array $job_items
   *   Array of job items.
   * @param string $source_language
   *   The source language code.
   * @param string $target_language
   *   The target language code.
   *
   * @return string
   *   The constructed project name.
   */
  private function buildProjectName(array $job_items, $source_language, $target_language) {
    $language_pair = "{$source_language} - {$target_language}";
    
    if (count($job_items) === 1) {
      $job_item = reset($job_items);
      return $this->getEntityTitle($job_item);
    } else {
      return "Multiple items ({$language_pair})";
    }
  }

  /**
   * Gets the title from the source entity of a job item.
   *
   * @param \Drupal\tmgmt\JobItemInterface $job_item
   *   The job item.
   *
   * @return string
   *   The entity title or a default title.
   */
  private function getEntityTitle($job_item) {
    $entity_type = $job_item->getItemType();
    $entity_id = $job_item->getItemId();
    
    try {
      $entity_manager = \Drupal::entityTypeManager();
      $entity = $entity_manager->getStorage($entity_type)->load($entity_id);
        
      if ($entity && $entity->hasField('title')) {
        $title = $entity->get('title')->value;
        if (!empty($title)) {
          return $title;
        }
      }
    } catch (\Exception $e) {
      \Drupal::logger('bureau_api')->error('Failed to get entity storage for type "@type": @message', [
        '@type' => $entity_type,
        '@message' => $e->getMessage(),
      ]);
    }
    
    // Fallback title
    return 'Content Item';
  }
}
