<?php
namespace Drupal\business_identity\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

declare(strict_types=1);
/**
 * Configure Business Opening Hours.
 */
class BusinessIdentityOpeningHoursForm extends ConfigFormBase {

  /**
   * Days of the week.
   *
   * @var array
   */
  protected const DAYS_OF_WEEK = [
    'monday' => 'Monday',
    'tuesday' => 'Tuesday',
    'wednesday' => 'Wednesday',
    'thursday' => 'Thursday',
    'friday' => 'Friday',
    'saturday' => 'Saturday',
    'sunday' => 'Sunday',
  ];

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'business_identity_opening_hours_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['business_identity.opening_hours'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('business_identity.opening_hours');

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure your business opening hours. Leave times empty for closed days.') . '</p>',
    ];

    foreach (self::DAYS_OF_WEEK as $day_key => $day_label) {
      $form[$day_key] = [
        '#type' => 'fieldset',
        '#title' => $this->t($day_label),
        '#attributes' => ['class' => ['opening-hours-day']],
      ];

      $form[$day_key][$day_key . '_open'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Open'),
        '#default_value' => $config->get($day_key . '_open') ?? FALSE,
      ];

      $form[$day_key][$day_key . '_times'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name="' . $day_key . '_open"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form[$day_key][$day_key . '_times'][$day_key . '_start'] = [
        '#type' => 'time',
        '#title' => $this->t('Opening Time'),
        '#default_value' => $config->get($day_key . '_start') ?? '09:00',
      ];

      $form[$day_key][$day_key . '_times'][$day_key . '_end'] = [
        '#type' => 'time',
        '#title' => $this->t('Closing Time'),
        '#default_value' => $config->get($day_key . '_end') ?? '17:00',
      ];

      $form[$day_key][$day_key . '_times'][$day_key . '_breaks'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Break Times'),
        '#description' => $this->t('One break per line, format: HH:MM-HH:MM'),
        '#default_value' => $config->get($day_key . '_breaks'),
        '#rows' => 2,
      ];
    }

    // Special days/holidays.
    $form['special_days'] = [
      '#type' => 'details',
      '#title' => $this->t('Special Days & Holidays'),
      '#open' => FALSE,
    ];

    $special_days = $config->get('special_days') ?? [];
    $form['special_days']['special_days_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Date'),
        $this->t('Description'),
        $this->t('Status'),
        $this->t('Opening Hours'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No special days configured.'),
    ];

    for ($i = 0; $i <= count($special_days); $i++) {
      $special_day = $special_days[$i] ?? [
        'date' => '',
        'description' => '',
        'status' => 'closed',
        'hours' => '',
      ];

      $form['special_days']['special_days_table'][$i]['date'] = [
        '#type' => 'date',
        '#default_value' => $special_day['date'],
      ];

      $form['special_days']['special_days_table'][$i]['description'] = [
        '#type' => 'textfield',
        '#default_value' => $special_day['description'],
        '#placeholder' => $this->t('Description'),
      ];

      $form['special_days']['special_days_table'][$i]['status'] = [
        '#type' => 'select',
        '#options' => [
          'closed' => $this->t('Closed'),
          'open' => $this->t('Open'),
          'reduced' => $this->t('Reduced Hours'),
        ],
        '#default_value' => $special_day['status'] ?? 'closed',
      ];

      $form['special_days']['special_days_table'][$i]['hours'] = [
        '#type' => 'textfield',
        '#default_value' => $special_day['hours'],
        '#placeholder' => $this->t('e.g., 10:00-14:00'),
        '#states' => [
          'visible' => [
            ':input[name="special_days_table[' . $i . '][status]"]' => [
              ['value' => 'open'],
              ['value' => 'reduced'],
            ],
          ],
        ],
      ];

      $form['special_days']['special_days_table'][$i]['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#submit' => ['::removeSpecialDay'],
        '#name' => 'remove_special_' . $i,
        '#ajax' => [
          'callback' => '::specialDaysCallback',
          'wrapper' => 'special-days-wrapper',
        ],
      ];
    }

    $form['special_days']['special_days_table']['#prefix'] = '<div id="special-days-wrapper">';
    $form['special_days']['special_days_table']['#suffix'] = '</div>';

    $form['special_days']['add_special'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Special Day'),
      '#submit' => ['::addSpecialDay'],
      '#ajax' => [
        'callback' => '::specialDaysCallback',
        'wrapper' => 'special-days-wrapper',
      ],
    ];

    $form['notes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Notes'),
      '#description' => $this->t('Any additional information about opening hours.'),
      '#default_value' => $config->get('notes'),
      '#rows' => 3,
    ];

    // Add library for better UI.
    $form['#attached']['library'][] = 'business_identity/opening_hours';

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for special days.
   */
  public function specialDaysCallback(array &$form, FormStateInterface $form_state): array {
    return $form['special_days']['special_days_table'];
  }

  /**
   * Add a special day row.
   */
  public function addSpecialDay(array &$form, FormStateInterface $form_state): void {
    $form_state->setRebuild();
  }

  /**
   * Remove a special day row.
   */
  public function removeSpecialDay(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();
    $index = str_replace('remove_special_', '', $trigger['#name']);

    $config = $this->config('business_identity.opening_hours');
    $special_days = $config->get('special_days') ?? [];

    if (isset($special_days[$index])) {
      unset($special_days[$index]);
      $special_days = array_values($special_days);
      $config->set('special_days', $special_days)->save();
    }

    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('business_identity.opening_hours');
    $values = $form_state->getValues();

    // Save regular hours.
    foreach (self::DAYS_OF_WEEK as $day_key => $day_label) {
      $config->set($day_key . '_open', (bool) $values[$day_key . '_open']);
      $config->set($day_key . '_start', $values[$day_key . '_start']);
      $config->set($day_key . '_end', $values[$day_key . '_end']);
      $config->set($day_key . '_breaks', $values[$day_key . '_breaks']);
    }

    // Process special days.
    $special_days = [];
    foreach ($values['special_days_table'] as $row) {
      if (!empty($row['date'])) {
        $special_days[] = [
          'date' => $row['date'],
          'description' => $row['description'],
          'status' => $row['status'],
          'hours' => $row['status'] !== 'closed' ? $row['hours'] : '',
        ];
      }
    }

    $config
      ->set('special_days', $special_days)
      ->set('notes', $values['notes'])
      ->save();

    parent::submitForm($form, $form_state);
    $this->messenger()->addStatus($this->t('Opening hours configuration saved.'));
  }

}
