<?php

declare(strict_types=1);

namespace Drupal\business_identity;

/**
 * Value object representing comprehensive business information.
 */
class BusinessInformation {

  /**
   * The business name.
   *
   * @var string
   */
  protected $name;

  /**
   * The business slogan/motto.
   *
   * @var string
   */
  protected $slogan;

  /**
   * The business description.
   *
   * @var string
   */
  protected $description;

  /**
   * The primary contact information.
   *
   * @var array
   */
  protected $primaryContact = [];

  /**
   * Business locations.
   *
   * @var array
   */
  protected $locations = [];

  /**
   * Social media links.
   *
   * @var array
   */
  protected $socialLinks = [];

  /**
   * Legal information.
   *
   * @var array
   */
  protected $legalInfo = [];

  /**
   * Operating hours.
   *
   * @var array
   */
  protected $operatingHours = [];

  /**
   * Business sector/type.
   *
   * @var string
   */
  protected $businessSector;

  /**
   * Year founded.
   *
   * @var int|null
   */
  protected $foundingYear;

  /**
   * BusinessInformation constructor.
   *
   * @param array $data
   *   The business data.
   */
  public function __construct(array $data = []) {
    foreach ($data as $key => $value) {
      if (property_exists($this, $key)) {
        $this->$key = $value;
      }
    }
  }

  /**
   * Gets the business name.
   *
   * @return string
   */
  public function getName(): string {
    return $this->name ?? '';
  }

  /**
   * Gets the business slogan.
   *
   * @return string
   */
  public function getSlogan(): string {
    return $this->slogan ?? '';
  }

  /**
   * Gets the business description.
   *
   * @return string
   */
  public function getDescription(): string {
    return $this->description ?? '';
  }

  /**
   * Gets the primary contact information.
   *
   * @return array
   */
  public function getPrimaryContact(): array {
    return $this->primaryContact ?? [];
  }

  /**
   * Gets all business locations.
   *
   * @return array
   */
  public function getLocations(): array {
    return $this->locations ?? [];
  }

  /**
   * Gets a specific location by type.
   *
   * @param string $type
   *   The location type.
   *
   * @return array|null
   */
  public function getLocation(string $type): ?array {
    return $this->locations[$type] ?? NULL;
  }

  /**
   * Gets social media links.
   *
   * @return array
   */
  public function getSocialLinks(): array {
    return $this->socialLinks ?? [];
  }

  /**
   * Gets legal information.
   *
   * @return array
   */
  public function getLegalInfo(): array {
    return $this->legalInfo ?? [];
  }

  /**
   * Gets operating hours.
   *
   * @return array
   */
  public function getOperatingHours(): array {
    return $this->operatingHours ?? [];
  }

  /**
   * Gets the business sector.
   *
   * @return string
   */
  public function getBusinessSector(): string {
    return $this->businessSector ?? '';
  }

  /**
   * Gets the founding year.
   *
   * @return int|null
   */
  public function getFoundingYear(): ?int {
    return $this->foundingYear;
  }

  /**
   * Checks if business has headquarters.
   *
   * @return bool
   */
  public function hasHeadquarters(): bool {
    return !empty($this->locations['headquarters']);
  }

  /**
   * Checks if business has point of sale.
   *
   * @return bool
   */
  public function hasPointOfSale(): bool {
    return !empty($this->locations['point_of_sale']);
  }

  /**
   * Converts to array.
   *
   * @return array
   */
  public function toArray(): array {
    return [
      'name' => $this->name,
      'slogan' => $this->slogan,
      'description' => $this->description,
      'primary_contact' => $this->primaryContact,
      'locations' => $this->locations,
      'social_links' => $this->socialLinks,
      'legal_info' => $this->legalInfo,
      'operating_hours' => $this->operatingHours,
      'business_sector' => $this->businessSector,
      'founding_year' => $this->foundingYear,
    ];
  }

  /**
   * Magic getter.
   *
   * @param string $name
   *   The property name.
   *
   * @return mixed
   */
  public function __get(string $name) {
    if (property_exists($this, $name)) {
      return $this->$name;
    }
    return NULL;
  }

  /**
   * Magic isset.
   *
   * @param string $name
   *   The property name.
   *
   * @return bool
   */
  public function __isset(string $name): bool {
    return property_exists($this, $name) && isset($this->$name);
  }

}
