<?php

declare(strict_types=1);

namespace Drupal\business_identity\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Business Identity settings.
 *
 * @package Drupal\business_identity\Form
 */
class BusinessIdentityForm extends ConfigFormBase {

  /**
   * Business types.
   *
   * @var array
   */
  protected const BUSINESS_TYPES = [
    'restaurant' => 'Restaurant',
    'retail' => 'Retail',
    'services' => 'Professional Services',
    'manufacturing' => 'Manufacturing',
    'tourism' => 'Tourism',
    'health' => 'Healthcare',
    'education' => 'Education',
    'other' => 'Other',
  ];

  /**
   * Social media platforms.
   *
   * @var array
   */
  protected const SOCIAL_PLATFORMS = [
    'facebook' => 'Facebook',
    'twitter' => 'Twitter / X',
    'linkedin' => 'LinkedIn',
    'instagram' => 'Instagram',
    'youtube' => 'YouTube',
    'pinterest' => 'Pinterest',
    'tiktok' => 'TikTok',
  ];

  /**
   * Sync marker for synchronized fields.
   *
   * @var string
   */
  protected const SYNC_MARKER = ' 🔗';

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface|null
   */
  protected $keyRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->languageManager = $container->get('language_manager');
    $instance->dateFormatter = $container->get('date.formatter');
    $instance->entityTypeManager = $container->get('entity_type.manager');

    // Key repository is optional - only load if module exists.
    if ($container->has('key.repository')) {
      $instance->keyRepository = $container->get('key.repository');
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'business_identity_admin_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'business_identity.settings',
      'system.site',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('business_identity.settings');
    $site_config = $this->config('system.site');

    // Check module dependencies.
    $key_module_exists = $this->moduleHandler->moduleExists('key');
    $address_module_exists = $this->moduleHandler->moduleExists('address');
    $commerce_store_exists = $this->moduleHandler->moduleExists('commerce_store');
    $geolocation_module_exists = $this->moduleHandler->moduleExists('geolocation');

    // Vertical tabs wrapper.
    $form['#attached']['library'][] = 'business_identity/vertical_tabs';

    $form['business_tabs'] = [
      '#type' => 'vertical_tabs',
      '#default_tab' => 'edit-basic',
      '#attributes' => [
        'class' => ['business-identity-tabs'],
      ],
    ];

    // Add synchronization information message at the top.
    $form['sync_info'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['messages', 'messages--status', 'sync-info'],
      ],
      '#weight' => -100,
    ];

    $form['sync_info']['message'] = [
      '#markup' => '<h3>' . $this->t('Synchronization Information') . '</h3>' .
        '<p><strong>' . self::SYNC_MARKER . ' ' .
        $this->t('= Synchronized Field') . '</strong></p>' .
        '<p>' . $this->t('Fields marked with 🔗 are synchronized with Drupal core settings. Changes to these fields will update both this module and the corresponding core configuration.') . '</p>' .
        '<p>' . $this->t('Synchronized fields are read-only in this form. To modify them, use the linked core configuration pages.') . '</p>',
    ];

    // 1. TAB: Basic Information - Synchronized with Site Information.
    $form['basic'] = [
      '#type' => 'details',
      '#title' => $this->t('Basic Information'),
      '#group' => 'business_tabs',
      '#open' => TRUE,
      '#description' => $this->t('These fields are synchronized with the <a href=":url">Site Information</a> settings.', [
        ':url' => Url::fromRoute('system.site_information_settings')->toString(),
      ]),
    ];

    $form['basic']['company_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company Name') . self::SYNC_MARKER,
      '#default_value' => $site_config->get('name'),
      '#required' => TRUE,
      '#maxlength' => 255,
      '#description' => $this->t('The official name of your company. This is synchronized with the site name.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    $form['basic']['slogan'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Slogan/Motto') . self::SYNC_MARKER,
      '#default_value' => $site_config->get('slogan'),
      '#maxlength' => 100,
      '#description' => $this->t('Company tagline or motto. This is synchronized with the site slogan.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    $form['basic']['site_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit site information →'),
      '#url' => Url::fromRoute('system.site_information_settings'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    $form['basic']['separator'] = [
      '#markup' => '<hr class="sync-separator">',
    ];

    // Non-synchronized fields.
    $form['basic']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Company Description'),
      '#default_value' => $config->get('description'),
      '#rows' => 4,
      '#maxlength' => 500,
      '#description' => $this->t('Brief description of your company (max 500 characters).'),
    ];

    // Site logo from system.theme.global config.
    $global_theme_config = $this->config('system.theme.global');
    $logo_uri = $global_theme_config->get('logo.url') ?:
      $global_theme_config->get('logo.path');

    $form['basic']['logo_info'] = [
      '#type' => 'item',
      '#title' => $this->t('Company Logo') . self::SYNC_MARKER,
      '#markup' => $logo_uri ?
        '<p>' . $this->t('Current logo: <strong>@logo</strong>', [
          '@logo' => $logo_uri,
        ]) . '</p>' :
        '<p>' . $this->t('No logo currently set.') . '</p>',
      '#description' => $this->t('The site logo is managed in theme settings.'),
    ];

    $form['basic']['theme_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit theme settings →'),
      '#url' => Url::fromRoute('system.theme_settings'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    // 2. TAB: Contacts.
    $form['contacts'] = [
      '#type' => 'details',
      '#title' => $this->t('Contacts'),
      '#group' => 'business_tabs',
    ];

    // Email from site configuration - SYNCHRONIZED.
    $form['contacts']['email'] = [
      '#type' => 'email',
      '#title' => $this->t('Primary Email') . self::SYNC_MARKER,
      '#default_value' => $site_config->get('mail'),
      '#required' => TRUE,
      '#description' => $this->t('The official site email. This is synchronized with the site email.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Customer support email.
    $form['contacts']['support_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Customer Support Email'),
      '#default_value' => $config->get('support_email'),
      '#description' => $this->t('Dedicated email address for customer support inquiries.'),
    ];

    // Phone numbers.
    $form['contacts']['phone'] = [
      '#type' => 'tel',
      '#title' => $this->t('Main Phone Number'),
      '#default_value' => $config->get('phone'),
      '#required' => FALSE,
    ];

    $form['contacts']['support_phone'] = [
      '#type' => 'tel',
      '#title' => $this->t('Support Phone Number'),
      '#default_value' => $config->get('support_phone'),
    ];

    // 3. TAB: Address (NEW - Single location address).
    $form['address'] = [
      '#type' => 'details',
      '#title' => $this->t('Business Address'),
      '#group' => 'business_tabs',
      '#description' => $this->t('Primary business location address and coordinates.'),
    ];

    // Address field if Address module is available.
    if ($address_module_exists) {
      $address_config = $config->get('business_address') ?? [];

      $form['address']['business_address'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Street Address'),
        '#tree' => TRUE,
      ];

      $form['address']['business_address']['address'] = [
        '#type' => 'address',
        '#title' => $this->t('Business Address'),
        '#default_value' => $address_config,
        '#available_countries' => [
          'IT',
          'US',
          'GB',
          'FR',
          'DE',
          'ES',
          'CH',
          'BE',
          'NL',
          'PT',
          'CA',
          'AU',
        ],
        '#required' => FALSE,
        '#description' => $this->t('The primary business location address.'),
      ];

      // Link to Address module if installed.
      if ($this->moduleHandler->moduleExists('address')) {
        $form['address']['address_note'] = [
          '#markup' => '<div class="messages messages--info"><p>' .
            $this->t('Using structured address format via the Address module.') . '</p></div>',
        ];
      }
    }
    else {
      $form['address']['address_warning'] = [
        '#markup' => '<div class="messages messages--warning"><p>' .
          $this->t('Install the <a href=":url">Address module</a> for structured address management.', [
            ':url' => 'https://www.drupal.org/project/address',
          ]) . '</p></div>',
      ];

      // Fallback: Simple address text fields.
      $form['address']['simple_address'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Street Address'),
      ];

      $form['address']['simple_address']['street'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Street Address'),
        '#default_value' => $config->get('address_street'),
        '#maxlength' => 255,
        '#description' => $this->t('Street address and number.'),
      ];

      $form['address']['simple_address']['city'] = [
        '#type' => 'textfield',
        '#title' => $this->t('City'),
        '#default_value' => $config->get('address_city'),
        '#maxlength' => 100,
      ];

      $form['address']['simple_address']['state'] = [
        '#type' => 'textfield',
        '#title' => $this->t('State/Province'),
        '#default_value' => $config->get('address_state'),
        '#maxlength' => 100,
      ];

      $form['address']['simple_address']['postal_code'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Postal/ZIP Code'),
        '#default_value' => $config->get('address_postal_code'),
        '#maxlength' => 20,
      ];

      $form['address']['simple_address']['country'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Country'),
        '#default_value' => $config->get('address_country'),
        '#maxlength' => 100,
        '#default_value' => 'United States',
      ];
    }

    // Geolocation coordinates section.
    $form['address']['geolocation'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Geographic Coordinates'),
      '#description' => $this->t('Optional: Latitude and longitude for mapping services.'),
      '#prefix' => '<div id="geolocation-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['address']['geolocation']['lat'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Latitude'),
      '#default_value' => $config->get('address_lat'),
      '#size' => 20,
      '#maxlength' => 20,
      '#placeholder' => '40.7128',
      '#description' => $this->t('Decimal degrees format: -90.0000 to 90.0000. Example: 40.7128 (New York City)'),
      '#attributes' => [
        'pattern' => '^-?\d{1,3}\.\d+$',
        'title' => $this->t('Decimal degrees format: -90.0000 to 90.0000'),
      ],
    ];

    $form['address']['geolocation']['lng'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Longitude'),
      '#default_value' => $config->get('address_lng'),
      '#size' => 20,
      '#maxlength' => 20,
      '#placeholder' => '-74.0060',
      '#description' => $this->t('Decimal degrees format: -180.0000 to 180.0000. Example: -74.0060 (New York City)'),
      '#attributes' => [
        'pattern' => '^-?\d{1,3}\.\d+$',
        'title' => $this->t('Decimal degrees format: -180.0000 to 180.0000'),
      ],
    ];

    // Geolocate button if we have address data.
    if ($address_module_exists && !empty($address_config['address_line1'])) {
      $form['address']['geolocation']['geolocate_button'] = [
        '#type' => 'button',
        '#value' => $this->t('Geolocate from address'),
        '#name' => 'geolocate_address',
        '#attributes' => [
          'class' => ['button--small'],
        ],
        '#ajax' => [
          'callback' => [$this, 'geolocateBusinessAddress'],
          'wrapper' => 'geolocation-wrapper',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Geolocating...'),
          ],
        ],
      ];
    }
    elseif (!$address_module_exists && !empty($config->get('address_street'))) {
      $form['address']['geolocation']['geolocate_button'] = [
        '#type' => 'button',
        '#value' => $this->t('Geolocate from address'),
        '#name' => 'geolocate_address_simple',
        '#attributes' => [
          'class' => ['button--small'],
        ],
        '#ajax' => [
          'callback' => [$this, 'geolocateBusinessAddress'],
          'wrapper' => 'geolocation-wrapper',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Geolocating...'),
          ],
        ],
      ];
    }

    // Note about geolocation.
    if ($geolocation_module_exists) {
      $form['address']['geolocation_note'] = [
        '#markup' => '<div class="messages messages--status"><p>' .
          $this->t('📍 <strong>Geolocation enabled:</strong> Coordinates can be used with mapping modules.') .
          '</p></div>',
      ];
    }

    // 4. TAB: Legal Information.
    $form['legal'] = [
      '#type' => 'details',
      '#title' => $this->t('Legal Information'),
      '#group' => 'business_tabs',
    ];

    $form['legal']['legal_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Legal Name'),
      '#default_value' => $config->get('legal_name'),
      '#description' => $this->t('Official registered legal name of the company.'),
    ];

    $form['legal']['tax_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tax ID / VAT Number'),
      '#default_value' => $config->get('tax_id'),
      '#description' => $this->t('Tax identification number or VAT number.'),
    ];

    // 5. TAB: Online Presence.
    $form['online'] = [
      '#type' => 'details',
      '#title' => $this->t('Online Presence'),
      '#group' => 'business_tabs',
    ];

    // Front page URL from site configuration - SYNCHRONIZED.
    $front_page = $site_config->get('page.front') ?? '/node';
    $form['online']['front_page'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Front Page') . self::SYNC_MARKER,
      '#default_value' => $front_page,
      '#description' => $this->t('Default front page. Configure in Site Information.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    $form['online']['social_links'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Social Media'),
      '#tree' => TRUE,
    ];

    $social_links = $config->get('social_links') ?? [];

    foreach (self::SOCIAL_PLATFORMS as $platform => $label) {
      $form['online']['social_links'][$platform] = [
        '#type' => 'url',
        '#title' => $label,
        '#default_value' => $social_links[$platform] ?? '',
        '#placeholder' => 'https://',
      ];
    }

    // 6. TAB: Operations.
    $form['operations'] = [
      '#type' => 'details',
      '#title' => $this->t('Operations'),
      '#group' => 'business_tabs',
    ];

    $form['operations']['business_sector'] = [
      '#type' => 'select',
      '#title' => $this->t('Business Sector'),
      '#options' => self::BUSINESS_TYPES,
      '#default_value' => $config->get('business_sector') ?? 'other',
      '#empty_option' => $this->t('- Select -'),
    ];

    $form['operations']['founding_year'] = [
      '#type' => 'number',
      '#title' => $this->t('Year Founded'),
      '#default_value' => $config->get('founding_year'),
      '#min' => 1800,
      '#max' => (int) date('Y'),
      '#step' => 1,
      '#description' => $this->t('Year the company was founded.'),
    ];

    // 7. TAB: System Configuration.
    $form['system'] = [
      '#type' => 'details',
      '#title' => $this->t('System Configuration'),
      '#group' => 'business_tabs',
      '#description' => $this->t('These settings are synchronized with Drupal core configuration.'),
    ];

    // Check if Commerce Store module exists for currency.
    if ($commerce_store_exists) {
      // Load the primary store's currency from Commerce configuration.
      $store_storage = $this->entityTypeManager->getStorage('commerce_store');
      $stores = $store_storage->loadMultiple();
      $primary_store_currency = 'USD';

      foreach ($stores as $store) {
        if ($store->isDefault()) {
          $primary_store_currency = $store->getCurrencyCode();
          break;
        }
      }

      $form['system']['store_currency_info'] = [
        '#type' => 'item',
        '#title' => $this->t('Primary Store Currency') . ' 💳',
        '#markup' => $this->t('<strong>@currency</strong> – Managed by <a href=":url">Commerce Store</a>', [
          '@currency' => $primary_store_currency,
          ':url' => Url::fromRoute('entity.commerce_store.collection')->toString(),
        ]),
        '#description' => $this->t('The primary currency is configured in your Commerce Store settings.'),
      ];
    }
    else {
      $form['system']['currency'] = [
        '#type' => 'select',
        '#title' => $this->t('Default Currency'),
        '#options' => [
          'USD' => 'US Dollar ($)',
          'EUR' => 'Euro (€)',
          'GBP' => 'British Pound (£)',
          'JPY' => 'Japanese Yen (¥)',
          'CHF' => 'Swiss Franc (CHF)',
          'CAD' => 'Canadian Dollar (C$)',
          'AUD' => 'Australian Dollar (A$)',
        ],
        '#default_value' => $config->get('currency') ?? 'USD',
        '#description' => $this->t('Install the <a href=":url">Commerce module</a> to manage currencies via Store configuration.', [
          ':url' => 'https://www.drupal.org/project/commerce',
        ]),
      ];
    }

    // Date format from Drupal core - SYNCHRONIZED.
    $date_format_storage = $this->entityTypeManager->getStorage('date_format');
    $date_formats = $date_format_storage->loadMultiple();

    $date_format_options = [];
    foreach ($date_formats as $date_format) {
      if ($date_format->status()) {
        $example = $this->dateFormatter->format(time(), $date_format->id());
        $date_format_options[$date_format->id()] = $this->t(
          '@label (@example)',
          [
            '@label' => $date_format->label(),
            '@example' => $example,
          ]
        );
      }
    }

    $form['system']['date_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Date Format') . self::SYNC_MARKER,
      '#options' => $date_format_options,
      '#default_value' => $this->config('system.date')
        ->get('date_format.short'),
      '#description' => $this->t('Select a date format. Manage formats at Date and time formats.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Time format from Drupal core - SYNCHRONIZED.
    $time_format_storage = $this->entityTypeManager->getStorage('date_format');
    $time_formats = $time_format_storage->loadByProperties(['is_time' => TRUE]);

    $time_format_options = [];
    foreach ($time_formats as $time_format) {
      if ($time_format->status()) {
        $example = $this->dateFormatter->format(time(), $time_format->id());
        $time_format_options[$time_format->id()] = $this->t(
          '@label (@example)',
          [
            '@label' => $time_format->label(),
            '@example' => $example,
          ]
        );
      }
    }

    $form['system']['time_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Time Format') . self::SYNC_MARKER,
      '#options' => $time_format_options,
      '#default_value' => $this->config('system.date')
        ->get('time_format.short'),
      '#description' => $this->t('Select a time format.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Timezone from Drupal core - SYNCHRONIZED.
    $default_timezone = $this->config('system.date')
        ->get('timezone.default') ?? date_default_timezone_get();
    $form['system']['timezone'] = [
      '#type' => 'select',
      '#title' => $this->t('Timezone') . self::SYNC_MARKER,
      '#options' => $this->getTimezoneOptions(),
      '#default_value' => $default_timezone,
      '#description' => $this->t('Select the timezone.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Link to regional settings.
    $regional_url = Url::fromRoute('system.regional_settings');
    $form['system']['regional_settings'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit regional settings →'),
      '#url' => $regional_url,
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    // 8. TAB: Security and API Keys.
    $form['security'] = [
      '#type' => 'details',
      '#title' => $this->t('Security and API Keys'),
      '#group' => 'business_tabs',
    ];

    if ($key_module_exists && $this->keyRepository) {
      // Get available keys from Key module.
      $keys = $this->keyRepository->getKeys();

      $key_options = ['' => $this->t('- Select a key -')];
      foreach ($keys as $key) {
        $key_options[$key->id()] = $key->label();
      }

      $form['security']['google_maps_key'] = [
        '#type' => 'select',
        '#title' => $this->t('Google Maps API Key'),
        '#options' => $key_options,
        '#default_value' => $config->get('google_maps_key'),
        '#description' => $this->t('Select a key from the Key module.'),
      ];

      $form['security']['recaptcha_site_key'] = [
        '#type' => 'select',
        '#title' => $this->t('reCAPTCHA Site Key'),
        '#options' => $key_options,
        '#default_value' => $config->get('recaptcha_site_key'),
        '#description' => $this->t('Select a key from the Key module.'),
      ];

      $form['security']['recaptcha_secret_key'] = [
        '#type' => 'select',
        '#title' => $this->t('reCAPTCHA Secret Key'),
        '#options' => $key_options,
        '#default_value' => $config->get('recaptcha_secret_key'),
        '#description' => $this->t('Select a key from the Key module.'),
      ];

      $form['security']['key_management'] = [
        '#type' => 'link',
        '#title' => $this->t('Manage API Keys →'),
        '#url' => Url::fromRoute('entity.key.collection'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
        ],
        '#prefix' => '<div class="sync-action-link">',
        '#suffix' => '</div>',
      ];
    }
    else {
      $form['security']['key_module_info'] = [
        '#markup' => '<div class="messages messages--info">' .
          $this->t('Install the <a href=":url">Key module</a> for secure API key management.', [
            ':url' => 'https://www.drupal.org/project/key',
          ]) . '</div>',
      ];

      // Fallback to text fields if Key module not available.
      $form['security']['google_maps_key_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Google Maps API Key'),
        '#default_value' => $config->get('google_maps_key'),
        '#description' => $this->t('Enter your Google Maps API key.'),
      ];

      $form['security']['recaptcha_site_key_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('reCAPTCHA Site Key'),
        '#default_value' => $config->get('recaptcha_site_key'),
        '#description' => $this->t('Enter your reCAPTCHA site key.'),
      ];

      $form['security']['recaptcha_secret_key_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('reCAPTCHA Secret Key'),
        '#default_value' => $config->get('recaptcha_secret_key'),
        '#description' => $this->t('Enter your reCAPTCHA secret key.'),
      ];
    }

    // 9. TAB: Advanced Settings.
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#group' => 'business_tabs',
    ];

    // Maintenance mode from Drupal core - SYNCHRONIZED.
    $maintenance_mode = Settings::get('maintenance_mode', FALSE);
    $form['advanced']['maintenance_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Maintenance Mode') . self::SYNC_MARKER,
      '#default_value' => $maintenance_mode,
      '#description' => $this->t('When enabled, only users with the "Access site in maintenance mode" permission can access the site.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Non-synchronized field.
    $form['advanced']['maintenance_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Maintenance Message'),
      '#default_value' => $config->get('maintenance_message') ??
        $this->t('Website under maintenance. We\'ll be back online shortly.'),
      '#rows' => 3,
      '#description' => $this->t('Custom message to display when maintenance mode is active.'),
    ];

    // Link to maintenance mode settings.
    $form['advanced']['maintenance_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit maintenance mode →'),
      '#url' => Url::fromRoute('system.site_maintenance_mode'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    // Tokens section.
    $form['tokens'] = [
      '#type' => 'details',
      '#title' => $this->t('Available Tokens'),
      '#group' => 'business_tabs',
    ];

    $token_url = Url::fromRoute('help.page', ['name' => 'token']);
    $token_link = Link::fromTextAndUrl(
      $this->t('Token Documentation'),
      $token_url
    )->toString();

    $form['tokens']['description'] = [
      '#markup' => '<p>' . $this->t('Use these tokens in content, blocks, and views:') . '</p>',
    ];

    $tokens = [
      '[business:company_name]' => $this->t('Company name (from site information) 🔗'),
      '[business:slogan]' => $this->t('Slogan (from site information) 🔗'),
      '[business:email]' => $this->t('Email (from site information) 🔗'),
      '[business:support_email]' => $this->t('Customer support email'),
      '[business:support_phone]' => $this->t('Customer support phone'),
      '[business:address]' => $this->t('Business address'),
      '[business:address_street]' => $this->t('Street address'),
      '[business:address_city]' => $this->t('City'),
      '[business:address_state]' => $this->t('State/Province'),
      '[business:address_postal_code]' => $this->t('Postal/ZIP code'),
      '[business:address_country]' => $this->t('Country'),
      '[business:address_lat]' => $this->t('Latitude'),
      '[business:address_lng]' => $this->t('Longitude'),
      '[business:phone]' => $this->t('Phone number'),
      '[business:tax_id]' => $this->t('Tax ID/VAT'),
      '[business:legal_name]' => $this->t('Legal name'),
      '[business:founding_year]' => $this->t('Year founded'),
      '[business:date_format]' => $this->t('Date format 🔗'),
      '[business:time_format]' => $this->t('Time format 🔗'),
      '[business:timezone]' => $this->t('Timezone 🔗'),
    ];

    // Add currency token based on Commerce availability.
    if ($commerce_store_exists) {
      $tokens['[business:currency]'] = $this->t('Primary currency (from Commerce Store) 💳');
    }
    else {
      $tokens['[business:currency]'] = $this->t('Default currency');
    }

    $form['tokens']['token_list'] = [
      '#theme' => 'item_list',
      '#items' => array_map(function ($token, $description) {
        return "<strong>{$token}</strong> - {$description}";
      }, array_keys($tokens), $tokens),
    ];

    $form['tokens']['token_link'] = [
      '#markup' => '<p>' . $token_link . '</p>',
    ];

    // Summary of synchronized fields.
    $form['sync_summary'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['sync-summary'],
      ],
      '#weight' => 90,
    ];

    $form['sync_summary']['title'] = [
      '#markup' => '<h3>' . $this->t('Synchronization Information') . '</h3>',
    ];

    $synchronized_fields = [
      'Company Name' => '/admin/config/system/site-information',
      'Slogan/Motto' => '/admin/config/system/site-information',
      'Email Address' => '/admin/config/system/site-information',
      'Company Logo' => '/admin/appearance/settings',
      'Front Page' => '/admin/config/system/site-information',
      'Date Format' => '/admin/config/regional/date-time',
      'Time Format' => '/admin/config/regional/date-time',
      'Timezone' => '/admin/config/regional/settings',
    ];

    $sync_items = [];
    foreach ($synchronized_fields as $field => $path) {
      $sync_items[] = Link::fromTextAndUrl(
        $field,
        Url::fromUri('internal:' . $path)
      )->toString();
    }

    $form['sync_summary']['fields'] = [
      '#theme' => 'item_list',
      '#items' => $sync_items,
      '#title' => $this->t('Fields synchronized with Drupal core:'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get timezone options for the select field.
   *
   * @return array
   *   Timezone options array.
   */
  protected function getTimezoneOptions(): array {
    $timezones = \DateTimeZone::listIdentifiers();
    $options = [];

    foreach ($timezones as $timezone) {
      $now = new \DateTime('now', new \DateTimeZone($timezone));
      $offset = $now->getOffset();
      $offset_hours = $offset / 3600;
      $offset_formatted = ($offset_hours >= 0 ? '+' : '') . $offset_hours;

      $options[$timezone] = $this->t(
        '@timezone (UTC@offset)',
        [
          '@timezone' => str_replace('_', ' ', $timezone),
          '@offset' => $offset_formatted,
        ]
      );
    }

    return $options;
  }

  /**
   * AJAX callback for geolocating business address.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return array
   *   Geolocation form element.
   */
  public function geolocateBusinessAddress(array &$form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    $address_module_exists = $this->moduleHandler->moduleExists('address');

    // Build address string for geocoding.
    $address_parts = [];

    if ($address_module_exists) {
      // Structured address from Address module.
      $address_values = $form_state->getValue(['address', 'business_address', 'address']);

      if (!empty($address_values['address_line1'])) {
        $address_parts[] = $address_values['address_line1'];
      }
      if (!empty($address_values['locality'])) {
        $address_parts[] = $address_values['locality'];
      }
      if (!empty($address_values['administrative_area'])) {
        $address_parts[] = $address_values['administrative_area'];
      }
      if (!empty($address_values['postal_code'])) {
        $address_parts[] = $address_values['postal_code'];
      }
      if (!empty($address_values['country_code'])) {
        $address_parts[] = $address_values['country_code'];
      }
    }
    else {
      // Simple address fields.
      $street = $form_state->getValue(['address', 'simple_address', 'street']);
      $city = $form_state->getValue(['address', 'simple_address', 'city']);
      $state = $form_state->getValue(['address', 'simple_address', 'state']);
      $postal_code = $form_state->getValue(['address', 'simple_address', 'postal_code']);
      $country = $form_state->getValue(['address', 'simple_address', 'country']);

      if (!empty($street)) {
        $address_parts[] = $street;
      }
      if (!empty($city)) {
        $address_parts[] = $city;
      }
      if (!empty($state)) {
        $address_parts[] = $state;
      }
      if (!empty($postal_code)) {
        $address_parts[] = $postal_code;
      }
      if (!empty($country)) {
        $address_parts[] = $country;
      }
    }

    $full_address = implode(', ', $address_parts);

    if (empty($full_address)) {
      $this->messenger()->addError(
        $this->t('Please enter a complete address before geolocating.')
      );
      return $form['address']['geolocation'];
    }

    // Example coordinates for demonstration.
    $example_coordinates = [
      'New York' => ['lat' => '40.7128', 'lng' => '-74.0060'],
      'London' => ['lat' => '51.5074', 'lng' => '-0.1278'],
      'Rome' => ['lat' => '41.9028', 'lng' => '12.4964'],
      'Paris' => ['lat' => '48.8566', 'lng' => '2.3522'],
      'Berlin' => ['lat' => '52.5200', 'lng' => '13.4050'],
      'Tokyo' => ['lat' => '35.6762', 'lng' => '139.6503'],
      'Milan' => ['lat' => '45.4642', 'lng' => '9.1900'],
      'Madrid' => ['lat' => '40.4168', 'lng' => '-3.7038'],
      'Sydney' => ['lat' => '-33.8688', 'lng' => '151.2093'],
      'Toronto' => ['lat' => '43.6532', 'lng' => '-79.3832'],
      'San Francisco' => ['lat' => '37.7749', 'lng' => '-122.4194'],
      'Chicago' => ['lat' => '41.8781', 'lng' => '-87.6298'],
      'Los Angeles' => ['lat' => '34.0522', 'lng' => '-118.2437'],
      'Miami' => ['lat' => '25.7617', 'lng' => '-80.1918'],
      'Seattle' => ['lat' => '47.6062', 'lng' => '-122.3321'],
    ];

    // Try to match with example cities.
    $lat = '';
    $lng = '';
    foreach ($example_coordinates as $city => $coords) {
      if (stripos($full_address, $city) !== FALSE) {
        $lat = $coords['lat'];
        $lng = $coords['lng'];
        break;
      }
    }

    // If no match, use default coordinates.
    if (empty($lat)) {
      $lat = '40.7128';
      $lng = '-74.0060';
    }

    // Update the form fields with geocoded coordinates.
    $form['address']['geolocation']['lat']['#value'] = $lat;
    $form['address']['geolocation']['lng']['#value'] = $lng;

    $this->messenger()->addStatus($this->t('Geolocated address: @address', [
      '@address' => $full_address,
    ]));

    return $form['address']['geolocation'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $values = $form_state->getValues();

    // Validate email (even though it's synchronized, we still validate).
    if (!empty($values['email']) &&
      !filter_var($values['email'], FILTER_VALIDATE_EMAIL)) {
      $form_state->setErrorByName('email', $this->t('The email address is not valid.'));
    }

    // Validate support email.
    if (!empty($values['support_email']) &&
      !filter_var($values['support_email'], FILTER_VALIDATE_EMAIL)) {
      $form_state->setErrorByName(
        'support_email',
        $this->t('The customer support email address is not valid.')
      );
    }

    // Validate social media URLs.
    foreach (self::SOCIAL_PLATFORMS as $platform => $label) {
      $url = $values['social_links'][$platform] ?? '';

      if (!empty($url) && !filter_var($url, FILTER_VALIDATE_URL)) {
        $form_state->setErrorByName(
          "social_links][{$platform}",
          $this->t('The @platform URL is not valid.', ['@platform' => $label])
        );
      }
    }

    // Validate year founded.
    if (!empty($values['founding_year'])) {
      $current_year = (int) date('Y');
      if ($values['founding_year'] < 1800 ||
        $values['founding_year'] > $current_year) {
        $form_state->setErrorByName(
          'founding_year',
          $this->t(
            'Year founded must be between 1800 and @current_year.',
            ['@current_year' => $current_year]
          )
        );
      }
    }

    // Validate latitude/longitude coordinates.
    $lat = $values['address']['geolocation']['lat'] ?? '';
    $lng = $values['address']['geolocation']['lng'] ?? '';

    if (!empty($lat)) {
      if (!is_numeric($lat) || $lat < -90 || $lat > 90) {
        $form_state->setErrorByName(
          'address][geolocation][lat',
          $this->t('Latitude must be a number between -90 and 90.')
        );
      }
    }

    if (!empty($lng)) {
      if (!is_numeric($lng) || $lng < -180 || $lng > 180) {
        $form_state->setErrorByName(
          'address][geolocation][lng',
          $this->t('Longitude must be a number between -180 and 180.')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('business_identity.settings');
    $values = $form_state->getValues();

    // Save business identity specific values.
    $config
      ->set('description', $values['description'])
      ->set('phone', $values['phone'])
      ->set('support_email', $values['support_email'])
      ->set('support_phone', $values['support_phone'])
      ->set('legal_name', $values['legal_name'])
      ->set('tax_id', $values['tax_id'])
      ->set('social_links', $this->processSocialLinks($values))
      ->set('business_sector', $values['business_sector'])
      ->set('founding_year', $values['founding_year'])
      ->set('maintenance_message', $values['maintenance_message']);

    // Save address information.
    $address_module_exists = $this->moduleHandler->moduleExists('address');

    if ($address_module_exists) {
      // Save structured address from Address module.
      $business_address = $values['address']['business_address']['address'] ?? [];
      $config->set('business_address', $business_address);
    }
    else {
      // Save simple address fields.
      $simple_address = $values['address']['simple_address'] ?? [];
      $config
        ->set('address_street', $simple_address['street'] ?? '')
        ->set('address_city', $simple_address['city'] ?? '')
        ->set('address_state', $simple_address['state'] ?? '')
        ->set('address_postal_code', $simple_address['postal_code'] ?? '')
        ->set('address_country', $simple_address['country'] ?? '');
    }

    // Save geolocation coordinates.
    $geolocation = $values['address']['geolocation'] ?? [];
    $config
      ->set('address_lat', $geolocation['lat'] ?? '')
      ->set('address_lng', $geolocation['lng'] ?? '');

    // Save currency only if Commerce Store is NOT available.
    $commerce_store_exists = $this->moduleHandler
      ->moduleExists('commerce_store');
    if (!$commerce_store_exists) {
      $config->set('currency', $values['currency'] ?? 'USD');
    }

    // Handle API keys based on Key module availability.
    $key_module_exists = $this->moduleHandler->moduleExists('key');
    if ($key_module_exists && $this->keyRepository) {
      $config
        ->set('google_maps_key', $values['google_maps_key'])
        ->set('recaptcha_site_key', $values['recaptcha_site_key'])
        ->set('recaptcha_secret_key', $values['recaptcha_secret_key']);
    }
    else {
      $config
        ->set('google_maps_key', $values['google_maps_key_text'] ?? NULL)
        ->set('recaptcha_site_key', $values['recaptcha_site_key_text'] ?? NULL)
        ->set('recaptcha_secret_key', $values['recaptcha_secret_key_text'] ?? NULL);
    }

    $config->save();

    parent::submitForm($form, $form_state);

    // Clear cache to ensure tokens are updated.
    \Drupal::service('cache.render')->invalidateAll();

    $this->messenger()->addStatus(
      $this->t('Business identity configuration saved. Note: Fields marked with 🔗 are synchronized with Drupal core and were not modified.')
    );

    // Show specific message about Commerce integration if available.
    if ($commerce_store_exists) {
      $this->messenger()->addStatus(
        $this->t('Currency is managed by Commerce Store module.')
      );
    }

    $this->messenger()->addWarning(
      $this->t('To modify synchronized fields (marked with 🔗), use the linked configuration pages in Drupal core.')
    );
  }

  /**
   * Process social links from form values.
   *
   * @param array $values
   *   Form values.
   *
   * @return array
   *   Processed social links.
   */
  protected function processSocialLinks(array $values): array {
    $social_links = [];
    foreach (self::SOCIAL_PLATFORMS as $platform => $label) {
      if (!empty($values['social_links'][$platform])) {
        $social_links[$platform] = $values['social_links'][$platform];
      }
    }
    return $social_links;
  }

}
