<?php

declare(strict_types=1);

namespace Drupal\business_identity\Token;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Utility\Token;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides tokens for Business Identity.
 */
class BusinessIdentityToken {

  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new BusinessIdentityToken.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Implements hook_token_info().
   */
  public function getTokenInfo(): array {
    $types['business'] = [
      'name' => $this->t('Business Information'),
      'description' => $this->t('Tokens for business identity information.'),
    ];

    $tokens['business']['legal_name'] = [
      'name' => $this->t('Legal Name'),
      'description' => $this->t('The legal business name.'),
    ];

    $tokens['business']['commercial_name'] = [
      'name' => $this->t('Commercial Name'),
      'description' => $this->t('The commercial/trading name.'),
    ];

    $tokens['business']['vat_number'] = [
      'name' => $this->t('VAT Number'),
      'description' => $this->t('VAT identification number.'),
    ];

    $tokens['business']['address'] = [
      'name' => $this->t('Full Address'),
      'description' => $this->t('Complete business address formatted.'),
    ];

    $tokens['business']['phone'] = [
      'name' => $this->t('Phone Number'),
      'description' => $this->t('Primary business phone number.'),
    ];

    $tokens['business']['email'] = [
      'name' => $this->t('Email Address'),
      'description' => $this->t('Primary business email address.'),
    ];

    $tokens['business']['opening_hours'] = [
      'name' => $this->t('Opening Hours'),
      'description' => $this->t('Formatted opening hours.'),
    ];

    $tokens['business']['website'] = [
      'name' => $this->t('Website'),
      'description' => $this->t('Business website URL.'),
    ];

    return [
      'types' => $types,
      'tokens' => $tokens,
    ];
  }

  /**
   * Implements hook_tokens().
   */
  public function getTokens($type, $tokens, array $data, array $options, Token $bubbleable_metadata): array {
    $replacements = [];

    if ($type === 'business') {
      $config = $this->configFactory->get('business_identity.settings');
      $opening_hours_config = $this->configFactory->get('business_identity.opening_hours');

      foreach ($tokens as $name => $original) {
        switch ($name) {
          case 'legal_name':
            $replacements[$original] = $config->get('legal_name') ?? '';
            break;

          case 'commercial_name':
            $replacements[$original] = $config->get('commercial_name') ?? '';
            break;

          case 'vat_number':
            $replacements[$original] = $config->get('vat_number') ?? '';
            break;

          case 'address':
            $address = $config->get('address');
            if ($address && isset($address['address_line1'])) {
              $replacements[$original] = $this->formatAddress($address);
            }
            else {
              $replacements[$original] = '';
            }
            break;

          case 'phone':
            $replacements[$original] = $config->get('phone') ?? '';
            break;

          case 'email':
            $replacements[$original] = $config->get('email') ?? '';
            break;

          case 'opening_hours':
            $replacements[$original] = $this->formatOpeningHours($opening_hours_config);
            break;

          case 'website':
            $replacements[$original] = $config->get('website') ?? '';
            break;
        }
      }
    }

    return $replacements;
  }

  /**
   * Format address for display.
   */
  protected function formatAddress(array $address): string {
    $parts = [];

    if (!empty($address['address_line1'])) {
      $parts[] = $address['address_line1'];
    }
    if (!empty($address['address_line2'])) {
      $parts[] = $address['address_line2'];
    }
    if (!empty($address['locality'])) {
      $parts[] = $address['locality'];
    }
    if (!empty($address['administrative_area'])) {
      $parts[] = $address['administrative_area'];
    }
    if (!empty($address['postal_code'])) {
      $parts[] = $address['postal_code'];
    }
    if (!empty($address['country_code'])) {
      $country_list = \Drupal::service('address.country_repository')->getList();
      if (isset($country_list[$address['country_code']])) {
        $parts[] = $country_list[$address['country_code']];
      }
    }

    return implode(', ', $parts);
  }

  /**
   * Format opening hours for display.
   */
  protected function formatOpeningHours($config): string {
    $days = [
      'monday' => $this->t('Monday'),
      'tuesday' => $this->t('Tuesday'),
      'wednesday' => $this->t('Wednesday'),
      'thursday' => $this->t('Thursday'),
      'friday' => $this->t('Friday'),
      'saturday' => $this->t('Saturday'),
      'sunday' => $this->t('Sunday'),
    ];

    $formatted = [];
    foreach ($days as $day_key => $day_label) {
      if ($config->get($day_key . '_open')) {
        $start = $config->get($day_key . '_start') ?? '09:00';
        $end = $config->get($day_key . '_end') ?? '17:00';
        $formatted[] = $day_label . ': ' . $start . ' - ' . $end;
      }
      else {
        $formatted[] = $day_label . ': ' . $this->t('Closed');
      }
    }

    return implode('<br>', $formatted);
  }

}
