<?php

declare(strict_types=1);

namespace Drupal\business_identity\Twig;

use Drupal\business_identity\BusinessIdentityManagerInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extension for business identity.
 */
class BusinessIdentityTwigExtension extends AbstractExtension {

  /**
   * The business identity manager.
   *
   * @var \Drupal\business_identity\BusinessIdentityManagerInterface
   */
  protected $businessIdentityManager;

  /**
   * Constructor.
   */
  public function __construct(BusinessIdentityManagerInterface $business_identity_manager) {
    $this->businessIdentityManager = $business_identity_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions(): array {
    return [
      new TwigFunction('business_info', [$this, 'getBusinessInfo']),
      new TwigFunction('business_name', [$this, 'getBusinessName']),
      new TwigFunction('business_contact', [$this, 'getBusinessContact']),
      new TwigFunction('business_address', [$this, 'getBusinessAddress']),
      new TwigFunction('business_social_links', [$this, 'getSocialLinks']),
      new TwigFunction('business_is_open', [$this, 'isBusinessOpen']),
    ];
  }

  /**
   * Gets business information.
   */
  public function getBusinessInfo(): array {
    return $this->businessIdentityManager->getBusinessInfo()->toArray();
  }

  /**
   * Gets business name.
   */
  public function getBusinessName(): string {
    return $this->businessIdentityManager->getBusinessName();
  }

  /**
   * Gets business contact information.
   */
  public function getBusinessContact(): array {
    return $this->businessIdentityManager->getPrimaryContact();
  }

  /**
   * Gets formatted business address.
   */
  public function getBusinessAddress(string $type = 'headquarters', string $format = 'html'): string {
    return $this->businessIdentityManager->getFormattedAddress($type, $format);
  }

  /**
   * Gets social links.
   */
  public function getSocialLinks(): array {
    return $this->businessIdentityManager->getSocialLinks();
  }

  /**
   * Checks if business is open.
   */
  public function isBusinessOpen(): bool {
    return $this->businessIdentityManager->isOpen();
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): string {
    return 'business_identity.twig_extension';
  }

}
