<?php

declare(strict_types=1);

namespace Drupal\business_identity;

/**
 * Interface for business identity management service.
 */
interface BusinessIdentityManagerInterface {

  /**
   * Gets comprehensive business information.
   *
   * @return \Drupal\business_identity\BusinessInformation
   *   The business information object.
   */
  public function getBusinessInfo(): BusinessInformation;

  /**
   * Gets the primary contact information.
   *
   * @return array
   *   An array containing primary contact details.
   */
  public function getPrimaryContact(): array;

  /**
   * Gets all business locations.
   *
   * @return array
   *   An array of location objects/arrays.
   */
  public function getLocations(): array;

  /**
   * Gets a specific location by type.
   *
   * @param string $type
   *   The location type (e.g., 'headquarters', 'point_of_sale').
   *
   * @return array|null
   *   The location data or NULL if not found.
   */
  public function getLocation(string $type): ?array;

  /**
   * Gets business social media links.
   *
   * @return array
   *   Array of social media platform URLs.
   */
  public function getSocialLinks(): array;

  /**
   * Gets business legal information.
   *
   * @return array
   *   Array containing legal name, tax ID, etc.
   */
  public function getLegalInfo(): array;

  /**
   * Gets formatted business address.
   *
   * @param string $type
   *   The location type.
   * @param string $format
   *   The format (html, inline, plain).
   *
   * @return string
   *   Formatted address string.
   */
  public function getFormattedAddress(string $type = 'headquarters', string $format = 'html'): string;

  /**
   * Gets business operating hours.
   *
   * @return array
   *   Array of operating hours by day.
   */
  public function getOperatingHours(): array;

  /**
   * Checks if business is currently open.
   *
   * @return bool
   *   TRUE if business is open, FALSE otherwise.
   */
  public function isOpen(): bool;

  /**
   * Gets business contact methods.
   *
   * @return array
   *   Array of available contact methods.
   */
  public function getContactMethods(): array;

  /**
   * Validates business information completeness.
   *
   * @return array
   *   Array of validation results/missing fields.
   */
  public function validateBusinessInfo(): array;

}
