<?php

declare(strict_types=1);

namespace Drupal\business_identity\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Returns responses for Business Identity routes.
 */
class BusinessIdentityController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new BusinessIdentityController.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * Returns JSON-LD structured data for the business.
   */
  public function jsonLd(): JsonResponse {
    $config = $this->configFactory->get('business_identity.settings');
    $opening_hours_config = $this->configFactory->get('business_identity.opening_hours');

    $json_ld = [
      '@context' => 'https://schema.org',
      '@type' => 'LocalBusiness',
      'name' => $config->get('commercial_name') ?: $config->get('legal_name'),
      'legalName' => $config->get('legal_name'),
      'vatID' => $config->get('vat_number'),
      'description' => $this->t('Business information'),
    ];

    // Address.
    $address = $config->get('address');
    if ($address && !empty($address['address_line1'])) {
      $json_ld['address'] = [
        '@type' => 'PostalAddress',
        'streetAddress' => $address['address_line1'],
        'addressLocality' => $address['locality'] ?? '',
        'addressRegion' => $address['administrative_area'] ?? '',
        'postalCode' => $address['postal_code'] ?? '',
        'addressCountry' => $address['country_code'] ?? '',
      ];
    }

    // Contact.
    if ($config->get('phone')) {
      $json_ld['telephone'] = $config->get('phone');
    }
    if ($config->get('email')) {
      $json_ld['email'] = $config->get('email');
    }
    if ($config->get('website')) {
      $json_ld['url'] = $config->get('website');
    }

    // Opening hours.
    $days = [
      'Monday' => 'monday',
      'Tuesday' => 'tuesday',
      'Wednesday' => 'wednesday',
      'Thursday' => 'thursday',
      'Friday' => 'friday',
      'Saturday' => 'saturday',
      'Sunday' => 'sunday',
    ];

    $opening_hours = [];
    foreach ($days as $day_name => $day_key) {
      if ($opening_hours_config->get($day_key . '_open')) {
        $start = $opening_hours_config->get($day_key . '_start') ?? '09:00';
        $end = $opening_hours_config->get($day_key . '_end') ?? '17:00';
        $opening_hours[] = $day_name . ' ' . $start . '-' . $end;
      }
    }

    if (!empty($opening_hours)) {
      $json_ld['openingHours'] = $opening_hours;
    }

    // Social links.
    $social_links = $config->get('social_links') ?? [];
    if (!empty($social_links)) {
      $json_ld['sameAs'] = [];
      foreach ($social_links as $link) {
        if (!empty($link['url'])) {
          $json_ld['sameAs'][] = $link['url'];
        }
      }
    }

    return new JsonResponse($json_ld, 200, [
      'Content-Type' => 'application/ld+json',
      'Cache-Control' => 'public, max-age=3600',
    ]);
  }

}
