/**
 * @file
 * JavaScript for German local laws module.
 */

(function ($, Drupal) {
  'use strict';

  Drupal.behaviors.businessIdentityLocalDE = {
    attach: function (context, settings) {
      // Auto-format USt-IdNr input
      $('input[data-ust-idnr-input]', context).once('ust-idnr-format').each(function () {
        $(this).on('input', function () {
          // Remove any non-digit characters
          var value = $(this).val().replace(/\D/g, '');

          // Limit to 9 digits
          if (value.length > 9) {
            value = value.substring(0, 9);
          }

          $(this).val(value);

          // Show validation feedback
          if (value.length === 9) {
            $(this).addClass('is-valid').removeClass('is-invalid');
            $('#ust-idnr-validation').remove();
            $(this).after('<div id="ust-idnr-validation" class="valid-feedback">' +
              Drupal.t('Valid USt-IdNr format') + '</div>');
          } else if (value.length > 0) {
            $(this).addClass('is-invalid').removeClass('is-valid');
            $('#ust-idnr-validation').remove();
            $(this).after('<div id="ust-idnr-validation" class="invalid-feedback">' +
              Drupal.t('USt-IdNr must be 9 digits') + '</div>');
          } else {
            $(this).removeClass('is-valid is-invalid');
            $('#ust-idnr-validation').remove();
          }
        });
      });

      // Auto-format IBAN
      $('input[data-drupal-selector*="iban"]', context).once('iban-format').each(function () {
        $(this).on('input', function () {
          var value = $(this).val().toUpperCase().replace(/[^A-Z0-9]/g, '');

          // Format with spaces every 4 characters
          if (value.length > 0) {
            value = value.match(/.{1,4}/g).join(' ');
          }

          $(this).val(value);
        });
      });

      // Auto-format BIC
      $('input[data-drupal-selector*="bic"]', context).once('bic-format').each(function () {
        $(this).on('input', function () {
          $(this).val($(this).val().toUpperCase());
        });
      });

      // Toggle Impressum fields
      var $impressumCheckbox = $('input[data-drupal-selector*="impressum-required"]', context);
      var $impressumFields = $('input[data-drupal-selector*="gesetzlicher_vertreter"], ' +
                               'input[data-drupal-selector*="aufsichtsbehoerde"], ' +
                               'input[data-drupal-selector*="berufsbezeichnung"]', context);

      function toggleImpressumFields() {
        if ($impressumCheckbox.is(':checked')) {
          $impressumFields.closest('.form-item').slideDown();
        } else {
          $impressumFields.closest('.form-item').slideUp();
        }
      }

      toggleImpressumFields();
      $impressumCheckbox.on('change', toggleImpressumFields);

      // Toggle Data Protection Officer fields
      var $dsbCheckbox = $('input[data-drupal-selector*="datenschutzbeauftragter"]', context);
      var $dsbField = $('input[data-drupal-selector*="dsb_contact"]', context);

      function toggleDSBField() {
        if ($dsbCheckbox.is(':checked')) {
          $dsbField.closest('.form-item').slideDown();
        } else {
          $dsbField.closest('.form-item').slideUp();
        }
      }

      toggleDSBField();
      $dsbCheckbox.on('change', toggleDSBField);

      // Validate German VAT number format
      $('input[data-drupal-selector*="vat-number"]', context).once('vat-validate').each(function () {
        $(this).on('blur', function () {
          var value = $(this).val().trim();

          if (value.length > 0) {
            // Check for common German VAT number patterns
            var isValid = /^(DE)?[0-9]{9}$|^[0-9]{2}\/[0-9]{3,5}\/[0-9]{4,5}$/.test(value);

            if (isValid) {
              $(this).addClass('is-valid').removeClass('is-invalid');
            } else {
              $(this).addClass('is-invalid').removeClass('is-valid');
            }
          }
        });
      });
    }
  };

})(jQuery, Drupal);
