<?php

declare(strict_types=1);

namespace Drupal\Tests\business_identity\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Business Identity module.
 *
 * @group business_identity
 */
class BusinessIdentityTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['business_identity'];

  /**
   * Tests the configuration form.
   */
  public function testConfigurationForm(): void {
    // Create an admin user.
    $admin_user = $this->drupalCreateUser([
      'administer business identity',
    ]);
    $this->drupalLogin($admin_user);

    // Access the configuration form.
    $this->drupalGet('/admin/config/system/business-identity');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Business Identity');

    // Fill and submit the form.
    $edit = [
      'legal_name' => 'Test Business LLC',
      'commercial_name' => 'Test Shop',
      'vat_number' => 'IT12345678901',
      'phone' => '+390123456789',
      'email' => 'test@example.com',
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify configuration was saved.
    $this->assertSession()->pageTextContains('Business identity configuration saved.');

    // Verify the values are present.
    $this->drupalGet('/admin/config/system/business-identity');
    $this->assertSession()->fieldValueEquals('legal_name', 'Test Business LLC');
    $this->assertSession()->fieldValueEquals('commercial_name', 'Test Shop');
  }

  /**
   * Tests the JSON-LD endpoint.
   */
  public function testJsonLdEndpoint(): void {
    // Set some configuration.
    $config = $this->container->get('config.factory')->getEditable('business_identity.settings');
    $config
      ->set('legal_name', 'Test Company')
      ->set('commercial_name', 'Test Store')
      ->set('vat_number', 'IT12345678901')
      ->set('phone', '+390123456789')
      ->set('email', 'info@example.com')
      ->save();

    // Access the JSON-LD endpoint.
    $this->drupalGet('/business-identity/jsonld');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseHeaderContains('Content-Type', 'application/ld+json');

    // Parse and verify JSON.
    $json = json_decode($this->getSession()->getPage()->getContent(), TRUE);
    $this->assertIsArray($json);
    $this->assertEquals('Test Store', $json['name']);
    $this->assertEquals('Test Company', $json['legalName']);
    $this->assertEquals('IT12345678901', $json['vatID']);
  }

  /**
   * Tests block availability.
   */
  public function testBlock(): void {
    // Place the block.
    $this->drupalPlaceBlock('business_identity_block', [
      'region' => 'footer',
    ]);

    // Visit the homepage.
    $this->drupalGet('');
    $this->assertSession()->pageTextContains('Business Identity');
  }

}
