<?php

declare(strict_types=1);

namespace Drupal\business_identity;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\address\AddressInterface;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Provides business identity management services.
 */
class BusinessIdentityManager implements BusinessIdentityManagerInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Business identity configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Site configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $siteConfig;

  /**
   * Constructor.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    LanguageManagerInterface $language_manager,
    TimeInterface $time
  ) {
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->languageManager = $language_manager;
    $this->time = $time;
    $this->config = $config_factory->get('business_identity.settings');
    $this->siteConfig = $config_factory->get('system.site');
  }

  /**
   * {@inheritdoc}
   */
  public function getBusinessInfo(): BusinessInformation {
    $data = [
      'name' => $this->getBusinessName(),
      'slogan' => $this->getBusinessSlogan(),
      'description' => $this->config->get('description') ?? '',
      'primaryContact' => $this->getPrimaryContact(),
      'locations' => $this->getLocations(),
      'socialLinks' => $this->getSocialLinks(),
      'legalInfo' => $this->getLegalInfo(),
      'operatingHours' => $this->getOperatingHours(),
      'businessSector' => $this->config->get('business_sector') ?? 'other',
      'foundingYear' => $this->config->get('founding_year'),
    ];

    return new BusinessInformation($data);
  }

  /**
   * {@inheritdoc}
   */
  public function getPrimaryContact(): array {
    return [
      'email' => $this->siteConfig->get('mail') ?? '',
      'support_email' => $this->config->get('support_email') ?? '',
      'phone' => $this->config->get('phone') ?? '',
      'support_phone' => $this->config->get('support_phone') ?? '',
      'website' => $this->config->get('website') ?? '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getLocations(): array {
    $locations = [];

    // Headquarters
    $headquarters = $this->config->get('headquarters_address');
    if ($headquarters) {
      $locations['headquarters'] = [
        'type' => 'headquarters',
        'label' => $this->t('Headquarters'),
        'address' => $headquarters,
        'coordinates' => [
          'lat' => $this->config->get('headquarters_lat'),
          'lng' => $this->config->get('headquarters_lng'),
        ],
        'primary' => TRUE,
      ];
    }

    // Point of Sale
    $point_of_sale = $this->config->get('point_of_sale_address');
    if ($point_of_sale) {
      $locations['point_of_sale'] = [
        'type' => 'point_of_sale',
        'label' => $this->t('Point of Sale'),
        'address' => $point_of_sale,
        'coordinates' => [
          'lat' => $this->config->get('point_of_sale_lat'),
          'lng' => $this->config->get('point_of_sale_lng'),
        ],
        'primary' => FALSE,
      ];
    }

    // Apply alter hook
    $this->moduleHandler->alter('business_identity_locations', $locations);

    return $locations;
  }

  /**
   * {@inheritdoc}
   */
  public function getLocation(string $type): ?array {
    $locations = $this->getLocations();
    return $locations[$type] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getSocialLinks(): array {
    $links = $this->config->get('social_links') ?? [];

    // Filter out empty links
    $filtered_links = array_filter($links, function($url) {
      return !empty($url);
    });

    // Apply alter hook
    $this->moduleHandler->alter('business_identity_social_links', $filtered_links);

    return $filtered_links;
  }

  /**
   * {@inheritdoc}
   */
  public function getLegalInfo(): array {
    return [
      'legal_name' => $this->config->get('legal_name') ?? '',
      'tax_id' => $this->config->get('tax_id') ?? '',
      'founding_year' => $this->config->get('founding_year'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedAddress(string $type = 'headquarters', string $format = 'html'): string {
    $location = $this->getLocation($type);
    if (!$location || empty($location['address'])) {
      return '';
    }

    $address = $location['address'];

    switch ($format) {
      case 'inline':
        return $this->formatAddressInline($address);

      case 'plain':
        return $this->formatAddressPlain($address);

      case 'html':
      default:
        return $this->formatAddressHtml($address);
    }
  }

  /**
   * Formats address as HTML.
   */
  protected function formatAddressHtml(array $address): string {
    $parts = [];

    if (!empty($address['address_line1'])) {
      $parts[] = '<div class="address-line1">' . $address['address_line1'] . '</div>';
    }
    if (!empty($address['address_line2'])) {
      $parts[] = '<div class="address-line2">' . $address['address_line2'] . '</div>';
    }
    if (!empty($address['locality'])) {
      $parts[] = '<div class="locality">' . $address['locality'] . '</div>';
    }
    if (!empty($address['administrative_area'])) {
      $parts[] = '<div class="administrative-area">' . $address['administrative_area'] . '</div>';
    }
    if (!empty($address['postal_code'])) {
      $parts[] = '<div class="postal-code">' . $address['postal_code'] . '</div>';
    }
    if (!empty($address['country_code'])) {
      $country_name = $this->getCountryName($address['country_code']);
      $parts[] = '<div class="country">' . $country_name . '</div>';
    }

    return '<div class="business-address">' . implode('', $parts) . '</div>';
  }

  /**
   * Formats address as inline text.
   */
  protected function formatAddressInline(array $address): string {
    $parts = [];

    if (!empty($address['address_line1'])) {
      $parts[] = $address['address_line1'];
    }
    if (!empty($address['address_line2'])) {
      $parts[] = $address['address_line2'];
    }
    if (!empty($address['locality'])) {
      $parts[] = $address['locality'];
    }
    if (!empty($address['administrative_area'])) {
      $parts[] = $address['administrative_area'];
    }
    if (!empty($address['postal_code'])) {
      $parts[] = $address['postal_code'];
    }
    if (!empty($address['country_code'])) {
      $parts[] = $this->getCountryName($address['country_code']);
    }

    return implode(', ', $parts);
  }

  /**
   * Formats address as plain text.
   */
  protected function formatAddressPlain(array $address): string {
    $parts = [];

    if (!empty($address['address_line1'])) {
      $parts[] = $address['address_line1'];
    }
    if (!empty($address['address_line2'])) {
      $parts[] = $address['address_line2'];
    }
    if (!empty($address['locality'])) {
      $parts[] = $address['locality'];
    }
    if (!empty($address['administrative_area'])) {
      $parts[] = $address['administrative_area'];
    }
    if (!empty($address['postal_code'])) {
      $parts[] = $address['postal_code'];
    }
    if (!empty($address['country_code'])) {
      $parts[] = $this->getCountryName($address['country_code']);
    }

    return implode("\n", $parts);
  }

  /**
   * Gets country name from code.
   */
  protected function getCountryName(string $country_code): string {
    $countries = \Drupal::service('address.country_repository')->getList();
    return $countries[$country_code] ?? $country_code;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperatingHours(): array {
    // Default operating hours - can be extended with a submodule
    $default_hours = [
      'monday' => ['open' => '09:00', 'close' => '17:00'],
      'tuesday' => ['open' => '09:00', 'close' => '17:00'],
      'wednesday' => ['open' => '09:00', 'close' => '17:00'],
      'thursday' => ['open' => '09:00', 'close' => '17:00'],
      'friday' => ['open' => '09:00', 'close' => '17:00'],
      'saturday' => ['open' => '10:00', 'close' => '14:00'],
      'sunday' => ['open' => '', 'close' => ''], // Closed
    ];

    // Allow modules to alter operating hours
    $hours = $this->config->get('operating_hours') ?? $default_hours;
    $this->moduleHandler->alter('business_identity_operating_hours', $hours);

    return $hours;
  }

  /**
   * {@inheritdoc}
   */
  public function isOpen(): bool {
    $hours = $this->getOperatingHours();
    $current_time = $this->time->getCurrentTime();
    $current_day = strtolower(date('l', $current_time));
    $current_hour = date('H:i', $current_time);

    if (!isset($hours[$current_day]) || empty($hours[$current_day]['open'])) {
      return FALSE;
    }

    $open_time = $hours[$current_day]['open'];
    $close_time = $hours[$current_day]['close'];

    return $current_hour >= $open_time && $current_hour <= $close_time;
  }

  /**
   * {@inheritdoc}
   */
  public function getContactMethods(): array {
    $contact = $this->getPrimaryContact();
    $methods = [];

    if (!empty($contact['email'])) {
      $methods['email'] = [
        'type' => 'email',
        'label' => $this->t('Email'),
        'value' => $contact['email'],
        'url' => 'mailto:' . $contact['email'],
      ];
    }

    if (!empty($contact['support_email'])) {
      $methods['support_email'] = [
        'type' => 'support_email',
        'label' => $this->t('Support Email'),
        'value' => $contact['support_email'],
        'url' => 'mailto:' . $contact['support_email'],
      ];
    }

    if (!empty($contact['phone'])) {
      $methods['phone'] = [
        'type' => 'phone',
        'label' => $this->t('Phone'),
        'value' => $contact['phone'],
        'url' => 'tel:' . preg_replace('/\D/', '', $contact['phone']),
      ];
    }

    if (!empty($contact['support_phone'])) {
      $methods['support_phone'] = [
        'type' => 'support_phone',
        'label' => $this->t('Support Phone'),
        'value' => $contact['support_phone'],
        'url' => 'tel:' . preg_replace('/\D/', '', $contact['support_phone']),
      ];
    }

    if (!empty($contact['website'])) {
      $methods['website'] = [
        'type' => 'website',
        'label' => $this->t('Website'),
        'value' => $contact['website'],
        'url' => $contact['website'],
      ];
    }

    // Add social media as contact methods
    $social_links = $this->getSocialLinks();
    foreach ($social_links as $platform => $url) {
      $methods[$platform] = [
        'type' => 'social',
        'label' => ucfirst($platform),
        'value' => $url,
        'url' => $url,
        'platform' => $platform,
      ];
    }

    return $methods;
  }

  /**
   * {@inheritdoc}
   */
  public function validateBusinessInfo(): array {
    $errors = [];
    $warnings = [];
    $info = $this->getBusinessInfo();

    // Required fields validation
    if (empty($info->getName())) {
      $errors[] = $this->t('Business name is required.');
    }

    if (empty($this->siteConfig->get('mail'))) {
      $errors[] = $this->t('Primary email is required.');
    }

    // Warning for recommended fields
    if (!$info->hasHeadquarters()) {
      $warnings[] = $this->t('Headquarters address is not set.');
    }

    if (empty($info->getPrimaryContact()['phone'])) {
      $warnings[] = $this->t('Primary phone number is not set.');
    }

    if (empty($info->getLegalInfo()['legal_name'])) {
      $warnings[] = $this->t('Legal name is not set.');
    }

    return [
      'errors' => $errors,
      'warnings' => $warnings,
      'completeness_score' => $this->calculateCompletenessScore($info),
      'is_valid' => empty($errors),
    ];
  }

  /**
   * Calculates business information completeness score.
   */
  protected function calculateCompletenessScore(BusinessInformation $info): int {
    $fields = [
      'name' => 15,
      'slogan' => 5,
      'description' => 10,
      'primary_contact.email' => 15,
      'primary_contact.phone' => 10,
      'locations.headquarters' => 20,
      'legal_info.legal_name' => 10,
      'legal_info.tax_id' => 5,
      'social_links' => 5,
      'website' => 5,
    ];

    $score = 0;

    // Check each field
    if (!empty($info->getName())) $score += $fields['name'];
    if (!empty($info->getSlogan())) $score += $fields['slogan'];
    if (!empty($info->getDescription())) $score += $fields['description'];

    $contact = $info->getPrimaryContact();
    if (!empty($contact['email'])) $score += $fields['primary_contact.email'];
    if (!empty($contact['phone'])) $score += $fields['primary_contact.phone'];

    if ($info->hasHeadquarters()) $score += $fields['locations.headquarters'];

    $legal = $info->getLegalInfo();
    if (!empty($legal['legal_name'])) $score += $fields['legal_info.legal_name'];
    if (!empty($legal['tax_id'])) $score += $fields['legal_info.tax_id'];

    if (!empty($info->getSocialLinks())) $score += $fields['social_links'];
    if (!empty($contact['website'])) $score += $fields['website'];

    return min(100, $score);
  }

  /**
   * Gets business name.
   */
  public function getBusinessName(): string {
    return $this->siteConfig->get('name') ?? '';
  }

  /**
   * Gets business slogan.
   */
  public function getBusinessSlogan(): string {
    return $this->siteConfig->get('slogan') ?? '';
  }

  /**
   * Gets business logo information.
   */
  public function getBusinessLogo(): array {
    $theme_config = $this->configFactory->get('system.theme.global');

    return [
      'url' => $theme_config->get('logo.url') ?? '',
      'path' => $theme_config->get('logo.path') ?? '',
    ];
  }

  /**
   * Gets business currency information.
   */
  public function getBusinessCurrency(): array {
    $commerce_exists = $this->moduleHandler->moduleExists('commerce_store');

    if ($commerce_exists) {
      // Get currency from Commerce Store
      $store_storage = \Drupal::entityTypeManager()->getStorage('commerce_store');
      $stores = $store_storage->loadMultiple();

      foreach ($stores as $store) {
        if ($store->isDefault()) {
          return [
            'code' => $store->getCurrencyCode(),
            'symbol' => $store->getCurrencySymbol(),
            'from_commerce' => TRUE,
          ];
        }
      }
    }

    // Fallback to config
    return [
      'code' => $this->config->get('currency') ?? 'USD',
      'symbol' => $this->getCurrencySymbol($this->config->get('currency') ?? 'USD'),
      'from_commerce' => FALSE,
    ];
  }

  /**
   * Gets currency symbol.
   */
  protected function getCurrencySymbol(string $currency_code): string {
    $symbols = [
      'USD' => '$',
      'EUR' => '€',
      'GBP' => '£',
      'JPY' => '¥',
      'CHF' => 'CHF',
      'CAD' => 'C$',
      'AUD' => 'A$',
    ];

    return $symbols[$currency_code] ?? $currency_code;
  }

  /**
   * Translates a string.
   */
  protected function t(string $string, array $args = []): string {
    return \Drupal::translation()->translate($string, $args);
  }

}
