<?php

declare(strict_types=1);

namespace Drupal\business_identity\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Business Identity settings.
 *
 * @package Drupal\business_identity\Form
 */
class BusinessIdentityForm extends ConfigFormBase {

  /**
   * Business types.
   *
   * @var array
   */
  protected const BUSINESS_TYPES = [
    'restaurant' => 'Restaurant',
    'retail' => 'Retail',
    'services' => 'Professional Services',
    'manufacturing' => 'Manufacturing',
    'tourism' => 'Tourism',
    'health' => 'Healthcare',
    'education' => 'Education',
    'other' => 'Other',
  ];

  /**
   * Social media platforms.
   *
   * @var array
   */
  protected const SOCIAL_PLATFORMS = [
    'facebook' => 'Facebook',
    'twitter' => 'Twitter / X',
    'linkedin' => 'LinkedIn',
    'instagram' => 'Instagram',
    'youtube' => 'YouTube',
    'pinterest' => 'Pinterest',
    'tiktok' => 'TikTok',
  ];

  /**
   * Review platforms.
   *
   * @var array
   */
  protected const REVIEW_PLATFORMS = [
    'google' => 'Google Reviews',
    'facebook' => 'Facebook Reviews',
    'yelp' => 'Yelp',
    'tripadvisor' => 'TripAdvisor',
    'trustpilot' => 'Trustpilot',
    'glassdoor' => 'Glassdoor',
    'bbb' => 'Better Business Bureau (BBB)',
    'angieslist' => 'Angie\'s List',
    'homeadvisor' => 'HomeAdvisor',
    'thumbtack' => 'Thumbtack',
    'other' => 'Other Platforms',
  ];

  /**
   * CTA Button styles.
   *
   * @var array
   */
  protected const CTA_STYLES = [
    'primary' => 'Primary (Main call-to-action)',
    'secondary' => 'Secondary (Alternative)',
    'outline' => 'Outline (Bordered)',
    'text' => 'Text Only (Link style)',
    'custom' => 'Custom CSS class',
  ];

  /**
   * Sync marker for synchronized fields.
   *
   * @var string
   */
  protected const SYNC_MARKER = ' 🔗';

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface|null
   */
  protected $keyRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->languageManager = $container->get('language_manager');
    $instance->dateFormatter = $container->get('date.formatter');
    $instance->entityTypeManager = $container->get('entity_type.manager');

    // Key repository is optional - only load if module exists.
    if ($container->has('key.repository')) {
      $instance->keyRepository = $container->get('key.repository');
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'business_identity_admin_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'business_identity.settings',
      'system.site',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('business_identity.settings');
    $site_config = $this->config('system.site');

    // Check module dependencies.
    $key_module_exists = $this->moduleHandler->moduleExists('key');
    $address_module_exists = $this->moduleHandler->moduleExists('address');
    $commerce_store_exists = $this->moduleHandler->moduleExists('commerce_store');
    $geolocation_module_exists = $this->moduleHandler->moduleExists('geolocation');

    // Vertical tabs wrapper.
    $form['#attached']['library'][] = 'business_identity/vertical_tabs';

    $form['business_tabs'] = [
      '#type' => 'vertical_tabs',
      '#default_tab' => 'edit-basic',
      '#attributes' => [
        'class' => ['business-identity-tabs'],
      ],
    ];

    // Add synchronization information message at the top.
    $form['sync_info'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['messages', 'messages--status', 'sync-info'],
      ],
      '#weight' => -100,
    ];

    $form['sync_info']['message'] = [
      '#markup' => '<h3>' . $this->t('Synchronization Information') . '</h3>' .
        '<p><strong>' . self::SYNC_MARKER . ' ' .
        $this->t('= Synchronized Field') . '</strong></p>' .
        '<p>' . $this->t('Fields marked with 🔗 are synchronized with Drupal core settings. Changes to these fields will update both this module and the corresponding core configuration.') . '</p>' .
        '<p>' . $this->t('Synchronized fields are read-only in this form. To modify them, use the linked core configuration pages.') . '</p>',
    ];

    // 1. TAB: Basic Information - Synchronized with Site Information.
    $form['basic'] = [
      '#type' => 'details',
      '#title' => $this->t('Basic Information'),
      '#group' => 'business_tabs',
      '#open' => TRUE,
      '#description' => $this->t('These fields are synchronized with the <a href=":url">Site Information</a> settings.', [
        ':url' => Url::fromRoute('system.site_information_settings')->toString(),
      ]),
    ];

    $form['basic']['company_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company Name') . self::SYNC_MARKER,
      '#default_value' => $site_config->get('name'),
      '#required' => TRUE,
      '#maxlength' => 255,
      '#description' => $this->t('The official name of your company. This is synchronized with the site name.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    $form['basic']['slogan'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Slogan/Motto') . self::SYNC_MARKER,
      '#default_value' => $site_config->get('slogan'),
      '#maxlength' => 100,
      '#description' => $this->t('Company tagline or motto. This is synchronized with the site slogan.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    $form['basic']['site_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit site information →'),
      '#url' => Url::fromRoute('system.site_information_settings'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    $form['basic']['separator'] = [
      '#markup' => '<hr class="sync-separator">',
    ];

    // Non-synchronized fields.
    $form['basic']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Company Description'),
      '#default_value' => $config->get('description'),
      '#rows' => 4,
      '#maxlength' => 500,
      '#description' => $this->t('Brief description of your company (max 500 characters).'),
    ];

    // Site logo from system.theme.global config.
    $global_theme_config = $this->config('system.theme.global');
    $logo_uri = $global_theme_config->get('logo.url') ?:
      $global_theme_config->get('logo.path');

    $form['basic']['logo_info'] = [
      '#type' => 'item',
      '#title' => $this->t('Company Logo') . self::SYNC_MARKER,
      '#markup' => $logo_uri ?
        '<p>' . $this->t('Current logo: <strong>@logo</strong>', [
          '@logo' => $logo_uri,
        ]) . '</p>' :
        '<p>' . $this->t('No logo currently set.') . '</p>',
      '#description' => $this->t('The site logo is managed in theme settings.'),
    ];

    $form['basic']['theme_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit theme settings →'),
      '#url' => Url::fromRoute('system.theme_settings'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    // 2. TAB: Contacts.
    $form['contacts'] = [
      '#type' => 'details',
      '#title' => $this->t('Contacts'),
      '#group' => 'business_tabs',
    ];

    // Email from site configuration - SYNCHRONIZED.
    $form['contacts']['email'] = [
      '#type' => 'email',
      '#title' => $this->t('Primary Email') . self::SYNC_MARKER,
      '#default_value' => $site_config->get('mail'),
      '#required' => TRUE,
      '#description' => $this->t('The official site email. This is synchronized with the site email.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Customer support email.
    $form['contacts']['support_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Customer Support Email'),
      '#default_value' => $config->get('support_email'),
      '#description' => $this->t('Dedicated email address for customer support inquiries.'),
    ];

    // Phone numbers.
    $form['contacts']['phone'] = [
      '#type' => 'tel',
      '#title' => $this->t('Main Phone Number'),
      '#default_value' => $config->get('phone'),
      '#required' => FALSE,
    ];

    $form['contacts']['support_phone'] = [
      '#type' => 'tel',
      '#title' => $this->t('Support Phone Number'),
      '#default_value' => $config->get('support_phone'),
    ];

    // 3. TAB: Address (NEW - Single location address).
    $form['address'] = [
      '#type' => 'details',
      '#title' => $this->t('Business Address'),
      '#group' => 'business_tabs',
      '#description' => $this->t('Primary business location address and coordinates.'),
    ];

    // ADD: Text area "Where are we"
    $form['address']['where_are_we'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Where are we?'),
      '#default_value' => $config->get('where_are_we'),
      '#rows' => 4,
      '#maxlength' => 1000,
      '#description' => $this->t('Provide a descriptive text about your location, how to reach you, parking information, public transportation options, landmarks, or any other relevant location details.'),
      '#weight' => -10,
    ];

    // Address field if Address module is available.
    if ($address_module_exists) {
      $address_config = $config->get('business_address') ?? [];

      $form['address']['business_address'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Street Address'),
        '#tree' => TRUE,
      ];

      $form['address']['business_address']['address'] = [
        '#type' => 'address',
        '#title' => $this->t('Business Address'),
        '#default_value' => $address_config,
        '#available_countries' => [
          'IT',
          'US',
          'GB',
          'FR',
          'DE',
          'ES',
          'CH',
          'BE',
          'NL',
          'PT',
          'CA',
          'AU',
        ],
        '#required' => FALSE,
        '#description' => $this->t('The primary business location address.'),
      ];

      // Link to Address module if installed.
      if ($this->moduleHandler->moduleExists('address')) {
        $form['address']['address_note'] = [
          '#markup' => '<div class="messages messages--info"><p>' .
            $this->t('Using structured address format via the Address module.') . '</p></div>',
        ];
      }
    }
    else {
      $form['address']['address_warning'] = [
        '#markup' => '<div class="messages messages--warning"><p>' .
          $this->t('Install the <a href=":url">Address module</a> for structured address management.', [
            ':url' => 'https://www.drupal.org/project/address',
          ]) . '</p></div>',
      ];

      // Fallback: Simple address text fields.
      $form['address']['simple_address'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Street Address'),
      ];

      $form['address']['simple_address']['street'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Street Address'),
        '#default_value' => $config->get('address_street'),
        '#maxlength' => 255,
        '#description' => $this->t('Street address and number.'),
      ];

      $form['address']['simple_address']['city'] = [
        '#type' => 'textfield',
        '#title' => $this->t('City'),
        '#default_value' => $config->get('address_city'),
        '#maxlength' => 100,
      ];

      $form['address']['simple_address']['state'] = [
        '#type' => 'textfield',
        '#title' => $this->t('State/Province'),
        '#default_value' => $config->get('address_state'),
        '#maxlength' => 100,
      ];

      $form['address']['simple_address']['postal_code'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Postal/ZIP Code'),
        '#default_value' => $config->get('address_postal_code'),
        '#maxlength' => 20,
      ];

      $form['address']['simple_address']['country'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Country'),
        '#default_value' => $config->get('address_country'),
        '#maxlength' => 100,
        '#default_value' => 'United States',
      ];
    }

    // Geolocation coordinates section.
    $form['address']['geolocation'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Geographic Coordinates'),
      '#description' => $this->t('Optional: Latitude and longitude for mapping services.'),
      '#prefix' => '<div id="geolocation-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['address']['geolocation']['lat'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Latitude'),
      '#default_value' => $config->get('address_lat'),
      '#size' => 20,
      '#maxlength' => 20,
      '#placeholder' => '40.7128',
      '#description' => $this->t('Decimal degrees format: -90.0000 to 90.0000. Example: 40.7128 (New York City)'),
      '#attributes' => [
        'pattern' => '^-?\d{1,3}\.\d+$',
        'title' => $this->t('Decimal degrees format: -90.0000 to 90.0000'),
      ],
    ];

    $form['address']['geolocation']['lng'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Longitude'),
      '#default_value' => $config->get('address_lng'),
      '#size' => 20,
      '#maxlength' => 20,
      '#placeholder' => '-74.0060',
      '#description' => $this->t('Decimal degrees format: -180.0000 to 180.0000. Example: -74.0060 (New York City)'),
      '#attributes' => [
        'pattern' => '^-?\d{1,3}\.\d+$',
        'title' => $this->t('Decimal degrees format: -180.0000 to 180.0000'),
      ],
    ];

    // Geolocate button if we have address data.
    if ($address_module_exists && !empty($address_config['address_line1'])) {
      $form['address']['geolocation']['geolocate_button'] = [
        '#type' => 'button',
        '#value' => $this->t('Geolocate from address'),
        '#name' => 'geolocate_address',
        '#attributes' => [
          'class' => ['button--small'],
        ],
        '#ajax' => [
          'callback' => [$this, 'geolocateBusinessAddress'],
          'wrapper' => 'geolocation-wrapper',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Geolocating...'),
          ],
        ],
      ];
    }
    elseif (!$address_module_exists && !empty($config->get('address_street'))) {
      $form['address']['geolocation']['geolocate_button'] = [
        '#type' => 'button',
        '#value' => $this->t('Geolocate from address'),
        '#name' => 'geolocate_address_simple',
        '#attributes' => [
          'class' => ['button--small'],
        ],
        '#ajax' => [
          'callback' => [$this, 'geolocateBusinessAddress'],
          'wrapper' => 'geolocation-wrapper',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Geolocating...'),
          ],
        ],
      ];
    }

    // Note about geolocation.
    if ($geolocation_module_exists) {
      $form['address']['geolocation_note'] = [
        '#markup' => '<div class="messages messages--status"><p>' .
          $this->t('📍 <strong>Geolocation enabled:</strong> Coordinates can be used with mapping modules.') .
          '</p></div>',
      ];
    }

    // 4. TAB: Legal Information.
    $form['legal'] = [
      '#type' => 'details',
      '#title' => $this->t('Legal Information'),
      '#group' => 'business_tabs',
    ];

    $form['legal']['legal_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Legal Name'),
      '#default_value' => $config->get('legal_name'),
      '#description' => $this->t('Official registered legal name of the company.'),
    ];

    $form['legal']['tax_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tax ID / VAT Number'),
      '#default_value' => $config->get('tax_id'),
      '#description' => $this->t('Tax identification number or VAT number.'),
    ];

    // 5. TAB: Online Presence.
    $form['online'] = [
      '#type' => 'details',
      '#title' => $this->t('Online Presence'),
      '#group' => 'business_tabs',
    ];

    // Front page URL from site configuration - SYNCHRONIZED.
    $front_page = $site_config->get('page.front') ?? '/node';
    $form['online']['front_page'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Front Page') . self::SYNC_MARKER,
      '#default_value' => $front_page,
      '#description' => $this->t('Default front page. Configure in Site Information.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    $form['online']['social_links'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Social Media'),
      '#tree' => TRUE,
    ];

    $social_links = $config->get('social_links') ?? [];

    foreach (self::SOCIAL_PLATFORMS as $platform => $label) {
      $form['online']['social_links'][$platform] = [
        '#type' => 'url',
        '#title' => $label,
        '#default_value' => $social_links[$platform] ?? '',
        '#placeholder' => 'https://',
      ];
    }

    // 6. TAB: Operations.
    $form['operations'] = [
      '#type' => 'details',
      '#title' => $this->t('Operations'),
      '#group' => 'business_tabs',
    ];

    $form['operations']['business_sector'] = [
      '#type' => 'select',
      '#title' => $this->t('Business Sector'),
      '#options' => self::BUSINESS_TYPES,
      '#default_value' => $config->get('business_sector') ?? 'other',
      '#empty_option' => $this->t('- Select -'),
    ];

    $form['operations']['founding_year'] = [
      '#type' => 'number',
      '#title' => $this->t('Year Founded'),
      '#default_value' => $config->get('founding_year'),
      '#min' => 1800,
      '#max' => (int) date('Y'),
      '#step' => 1,
      '#description' => $this->t('Year the company was founded.'),
    ];

    // 7. TAB: System Configuration.
    $form['system'] = [
      '#type' => 'details',
      '#title' => $this->t('System Configuration'),
      '#group' => 'business_tabs',
      '#description' => $this->t('These settings are synchronized with Drupal core configuration.'),
    ];

    // Check if Commerce Store module exists for currency.
    if ($commerce_store_exists) {
      // Load the primary store's currency from Commerce configuration.
      $store_storage = $this->entityTypeManager->getStorage('commerce_store');
      $stores = $store_storage->loadMultiple();
      $primary_store_currency = 'USD';

      foreach ($stores as $store) {
        if ($store->isDefault()) {
          $primary_store_currency = $store->getCurrencyCode();
          break;
        }
      }

      $form['system']['store_currency_info'] = [
        '#type' => 'item',
        '#title' => $this->t('Primary Store Currency') . ' 💳',
        '#markup' => $this->t('<strong>@currency</strong> – Managed by <a href=":url">Commerce Store</a>', [
          '@currency' => $primary_store_currency,
          // ':url' => Url::fromRoute('entity.commerce_store.collection')->toString(),
        ]),
        '#description' => $this->t('The primary currency is configured in your Commerce Store settings.'),
      ];
    }
    else {
      $form['system']['currency'] = [
        '#type' => 'select',
        '#title' => $this->t('Default Currency'),
        '#options' => [
          'USD' => 'US Dollar ($)',
          'EUR' => 'Euro (€)',
          'GBP' => 'British Pound (£)',
          'JPY' => 'Japanese Yen (¥)',
          'CHF' => 'Swiss Franc (CHF)',
          'CAD' => 'Canadian Dollar (C$)',
          'AUD' => 'Australian Dollar (A$)',
        ],
        '#default_value' => $config->get('currency') ?? 'USD',
        '#description' => $this->t('Install the <a href=":url">Commerce module</a> to manage currencies via Store configuration.', [
          ':url' => 'https://www.drupal.org/project/commerce',
        ]),
      ];
    }

    // Date format from Drupal core - SYNCHRONIZED.
    $date_format_storage = $this->entityTypeManager->getStorage('date_format');
    $date_formats = $date_format_storage->loadMultiple();

    $date_format_options = [];
    foreach ($date_formats as $date_format) {
      if ($date_format->status()) {
        $example = $this->dateFormatter->format(time(), $date_format->id());
        $date_format_options[$date_format->id()] = $this->t(
          '@label (@example)',
          [
            '@label' => $date_format->label(),
            '@example' => $example,
          ]
        );
      }
    }

    $form['system']['date_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Date Format') . self::SYNC_MARKER,
      '#options' => $date_format_options,
      '#default_value' => $this->config('system.date')
        ->get('date_format.short'),
      '#description' => $this->t('Select a date format. Manage formats at Date and time formats.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Time format from Drupal core - SYNCHRONIZED.
    $time_format_storage = $this->entityTypeManager->getStorage('date_format');
    $time_formats = $time_format_storage->loadByProperties(['is_time' => TRUE]);

    $time_format_options = [];
    foreach ($time_formats as $time_format) {
      if ($time_format->status()) {
        $example = $this->dateFormatter->format(time(), $time_format->id());
        $time_format_options[$time_format->id()] = $this->t(
          '@label (@example)',
          [
            '@label' => $time_format->label(),
            '@example' => $example,
          ]
        );
      }
    }

    $form['system']['time_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Time Format') . self::SYNC_MARKER,
      '#options' => $time_format_options,
      '#default_value' => $this->config('system.date')
        ->get('time_format.short'),
      '#description' => $this->t('Select a time format.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Timezone from Drupal core - SYNCHRONIZED.
    $default_timezone = $this->config('system.date')
        ->get('timezone.default') ?? date_default_timezone_get();
    $form['system']['timezone'] = [
      '#type' => 'select',
      '#title' => $this->t('Timezone') . self::SYNC_MARKER,
      '#options' => $this->getTimezoneOptions(),
      '#default_value' => $default_timezone,
      '#description' => $this->t('Select the timezone.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Link to regional settings.
    $regional_url = Url::fromRoute('system.regional_settings');
    $form['system']['regional_settings'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit regional settings →'),
      '#url' => $regional_url,
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];

    // 8. TAB: Ecommerce (NEW TAB)
    $form['ecommerce'] = [
      '#type' => 'details',
      '#title' => $this->t('Ecommerce'),
      '#group' => 'business_tabs',
      '#description' => $this->t('Configure ecommerce settings and killer call-to-action links.'),
    ];

    // Check if Commerce module is enabled.
    $commerce_enabled = $this->moduleHandler->moduleExists('commerce');

    if ($commerce_enabled) {
      $form['ecommerce']['commerce_info'] = [
        '#markup' => '<div class="messages messages--status">' .
          $this->t('🎯 <strong>Commerce module detected!</strong> Your ecommerce functionality is managed through Drupal Commerce.') .
          '</div>',
      ];

      // Link to Commerce store configuration.
      $form['ecommerce']['commerce_link'] = [
        '#type' => 'link',
        '#title' => $this->t('Manage Commerce Store →'),
        // '#url' => Url::fromRoute('entity.commerce_store.collection'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
        ],
        '#prefix' => '<div class="sync-action-link">',
        '#suffix' => '</div>',
      ];

      $form['ecommerce']['separator'] = [
        '#markup' => '<hr>',
      ];
    }

    // Killer Link: Buy Now
    $form['ecommerce']['buy_now'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Buy Now - Killer Link'),
      '#description' => $this->t('Configure the primary call-to-action button that appears throughout your site.'),
    ];

    $form['ecommerce']['buy_now']['buy_now_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable "Buy Now" button'),
      '#default_value' => $config->get('buy_now_enabled') ?? TRUE,
      '#description' => $this->t('Show the "Buy Now" button in headers, footers, and key pages.'),
    ];

    $form['ecommerce']['buy_now']['buy_now_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button Label'),
      '#default_value' => $config->get('buy_now_label') ?? $this->t('Buy Now'),
      '#description' => $this->t('Text displayed on the button.'),
      '#size' => 30,
      '#maxlength' => 50,
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['ecommerce']['buy_now']['buy_now_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Button URL'),
      '#default_value' => $config->get('buy_now_url') ?? '',
      '#description' => $this->t('Link destination (e.g., /products, /store, or external URL like https://checkout.example.com).'),
      '#placeholder' => 'https://',
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Open in new window option.
    $form['ecommerce']['buy_now']['buy_now_new_window'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Open in new window'),
      '#default_value' => $config->get('buy_now_new_window') ?? FALSE,
      '#description' => $this->t('Open the link in a new browser tab.'),
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Button style.
    $form['ecommerce']['buy_now']['buy_now_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Button Style'),
      '#options' => self::CTA_STYLES,
      '#default_value' => $config->get('buy_now_style') ?? 'primary',
      '#description' => $this->t('Visual style for the button.'),
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Custom CSS class.
    $form['ecommerce']['buy_now']['buy_now_custom_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom CSS Class'),
      '#default_value' => $config->get('buy_now_custom_class') ?? '',
      '#description' => $this->t('Additional CSS class for custom styling.'),
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
          ':input[name="buy_now_style"]' => ['value' => 'custom'],
        ],
      ],
    ];

    // Icon for the button.
    $form['ecommerce']['buy_now']['buy_now_icon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button Icon'),
      '#default_value' => $config->get('buy_now_icon') ?? '',
      '#description' => $this->t('Optional: CSS class for an icon (e.g., "fa-shopping-cart", "bi-bag"). Leave empty for no icon.'),
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Display settings.
    $form['ecommerce']['display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Settings'),
      '#states' => [
        'visible' => [
          ':input[name="buy_now_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['ecommerce']['display']['buy_now_header'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show in header'),
      '#default_value' => $config->get('buy_now_header') ?? TRUE,
      '#description' => $this->t('Display the button in the site header.'),
    ];

    $form['ecommerce']['display']['buy_now_footer'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show in footer'),
      '#default_value' => $config->get('buy_now_footer') ?? TRUE,
      '#description' => $this->t('Display the button in the site footer.'),
    ];

    $form['ecommerce']['display']['buy_now_sidebar'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show in sidebar'),
      '#default_value' => $config->get('buy_now_sidebar') ?? FALSE,
      '#description' => $this->t('Display the button in sidebars (if applicable).'),
    ];

    $form['ecommerce']['display']['buy_now_product_pages'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show on product pages'),
      '#default_value' => $config->get('buy_now_product_pages') ?? TRUE,
      '#description' => $this->t('Display the button on product detail pages.'),
    ];

    // Alternative CTA (for A/B testing or different flows).
    $form['ecommerce']['alternative'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Alternative Call-to-Action'),
      '#description' => $this->t('Optional secondary call-to-action for different user flows.'),
    ];

    $form['ecommerce']['alternative']['alt_cta_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable alternative CTA'),
      '#default_value' => $config->get('alt_cta_enabled') ?? FALSE,
      '#description' => $this->t('Show an alternative call-to-action (e.g., "Learn More", "Get Quote").'),
    ];

    $form['ecommerce']['alternative']['alt_cta_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Alternative Label'),
      '#default_value' => $config->get('alt_cta_label') ?? $this->t('Learn More'),
      '#description' => $this->t('Text for the alternative button.'),
      '#states' => [
        'visible' => [
          ':input[name="alt_cta_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['ecommerce']['alternative']['alt_cta_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Alternative URL'),
      '#default_value' => $config->get('alt_cta_url') ?? '',
      '#description' => $this->t('Link destination for the alternative CTA.'),
      '#states' => [
        'visible' => [
          ':input[name="alt_cta_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // 9. TAB: Business Hours (Office Hours integration).
    $form['office_hours'] = [
      '#type' => 'details',
      '#title' => $this->t('Business Hours'),
      '#group' => 'business_tabs',
      '#description' => $this->t('Configure your business opening hours.'),
      '#weight' => 50, // Regola il peso per la posizione nella vertical tabs
    ];

    // Check if the Office Hours module is available.
    $office_hours_module_exists = $this->moduleHandler->moduleExists('office_hours');

    if ($office_hours_module_exists) {
      // Office Hours module is installed - provide integration.
      $form['office_hours']['module_info'] = [
        '#markup' => '<div class="messages messages--status">' .
          $this->t('🕒 <strong>Office Hours module is active!</strong> You can manage your business hours through the dedicated field on content types.') .
          '</div>',
      ];

      // Load any existing office hours configuration.
      $office_hours_config = $config->get('office_hours') ?? [];
      $form['office_hours']['configuration'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Hours Configuration'),
        '#description' => $this->t('Note: For full functionality, add an "Office Hours" field to your content types.'),
      ];

      // Example: Provide a simplified configuration.
      $form['office_hours']['configuration']['timezone'] = [
        '#type' => 'select',
        '#title' => $this->t('Hours Timezone'),
        '#options' => $this->getTimezoneOptions(),
        '#default_value' => $office_hours_config['timezone'] ?? $this->config('system.date')->get('timezone.default'),
        '#description' => $this->t('Timezone for your business hours.'),
      ];

      $form['office_hours']['configuration']['hours_format'] = [
        '#type' => 'select',
        '#title' => $this->t('Time Format'),
        '#options' => [
          'G:i' => $this->t('24-hour format (14:00)'),
          'g:i a' => $this->t('12-hour format (2:00 pm)'),
        ],
        '#default_value' => $office_hours_config['hours_format'] ?? 'G:i',
        '#description' => $this->t('Display format for hours.'),
      ];

      // Link to content types to add office hours field.
      $form['office_hours']['manage_link'] = [
        '#type' => 'link',
        '#title' => $this->t('Manage Office Hours fields →'),
        '#url' => Url::fromRoute('entity.node_type.collection'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
        ],
        '#prefix' => '<div class="sync-action-link">',
        '#suffix' => '</div>',
      ];

      // Add a token for business hours.
      $form['tokens']['description']['#markup'] .= '<p>' . $this->t('Office Hours tokens available when module is enabled:') . '</p>';
      $tokens['[business:office_hours]'] = $this->t('Formatted business hours');
      $tokens['[business:office_hours_today]'] = $this->t("Today's hours");
      $tokens['[business:office_hours_next_open]'] = $this->t('Next opening time');

    } else {
      // Office Hours module is NOT installed - show installation prompt.
      $form['office_hours']['module_warning'] = [
        '#markup' => '<div class="messages messages--warning">' .
          $this->t('The <strong>Office Hours module is not installed</strong>.') .
          '</div>',
      ];

      $form['office_hours']['installation'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Installation Instructions'),
        '#description' => $this->t('To manage business hours effectively, install the Office Hours module.'),
      ];

      $form['office_hours']['installation']['steps'] = [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Install via Composer: <code>composer require drupal/office_hours</code>'),
          $this->t('Enable the module at <a href=":url">Extend</a>.', [
            ':url' => Url::fromRoute('system.modules_list')->toString(),
          ]),
          $this->t('Add an "Office Hours" field to your content types.'),
        ],
      ];

      // Link to the Office Hours module page.
      $form['office_hours']['installation']['module_link'] = [
        '#type' => 'link',
        '#title' => $this->t('Visit Office Hours project page →'),
        '#url' => Url::fromUri('https://www.drupal.org/project/office_hours'),
        '#attributes' => [
          'class' => ['button', 'button--small', 'button--primary'],
          'target' => '_blank',
        ],
        '#prefix' => '<div class="install-link">',
        '#suffix' => '</div>',
      ];

      // Provide simple fallback fields.
      $form['office_hours']['fallback'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Basic Hours (Fallback)'),
        '#description' => $this->t('Simple hours configuration until module is installed.'),
      ];

      $form['office_hours']['fallback']['business_hours_text'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Business Hours Description'),
        '#default_value' => $config->get('business_hours_text') ?? '',
        '#rows' => 4,
        '#description' => $this->t('Describe your business hours (e.g., "Monday-Friday: 9:00-18:00, Saturday: 10:00-14:00").'),
      ];
    }

    // 9.b TAB: Security and API Keys.
    $form['security'] = [
      '#type' => 'details',
      '#title' => $this->t('Security and API Keys'),
      '#group' => 'business_tabs',
    ];

    if ($key_module_exists && $this->keyRepository) {
      // Get available keys from Key module.
      $keys = $this->keyRepository->getKeys();

      $key_options = ['' => $this->t('- Select a key -')];
      foreach ($keys as $key) {
        $key_options[$key->id()] = $key->label();
      }

      $form['security']['google_maps_key'] = [
        '#type' => 'select',
        '#title' => $this->t('Google Maps API Key'),
        '#options' => $key_options,
        '#default_value' => $config->get('google_maps_key'),
        '#description' => $this->t('Select a key from the Key module.'),
      ];

      $form['security']['recaptcha_site_key'] = [
        '#type' => 'select',
        '#title' => $this->t('reCAPTCHA Site Key'),
        '#options' => $key_options,
        '#default_value' => $config->get('recaptcha_site_key'),
        '#description' => $this->t('Select a key from the Key module.'),
      ];

      $form['security']['recaptcha_secret_key'] = [
        '#type' => 'select',
        '#title' => $this->t('reCAPTCHA Secret Key'),
        '#options' => $key_options,
        '#default_value' => $config->get('recaptcha_secret_key'),
        '#description' => $this->t('Select a key from the Key module.'),
      ];

      $form['security']['key_management'] = [
        '#type' => 'link',
        '#title' => $this->t('Manage API Keys →'),
        // '#url' => Url::fromRoute('entity.key.collection'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
        ],
        '#prefix' => '<div class="sync-action-link">',
        '#suffix' => '</div>',
      ];
    }
    else {
      $form['security']['key_module_info'] = [
        '#markup' => '<div class="messages messages--info">' .
          $this->t('Install the <a href=":url">Key module</a> for secure API key management.', [
            ':url' => 'https://www.drupal.org/project/key',
          ]) . '</div>',
      ];

      // Fallback to text fields if Key module not available.
      $form['security']['google_maps_key_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Google Maps API Key'),
        '#default_value' => $config->get('google_maps_key'),
        '#description' => $this->t('Enter your Google Maps API key.'),
      ];

      $form['security']['recaptcha_site_key_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('reCAPTCHA Site Key'),
        '#default_value' => $config->get('recaptcha_site_key'),
        '#description' => $this->t('Enter your reCAPTCHA site key.'),
      ];

      $form['security']['recaptcha_secret_key_text'] = [
        '#type' => 'textfield',
        '#title' => $this->t('reCAPTCHA Secret Key'),
        '#default_value' => $config->get('recaptcha_secret_key'),
        '#description' => $this->t('Enter your reCAPTCHA secret key.'),
      ];
    }

    // 10. TAB: Company Reviews (NEW TAB)
    $form['reviews'] = [
      '#type' => 'details',
      '#title' => $this->t('Company Reviews'),
      '#group' => 'business_tabs',
      '#description' => $this->t('Configure review platform settings and manage your company\'s online reputation.'),
    ];

    // Enable/disable reviews feature.
    $form['reviews']['reviews_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable reviews feature'),
      '#default_value' => $config->get('reviews_enabled') ?? FALSE,
      '#description' => $this->t('When enabled, review links and ratings will be displayed across the site.'),
    ];

    // Overall rating.
    $form['reviews']['overall_rating'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Overall Rating'),
      '#states' => [
        'visible' => [
          ':input[name="reviews_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['reviews']['overall_rating']['rating_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Average Rating'),
      '#default_value' => $config->get('rating_value') ?? 0,
      '#min' => 0,
      '#max' => 5,
      '#step' => 0.1,
      '#field_suffix' => '/5',
      '#description' => $this->t('Your company\'s average rating across all platforms (0-5).'),
    ];

    $form['reviews']['overall_rating']['rating_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Total Reviews'),
      '#default_value' => $config->get('rating_count') ?? 0,
      '#min' => 0,
      '#step' => 1,
      '#description' => $this->t('Total number of reviews across all platforms.'),
    ];

    $form['reviews']['overall_rating']['last_updated'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Last Updated'),
      '#default_value' => $config->get('rating_last_updated') ?? '',
      '#description' => $this->t('Date when ratings were last updated (e.g., January 2024).'),
      '#size' => 20,
    ];

    // Review platforms configuration.
    $form['reviews']['platforms'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Review Platforms'),
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="reviews_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $review_platforms_config = $config->get('review_platforms') ?? [];

    foreach (self::REVIEW_PLATFORMS as $platform => $label) {
      $form['reviews']['platforms'][$platform] = [
        '#type' => 'fieldset',
        '#title' => $label,
        '#collapsible' => TRUE,
        '#collapsed' => TRUE,
      ];

      $form['reviews']['platforms'][$platform]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable @platform', ['@platform' => $label]),
        '#default_value' => $review_platforms_config[$platform]['enabled'] ?? FALSE,
      ];

      $form['reviews']['platforms'][$platform]['url'] = [
        '#type' => 'url',
        '#title' => $this->t('Review Page URL'),
        '#default_value' => $review_platforms_config[$platform]['url'] ?? '',
        '#placeholder' => 'https://',
        '#states' => [
          'visible' => [
            ':input[name="platforms[' . $platform . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['reviews']['platforms'][$platform]['rating'] = [
        '#type' => 'number',
        '#title' => $this->t('Platform Rating'),
        '#default_value' => $review_platforms_config[$platform]['rating'] ?? 0,
        '#min' => 0,
        '#max' => 5,
        '#step' => 0.1,
        '#field_suffix' => '/5',
        '#states' => [
          'visible' => [
            ':input[name="platforms[' . $platform . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['reviews']['platforms'][$platform]['count'] = [
        '#type' => 'number',
        '#title' => $this->t('Number of Reviews'),
        '#default_value' => $review_platforms_config[$platform]['count'] ?? 0,
        '#min' => 0,
        '#step' => 1,
        '#states' => [
          'visible' => [
            ':input[name="platforms[' . $platform . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      // For "other" platforms, allow custom name
      if ($platform === 'other') {
        $form['reviews']['platforms'][$platform]['custom_name'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Platform Name'),
          '#default_value' => $review_platforms_config[$platform]['custom_name'] ?? '',
          '#description' => $this->t('Enter the name of the review platform.'),
          '#states' => [
            'visible' => [
              ':input[name="platforms[' . $platform . '][enabled]"]' => ['checked' => TRUE],
            ],
          ],
        ];
      }
    }

    // Review request settings.
    $form['reviews']['review_request'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Review Request Settings'),
      '#states' => [
        'visible' => [
          ':input[name="reviews_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['reviews']['review_request']['request_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Review Request Message'),
      '#default_value' => $config->get('review_request_message') ?? $this->t('Help others by sharing your experience! Please consider leaving a review.'),
      '#rows' => 3,
      '#description' => $this->t('Message to display when asking customers for reviews.'),
    ];

    $form['reviews']['review_request']['primary_platform'] = [
      '#type' => 'select',
      '#title' => $this->t('Primary Review Platform'),
      '#options' => self::REVIEW_PLATFORMS,
      '#default_value' => $config->get('primary_review_platform') ?? 'google',
      '#description' => $this->t('The main platform where you\'d like customers to leave reviews.'),
    ];

    $form['reviews']['review_request']['email_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Review Request Email Subject'),
      '#default_value' => $config->get('review_email_subject') ?? $this->t('Share your experience with [site:name]'),
      '#description' => $this->t('Subject line for review request emails.'),
    ];

    $form['reviews']['review_request']['email_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Review Request Email Template'),
      '#default_value' => $config->get('review_email_body') ?? $this->t("Dear [customer:name],\n\nThank you for choosing [site:name]! We hope you were satisfied with our services.\n\nWould you mind taking a moment to share your experience? Your feedback helps us improve and helps others make informed decisions.\n\nLeave a review here: [business:primary_review_url]\n\nThank you,\nThe [site:name] Team"),
      '#rows' => 6,
      '#description' => $this->t('Email template for review requests. Use tokens like [site:name] and [business:primary_review_url].'),
    ];

    // Badge display settings.
    $form['reviews']['display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Settings'),
      '#states' => [
        'visible' => [
          ':input[name="reviews_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['reviews']['display']['show_badge'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display rating badge'),
      '#default_value' => $config->get('show_rating_badge') ?? TRUE,
      '#description' => $this->t('Show a badge with your average rating in the site footer or other locations.'),
    ];

    $form['reviews']['display']['badge_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Badge Position'),
      '#options' => [
        'footer' => $this->t('Footer'),
        'sidebar' => $this->t('Sidebar'),
        'header' => $this->t('Header'),
        'custom' => $this->t('Custom (manual placement)'),
      ],
      '#default_value' => $config->get('badge_position') ?? 'footer',
      '#states' => [
        'visible' => [
          ':input[name="show_badge"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['reviews']['display']['show_all_platforms'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show all review platforms'),
      '#default_value' => $config->get('show_all_platforms') ?? FALSE,
      '#description' => $this->t('Display links to all enabled review platforms, not just the primary one.'),
    ];

    // 11. TAB: Advanced Settings.
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#group' => 'business_tabs',
    ];

    // Maintenance mode from Drupal core - SYNCHRONIZED.
    $maintenance_mode = Settings::get('maintenance_mode', FALSE);
    $form['advanced']['maintenance_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Maintenance Mode') . self::SYNC_MARKER,
      '#default_value' => $maintenance_mode,
      '#description' => $this->t('When enabled, only users with the "Access site in maintenance mode" permission can access the site.'),
      '#disabled' => TRUE,
      '#attributes' => [
        'class' => ['field-synchronized'],
      ],
    ];

    // Non-synchronized field.
    $form['advanced']['maintenance_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Maintenance Message'),
      '#default_value' => $config->get('maintenance_message') ??
        $this->t('Website under maintenance. We\'ll be back online shortly.'),
      '#rows' => 3,
      '#description' => $this->t('Custom message to display when maintenance mode is active.'),
    ];

    // Link to maintenance mode settings.
    $form['advanced']['maintenance_link'] = [
      '#type' => 'link',
      '#title' => $this->t('Edit maintenance mode →'),
      '#url' => Url::fromRoute('system.site_maintenance_mode'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
      ],
      '#prefix' => '<div class="sync-action-link">',
      '#suffix' => '</div>',
    ];



// 12. TAB: Local Laws (Dynamic from submodules)
$form['local_laws'] = [
  '#type' => 'details',
  '#title' => $this->t('Local Laws & Regulations'),
  '#group' => 'business_tabs',
  '#description' => $this->t('Configure local legal requirements for different nations/regions. Fields are loaded from installed submodules.'),
  '#weight' => 60,
];

// Container for region/nation selection
$form['local_laws']['region_selector'] = [
  '#type' => 'fieldset',
  '#title' => $this->t('Select Region/Nation'),
  '#description' => $this->t('Choose a region to configure its specific legal requirements.'),
];

// Get available local law modules
$local_law_modules = $this->getAvailableLocalLawModules();

if (!empty($local_law_modules)) {
  $region_options = [];
  foreach ($local_law_modules as $module => $info) {
    $region_options[$module] = $info['name'];
  }

  $current_region = $config->get('current_local_law_region') ?? key($region_options);

  $form['local_laws']['region_selector']['current_region'] = [
    '#type' => 'select',
    '#title' => $this->t('Active Region'),
    '#options' => $region_options,
    '#default_value' => $current_region,
    '#description' => $this->t('Select the region you want to configure.'),
    '#ajax' => [
      'callback' => [$this, 'updateLocalLawsFields'],
      'wrapper' => 'local-laws-fields-wrapper',
      'progress' => [
        'type' => 'throbber',
        'message' => $this->t('Loading region configuration...'),
      ],
    ],
  ];

  // Container for dynamic fields
  $form['local_laws']['fields_container'] = [
    '#type' => 'container',
    '#attributes' => ['id' => 'local-laws-fields-wrapper'],
  ];

  // Load fields for current region
  $form = $this->loadLocalLawFields($form, $form_state, $current_region, $config);

} else {
  // No local law modules installed
  $form['local_laws']['no_modules'] = [
    '#type' => 'fieldset',
    '#title' => $this->t('No Local Law Modules Installed'),
    '#description' => $this->t('Install local law submodules to configure region-specific legal requirements.'),
  ];

  $form['local_laws']['no_modules']['message'] = [
    '#markup' => '<div class="messages messages--warning">' .
      $this->t('<strong>No local law modules found.</strong> To add region-specific legal configuration:') .
      '</div>',
  ];

  $form['local_laws']['no_modules']['instructions'] = [
    '#theme' => 'item_list',
    '#items' => [
      $this->t('Install modules like <code>business_identity_local_us</code>, <code>business_identity_local_eu</code>, etc.'),
      $this->t('Each module should implement <code>hook_business_identity_local_laws()</code>'),
      $this->t('Modules automatically appear in the region selector when installed'),
    ],
  ];

  $form['local_laws']['no_modules']['documentation'] = [
    '#markup' => '<p>' .
      $this->t('See <a href=":url" target="_blank">module development documentation</a> for creating local law modules.', [
        ':url' => 'https://www.drupal.org/docs/develop/creating-modules'
      ]) .
      '</p>',
  ];
}

// Module installation notice
$form['local_laws']['module_notice'] = [
  '#type' => 'fieldset',
  '#title' => $this->t('Module Installation'),
  '#collapsible' => TRUE,
  '#collapsed' => TRUE,
];

$form['local_laws']['module_notice']['info'] = [
  '#markup' => '<div class="messages messages--info">' .
    $this->t('Local law modules are submodules of Business Identity that add region-specific legal fields.') .
    '</div>',
];




    // 13. TAB: Tokens.
    $form['tokens'] = [
      '#type' => 'details',
      '#title' => $this->t('Available Tokens'),
      '#group' => 'business_tabs',
    ];

    $token_url = Url::fromRoute('help.page', ['name' => 'token']);
    $token_link = Link::fromTextAndUrl(
      $this->t('Token Documentation'),
      $token_url
    )->toString();

    $form['tokens']['description'] = [
      '#markup' => '<p>' . $this->t('Use these tokens in content, blocks, and views:') . '</p>',
    ];

    $tokens = [
      '[business:company_name]' => $this->t('Company name (from site information) 🔗'),
      '[business:slogan]' => $this->t('Slogan (from site information) 🔗'),
      '[business:email]' => $this->t('Email (from site information) 🔗'),
      '[business:support_email]' => $this->t('Customer support email'),
      '[business:support_phone]' => $this->t('Customer support phone'),
      '[business:address]' => $this->t('Business address'),
      '[business:address_street]' => $this->t('Street address'),
      '[business:address_city]' => $this->t('City'),
      '[business:address_state]' => $this->t('State/Province'),
      '[business:address_postal_code]' => $this->t('Postal/ZIP code'),
      '[business:address_country]' => $this->t('Country'),
      '[business:address_lat]' => $this->t('Latitude'),
      '[business:address_lng]' => $this->t('Longitude'),
      '[business:where_are_we]' => $this->t('Where are we? description'),
      '[business:phone]' => $this->t('Phone number'),
      '[business:tax_id]' => $this->t('Tax ID/VAT'),
      '[business:legal_name]' => $this->t('Legal name'),
      '[business:founding_year]' => $this->t('Year founded'),
      '[business:date_format]' => $this->t('Date format 🔗'),
      '[business:time_format]' => $this->t('Time format 🔗'),
      '[business:timezone]' => $this->t('Timezone 🔗'),
      // Ecommerce tokens
      '[business:buy_now_label]' => $this->t('Buy Now button label'),
      '[business:buy_now_url]' => $this->t('Buy Now button URL'),
      '[business:buy_now_icon]' => $this->t('Buy Now button icon class'),
      '[business:alt_cta_label]' => $this->t('Alternative CTA label'),
      '[business:alt_cta_url]' => $this->t('Alternative CTA URL'),
      // New review tokens
      '[business:rating]' => $this->t('Average rating'),
      '[business:rating_count]' => $this->t('Total number of reviews'),
      '[business:rating_stars]' => $this->t('Star rating (e.g., ★★★★☆)'),
      '[business:primary_review_url]' => $this->t('Primary review platform URL'),
      '[business:review_request_message]' => $this->t('Review request message'),
      '[business:office_hours]' => $this->t('Business hours information'),
    ];

    // Add review platform tokens
    foreach (self::REVIEW_PLATFORMS as $platform => $label) {
      $tokens["[business:review_{$platform}_url]"] = $this->t('@platform review URL', ['@platform' => $label]);
      $tokens["[business:review_{$platform}_rating]"] = $this->t('@platform rating', ['@platform' => $label]);
      $tokens["[business:review_{$platform}_count]"] = $this->t('@platform review count', ['@platform' => $label]);
    }

    // Add currency token based on Commerce availability.
    if ($commerce_store_exists) {
      $tokens['[business:currency]'] = $this->t('Primary currency (from Commerce Store) 💳');
    }
    else {
      $tokens['[business:currency]'] = $this->t('Default currency');
    }



    // Add local laws tokens
    $local_law_modules = $this->getAvailableLocalLawModules();
    foreach ($local_law_modules as $module => $info) {
      $region_name = $info['name'];
      $region_code = strtolower($info['code'] ?? str_replace('business_identity_local_', '', $module));

      // Invoke hook to get token info from module
      $module_handler = $this->moduleHandler;
      $token_info = $module_handler->invoke($module, 'business_identity_local_laws_tokens');

      if (is_array($token_info)) {
        foreach ($token_info as $token => $description) {
          $tokens["[business:local_laws_{$region_code}_{$token}]"] =
            $this->t('@description (@region)', [
              '@description' => $description,
              '@region' => $region_name
            ]);
        }
      } else {
        // Default tokens if module doesn't provide specific ones
        $tokens["[business:local_laws_{$region_code}_configured]"] =
          $this->t('Whether @region laws are configured', ['@region' => $region_name]);
      }
    }











    $form['tokens']['token_list'] = [
      '#theme' => 'item_list',
      '#items' => array_map(function ($token, $description) {
        return "{$token} - {$description}";
      }, array_keys($tokens), $tokens),
    ];

    $form['tokens']['token_link'] = [
      '#markup' => '<p>' . $token_link . '</p>',
    ];

    // Summary of synchronized fields.
    $form['sync_summary'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['sync-summary'],
      ],
      '#weight' => 90,
    ];

    $form['sync_summary']['title'] = [
      '#markup' => '<h3>' . $this->t('Synchronization Information') . '</h3>',
    ];

    $synchronized_fields = [
      'Company Name' => '/admin/config/system/site-information',
      'Slogan/Motto' => '/admin/config/system/site-information',
      'Email Address' => '/admin/config/system/site-information',
      'Company Logo' => '/admin/appearance/settings',
      'Front Page' => '/admin/config/system/site-information',
      'Date Format' => '/admin/config/regional/date-time',
      'Time Format' => '/admin/config/regional/date-time',
      'Timezone' => '/admin/config/regional/settings',
    ];

    $sync_items = [];
    foreach ($synchronized_fields as $field => $path) {
      $sync_items[] = Link::fromTextAndUrl(
        $field,
        Url::fromUri('internal:' . $path)
      )->toString();
    }

    $form['sync_summary']['fields'] = [
      '#theme' => 'item_list',
      '#items' => $sync_items,
      '#title' => $this->t('Fields synchronized with Drupal core:'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get timezone options for the select field.
   *
   * @return array
   *   Timezone options array.
   */
  protected function getTimezoneOptions(): array {
    $timezones = \DateTimeZone::listIdentifiers();
    $options = [];

    foreach ($timezones as $timezone) {
      $now = new \DateTime('now', new \DateTimeZone($timezone));
      $offset = $now->getOffset();
      $offset_hours = $offset / 3600;
      $offset_formatted = ($offset_hours >= 0 ? '+' : '') . $offset_hours;

      $options[$timezone] = $this->t(
        '@timezone (UTC@offset)',
        [
          '@timezone' => str_replace('_', ' ', $timezone),
          '@offset' => $offset_formatted,
        ]
      );
    }

    return $options;
  }

  /**
   * AJAX callback for geolocating business address.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return array
   *   Geolocation form element.
   */
  public function geolocateBusinessAddress(array &$form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    $address_module_exists = $this->moduleHandler->moduleExists('address');

    // Build address string for geocoding.
    $address_parts = [];

    if ($address_module_exists) {
      // CORRECTED PATH: $values['address']['business_address']['address']
      $address_values = $form_state->getValue(['address', 'business_address', 'address']) ?? [];

      if (!empty($address_values['address_line1'])) {
        $address_parts[] = $address_values['address_line1'];
      }
      if (!empty($address_values['locality'])) {
        $address_parts[] = $address_values['locality'];
      }
      if (!empty($address_values['administrative_area'])) {
        $address_parts[] = $address_values['administrative_area'];
      }
      if (!empty($address_values['postal_code'])) {
        $address_parts[] = $address_values['postal_code'];
      }
      if (!empty($address_values['country_code'])) {
        $address_parts[] = $address_values['country_code'];
      }
    }
    else {
      // CORRECTED PATH: $values['address']['simple_address']
      $simple_address = $form_state->getValue(['address', 'simple_address']) ?? [];

      $street = $simple_address['street'] ?? '';
      $city = $simple_address['city'] ?? '';
      $state = $simple_address['state'] ?? '';
      $postal_code = $simple_address['postal_code'] ?? '';
      $country = $simple_address['country'] ?? '';

      if (!empty($street)) {
        $address_parts[] = $street;
      }
      if (!empty($city)) {
        $address_parts[] = $city;
      }
      if (!empty($state)) {
        $address_parts[] = $state;
      }
      if (!empty($postal_code)) {
        $address_parts[] = $postal_code;
      }
      if (!empty($country)) {
        $address_parts[] = $country;
      }
    }

    $full_address = implode(', ', $address_parts);

    if (empty($full_address)) {
      $this->messenger()->addError(
        $this->t('Please enter a complete address before geolocating.')
      );
      return $form['address']['geolocation'];
    }

    // Example coordinates for demonstration.
    $example_coordinates = [
      'New York' => ['lat' => '40.7128', 'lng' => '-74.0060'],
      'London' => ['lat' => '51.5074', 'lng' => '-0.1278'],
      'Rome' => ['lat' => '41.9028', 'lng' => '12.4964'],
      'Paris' => ['lat' => '48.8566', 'lng' => '2.3522'],
      'Berlin' => ['lat' => '52.5200', 'lng' => '13.4050'],
      'Tokyo' => ['lat' => '35.6762', 'lng' => '139.6503'],
      'Milan' => ['lat' => '45.4642', 'lng' => '9.1900'],
      'Madrid' => ['lat' => '40.4168', 'lng' => '-3.7038'],
      'Sydney' => ['lat' => '-33.8688', 'lng' => '151.2093'],
      'Toronto' => ['lat' => '43.6532', 'lng' => '-79.3832'],
    ];

    // Try to match with example cities.
    $lat = '';
    $lng = '';
    foreach ($example_coordinates as $city => $coords) {
      if (stripos($full_address, $city) !== FALSE) {
        $lat = $coords['lat'];
        $lng = $coords['lng'];
        break;
      }
    }

    // If no match, use default coordinates.
    if (empty($lat)) {
      $lat = '40.7128';
      $lng = '-74.0060';
    }

    // Update the form fields with geocoded coordinates.
    // Set the value directly in the form state user input.
    $user_input = $form_state->getUserInput();
    $user_input['address']['geolocation']['lat'] = $lat;
    $user_input['address']['geolocation']['lng'] = $lng;
    $form_state->setUserInput($user_input);

    // Also set the values in the form array.
    $form['address']['geolocation']['lat']['#value'] = $lat;
    $form['address']['geolocation']['lng']['#value'] = $lng;

    $this->messenger()->addStatus($this->t('Geolocated address: @address', [
      '@address' => $full_address,
    ]));

    return $form['address']['geolocation'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $values = $form_state->getValues();

    // Validate email (even though it's synchronized, we still validate).
    if (!empty($values['email']) &&
      !filter_var($values['email'], FILTER_VALIDATE_EMAIL)) {
      $form_state->setErrorByName('email', $this->t('The email address is not valid.'));
    }

    // Validate support email.
    if (!empty($values['support_email']) &&
      !filter_var($values['support_email'], FILTER_VALIDATE_EMAIL)) {
      $form_state->setErrorByName(
        'support_email',
        $this->t('The customer support email address is not valid.')
      );
    }

    // Validate social media URLs.
    foreach (self::SOCIAL_PLATFORMS as $platform => $label) {
      $url = $values['social_links'][$platform] ?? '';

      if (!empty($url) && !filter_var($url, FILTER_VALIDATE_URL)) {
        $form_state->setErrorByName(
          "social_links][{$platform}",
          $this->t('The @platform URL is not valid.', ['@platform' => $label])
        );
      }
    }

    // Validate ecommerce URLs.
    if (!empty($values['buy_now_enabled']) && !empty($values['buy_now_url'])) {
      if (!filter_var($values['buy_now_url'], FILTER_VALIDATE_URL)) {
        $form_state->setErrorByName(
          'buy_now_url',
          $this->t('The "Buy Now" URL is not valid.')
        );
      }
    }

    if (!empty($values['alt_cta_enabled']) && !empty($values['alt_cta_url'])) {
      if (!filter_var($values['alt_cta_url'], FILTER_VALIDATE_URL)) {
        $form_state->setErrorByName(
          'alt_cta_url',
          $this->t('The alternative CTA URL is not valid.')
        );
      }
    }

    // Validate review URLs.
    if (!empty($values['reviews_enabled'])) {
      foreach (self::REVIEW_PLATFORMS as $platform => $label) {
        $platform_data = $values['platforms'][$platform] ?? [];
        if (!empty($platform_data['enabled']) && !empty($platform_data['url'])) {
          if (!filter_var($platform_data['url'], FILTER_VALIDATE_URL)) {
            $form_state->setErrorByName(
              "platforms][{$platform}][url",
              $this->t('The @platform URL is not valid.', ['@platform' => $label])
            );
          }
        }
      }
    }

    // Validate year founded.
    if (!empty($values['founding_year'])) {
      $current_year = (int) date('Y');
      if ($values['founding_year'] < 1800 ||
        $values['founding_year'] > $current_year) {
        $form_state->setErrorByName(
          'founding_year',
          $this->t(
            'Year founded must be between 1800 and @current_year.',
            ['@current_year' => $current_year]
          )
        );
      }
    }

    // Validate latitude/longitude coordinates.
    // CORRECT PATH: $values['address']['geolocation']
    $geolocation = $values['address']['geolocation'] ?? [];
    $lat = $geolocation['lat'] ?? '';
    $lng = $geolocation['lng'] ?? '';

    if (!empty($lat)) {
      if (!is_numeric($lat) || $lat < -90 || $lat > 90) {
        $form_state->setErrorByName(
          'address][geolocation][lat',
          $this->t('Latitude must be a number between -90 and 90.')
        );
      }
    }

    if (!empty($lng)) {
      if (!is_numeric($lng) || $lng < -180 || $lng > 180) {
        $form_state->setErrorByName(
          'address][geolocation][lng',
          $this->t('Longitude must be a number between -180 and 180.')
        );
      }
    }

    // Validate rating values.
    if (!empty($values['reviews_enabled'])) {
      $rating_value = $values['rating_value'] ?? 0;
      if ($rating_value < 0 || $rating_value > 5) {
        $form_state->setErrorByName(
          'rating_value',
          $this->t('Average rating must be between 0 and 5.')
        );
      }

      $rating_count = $values['rating_count'] ?? 0;
      if ($rating_count < 0) {
        $form_state->setErrorByName(
          'rating_count',
          $this->t('Total reviews cannot be negative.')
        );
      }
    }


    // Validate local laws if any modules are installed
    if (!empty($values['local_laws'])) {
      $local_law_modules = $this->getAvailableLocalLawModules();

      foreach ($values['local_laws'] as $region => $region_values) {
        if (isset($local_law_modules[$region])) {
          // Invoke module-specific validation if available
          $validation_errors = $this->moduleHandler->invoke($region, 'business_identity_local_laws_validate', [
            $region_values,
            $form_state,
          ]);

          if (is_array($validation_errors)) {
            foreach ($validation_errors as $field_name => $error_message) {
              $form_state->setErrorByName(
                "local_laws][{$region}][{$field_name}",
                $error_message
              );
            }
          }
        }
      }
    }







  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('business_identity.settings');
    $values = $form_state->getValues();

    // Debug: Check the structure of $values
    \Drupal::logger('business_identity')->debug('Form values structure: @values', [
      '@values' => print_r($values, TRUE)
    ]);

    // Save business identity specific values.
    $config
      ->set('description', $values['description'])
      ->set('phone', $values['phone'])
      ->set('support_email', $values['support_email'])
      ->set('support_phone', $values['support_phone'])
      ->set('legal_name', $values['legal_name'])
      ->set('tax_id', $values['tax_id'])
      ->set('where_are_we', $values['where_are_we']) // AGGIUNTA: Salva il nuovo campo
      ->set('social_links', $this->processSocialLinks($values))
      ->set('business_sector', $values['business_sector'])
      ->set('founding_year', $values['founding_year'])
      ->set('maintenance_message', $values['maintenance_message']);

    // Save ecommerce settings.
    $config
      ->set('buy_now_enabled', $values['buy_now_enabled'] ?? FALSE)
      ->set('buy_now_label', $values['buy_now_label'] ?? $this->t('Buy Now'))
      ->set('buy_now_url', $values['buy_now_url'] ?? '')
      ->set('buy_now_new_window', $values['buy_now_new_window'] ?? FALSE)
      ->set('buy_now_style', $values['buy_now_style'] ?? 'primary')
      ->set('buy_now_custom_class', $values['buy_now_custom_class'] ?? '')
      ->set('buy_now_icon', $values['buy_now_icon'] ?? '')
      ->set('buy_now_header', $values['buy_now_header'] ?? TRUE)
      ->set('buy_now_footer', $values['buy_now_footer'] ?? TRUE)
      ->set('buy_now_sidebar', $values['buy_now_sidebar'] ?? FALSE)
      ->set('buy_now_product_pages', $values['buy_now_product_pages'] ?? TRUE)
      ->set('alt_cta_enabled', $values['alt_cta_enabled'] ?? FALSE)
      ->set('alt_cta_label', $values['alt_cta_label'] ?? $this->t('Learn More'))
      ->set('alt_cta_url', $values['alt_cta_url'] ?? '');

    // Save review settings.
    $config
      ->set('reviews_enabled', $values['reviews_enabled'] ?? FALSE)
      ->set('rating_value', $values['rating_value'] ?? 0)
      ->set('rating_count', $values['rating_count'] ?? 0)
      ->set('rating_last_updated', $values['last_updated'] ?? '')
      ->set('review_platforms', $this->processReviewPlatforms($values['platforms'] ?? []))
      ->set('review_request_message', $values['review_request_message'] ?? '')
      ->set('primary_review_platform', $values['primary_platform'] ?? 'google')
      ->set('review_email_subject', $values['email_subject'] ?? '')
      ->set('review_email_body', $values['email_body'] ?? '')
      ->set('show_rating_badge', $values['show_badge'] ?? TRUE)
      ->set('badge_position', $values['badge_position'] ?? 'footer')
      ->set('show_all_platforms', $values['show_all_platforms'] ?? FALSE);

    // Check if the Office Hours module is available.
    $office_hours_module_exists = $this->moduleHandler->moduleExists('office_hours');
    // Save office hours configuration
    if ($office_hours_module_exists) {
      $config->set('office_hours', [
        'timezone' => $values['timezone'] ?? '',
        'hours_format' => $values['hours_format'] ?? 'G:i',
      ]);
    } else {
      $config->set('business_hours_text', $values['business_hours_text'] ?? '');
    }

    // FIXED: Save address information - Use the correct path from $values
    $address_module_exists = $this->moduleHandler->moduleExists('address');

    if ($address_module_exists) {
      $business_address = $values['business_address']['address'] ?? [];

      \Drupal::logger('business_identity')->debug('Structured address to save: @data', [
        '@data' => print_r($business_address, TRUE)
      ]);

      $config->set('business_address', $business_address);
    }
    else {
      // CORRECT PATH: $values['address']['simple_address'].
      $simple_address = $values['address']['simple_address'] ?? [];

      \Drupal::logger('business_identity')->debug('Simple address to save: @data', [
        '@data' => print_r($simple_address, TRUE)
      ]);

      if (is_array($simple_address)) {
        $config
          ->set('address_street', $simple_address['street'] ?? '')
          ->set('address_city', $simple_address['city'] ?? '')
          ->set('address_state', $simple_address['state'] ?? '')
          ->set('address_postal_code', $simple_address['postal_code'] ?? '')
          ->set('address_country', $simple_address['country'] ?? '');
      }
    }

    // Save geolocation coordinates.
    // CORRECT PATH: $values['address']['geolocation']
    $geolocation = $values['address']['geolocation'] ?? [];

    \Drupal::logger('business_identity')->debug('Geolocation to save: @data', [
      '@data' => print_r($geolocation, TRUE)
    ]);

    if (is_array($geolocation)) {
      $config
        ->set('address_lat', $geolocation['lat'] ?? '')
        ->set('address_lng', $geolocation['lng'] ?? '');
    }

    // Save currency only if Commerce Store is NOT available.
    $commerce_store_exists = $this->moduleHandler
      ->moduleExists('commerce_store');
    if (!$commerce_store_exists) {
      $config->set('currency', $values['currency'] ?? 'USD');
    }

    // Handle API keys based on Key module availability.
    $key_module_exists = $this->moduleHandler->moduleExists('key');
    if ($key_module_exists && $this->keyRepository) {
      $config
        ->set('google_maps_key', $values['google_maps_key'])
        ->set('recaptcha_site_key', $values['recaptcha_site_key'])
        ->set('recaptcha_secret_key', $values['recaptcha_secret_key']);
    }
    else {
      $config
        ->set('google_maps_key', $values['google_maps_key_text'] ?? NULL)
        ->set('recaptcha_site_key', $values['recaptcha_site_key_text'] ?? NULL)
        ->set('recaptcha_secret_key', $values['recaptcha_secret_key_text'] ?? NULL);
    }

    // Save the configuration
    $config->save();

    // Debug: Check what was actually saved.
    $saved_config = $this->config('business_identity.settings')->get();
    \Drupal::logger('business_identity')->debug('Saved configuration: @config', [
      '@config' => print_r($saved_config, TRUE)
    ]);

    parent::submitForm($form, $form_state);

    // Clear cache to ensure tokens are updated.
    \Drupal::service('cache.render')->invalidateAll();

    $this->messenger()->addStatus(
      $this->t('Business identity configuration saved. Note: Fields marked with 🔗 are synchronized with Drupal core and were not modified.')
    );

    // Show specific message about Commerce integration if available.
    if ($commerce_store_exists) {
      $this->messenger()->addStatus(
        $this->t('Currency is managed by Commerce Store module.')
      );
    }

    // Show message about ecommerce settings.
    if ($values['buy_now_enabled'] ?? FALSE) {
      $this->messenger()->addStatus(
        $this->t('"Buy Now" button has been configured and will appear in the specified locations.')
      );
    }

    // Show message about review settings.
    if ($values['reviews_enabled'] ?? FALSE) {
      $this->messenger()->addStatus(
        $this->t('Review settings have been saved. Remember to update your ratings regularly.')
      );
    }

    $this->messenger()->addWarning(
      $this->t('To modify synchronized fields (marked with 🔗), use the linked configuration pages in Drupal core.')
    );


    // Update the submitForm method to save local laws configuration:

    // Add this in the submitForm method after other configuration saves:

    // Save local laws configuration
    if (!empty($values['local_laws'])) {
      $local_laws_config = [];
      foreach ($values['local_laws'] as $region => $region_values) {
        $local_laws_config[$region] = $region_values;
      }
      $config->set('local_laws', $local_laws_config);
    }

    // Save current region
    if (!empty($values['current_region'])) {
      $config->set('current_local_law_region', $values['current_region']);
    }

  }

  /**
   * Process social links from form values.
   *
   * @param array $values
   *   Form values.
   *
   * @return array
   *   Processed social links.
   */
  protected function processSocialLinks(array $values): array {
    $social_links = [];
    foreach (self::SOCIAL_PLATFORMS as $platform => $label) {
      if (!empty($values['social_links'][$platform])) {
        $social_links[$platform] = $values['social_links'][$platform];
      }
    }
    return $social_links;
  }

  /**
   * Process review platforms from form values.
   *
   * @param array $platforms_data
   *   Review platforms data from form.
   *
   * @return array
   *   Processed review platforms.
   */
  protected function processReviewPlatforms(array $platforms_data): array {
    $processed = [];
    foreach (self::REVIEW_PLATFORMS as $platform => $label) {
      $platform_data = $platforms_data[$platform] ?? [];
      if (!empty($platform_data['enabled'])) {
        $processed[$platform] = [
          'enabled' => TRUE,
          'url' => $platform_data['url'] ?? '',
          'rating' => floatval($platform_data['rating'] ?? 0),
          'count' => intval($platform_data['count'] ?? 0),
        ];

        if ($platform === 'other' && !empty($platform_data['custom_name'])) {
          $processed[$platform]['custom_name'] = $platform_data['custom_name'];
        }
      }
    }
    return $processed;
  }



  /**
  * Get available local law modules.
  *
  * @return array
  *   Array of available local law modules with their information.
  */

  /**
 * Get available local law modules.
 *
 * @return array
 *   Array of available local law modules with their information.
 */








/**
 * Get available local law modules.
 *
 * @return array
 *   Array of available local law modules with their information.
 */
// protected function getAvailableLocalLawModules(): array {
//   $modules = [];
//
//   // Get extension list module service
//   $extension_list = \Drupal::service('extension.list.module');
//
//   // Get all installed modules info
//   $installed_modules = $extension_list->getAllInstalledInfo();
//
//   foreach ($installed_modules as $module_name => $module_info) {
//     // Check for our naming pattern
//     if (strpos($module_name, 'business_identity_local_') === 0) {
//       $region_code = strtoupper(str_replace('business_identity_local_', '', $module_name));
//
//       // Use module name from .info.yml or generate from machine name
//       $display_name = $module_info['name'] ?? ucwords(str_replace('_', ' ', $region_code));
//
//       $modules[$module_name] = [
//         'name' => $display_name,
//         'code' => $region_code,
//         'description' => $module_info['description'] ?? $this->t('Local laws for %region', [
//           '%region' => $display_name
//         ]),
//         'version' => $module_info['version'] ?? '1.0.0',
//       ];
//
//       // Try to get hook info for additional data
//       try {
//         $hook_info = $this->moduleHandler->invoke($module_name, 'business_identity_local_laws');
//         if (is_array($hook_info) && !empty($hook_info['name'])) {
//           // Merge hook info
//           $modules[$module_name] = array_merge($modules[$module_name], $hook_info);
//         }
//       } catch (\Exception $e) {
//         // Hook doesn't exist, use .info.yml data only
//       }
//     }
//   }
//
//   // Sort modules alphabetically by name
//   uasort($modules, function($a, $b) {
//     return strcmp($a['name'], $b['name']);
//   });
//
//   return $modules;
// }


/**
 * Get available local law modules.
 *
 * @return array
 *   Array of available local law modules with their information.
 */
protected function getAvailableLocalLawModules(): array {
  $modules = [];

  // Get extension list module service
  $extension_list = \Drupal::service('extension.list.module');

  // Get all installed modules info
  $installed_modules = $extension_list->getAllInstalledInfo();

  foreach ($installed_modules as $module_name => $module_info) {
    // Check for our naming pattern
    if (strpos($module_name, 'business_identity_local_') === 0) {
      $region_code = strtoupper(str_replace('business_identity_local_', '', $module_name));

      // Use module name from .info.yml or generate from machine name
      $display_name = $module_info['name'] ?? ucwords(str_replace('_', ' ', $region_code));

      $modules[$module_name] = [
        'name' => $display_name,
        'code' => $region_code,
        'description' => $module_info['description'] ?? $this->t('Local laws for %region', [
          '%region' => $display_name
        ]),
        'version' => $module_info['version'] ?? '1.0.0',
      ];

      // Try to get hook info for additional data
      try {
        $hook_info = $this->moduleHandler->invoke($module_name, 'business_identity_local_laws');
        if (is_array($hook_info) && !empty($hook_info['name'])) {
          // Merge hook info
          $modules[$module_name] = array_merge($modules[$module_name], $hook_info);
        }
      } catch (\Exception $e) {
        // Hook doesn't exist, use .info.yml data only
      }
    }
  }

  // Sort modules alphabetically by name - convert to string first
  uasort($modules, function($a, $b) {
    // Convert TranslatableMarkup to string for comparison
    $name_a = (string) ($a['name'] ?? '');
    $name_b = (string) ($b['name'] ?? '');
    return strcmp($name_a, $name_b);
  });

  return $modules;
}



  /**
  * Load local law fields for a specific region.
  *
  * @param array $form
  *   Form array.
  * @param \Drupal\Core\Form\FormStateInterface $form_state
  *   Form state.
  * @param string $region
  *   Region module name.
  * @param \Drupal\Core\Config\Config $config
  *   Configuration object.
  *
  * @return array
  *   Updated form array.
  */
  protected function loadLocalLawFields(array $form, FormStateInterface $form_state, string $region, \Drupal\Core\Config\Config $config): array {
    $module_handler = $this->moduleHandler;

    // Invoke hook to get fields from module
    $fields = $module_handler->invoke($region, 'business_identity_local_laws_fields');

    if (empty($fields) || !is_array($fields)) {
      $form['local_laws']['fields_container']['no_fields'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('No Configuration Required'),
        '#description' => $this->t('This region does not require any specific legal configuration.'),
      ];
      return $form;
    }

    // Get module info for display
    $module_info = $module_handler->invoke($region, 'business_identity_local_laws');
    $region_name = $module_info['name'] ?? ucfirst(str_replace('business_identity_local_', '', $region));
    $region_code = $module_info['code'] ?? strtoupper($region);

    $form['local_laws']['fields_container']['region_header'] = [
      '#markup' => '<h3>' . $this->t('%region Legal Requirements', ['%region' => $region_name]) .
        ' <small>(' . $region_code . ')</small></h3>',
    ];

    if (isset($module_info['description'])) {
      $form['local_laws']['fields_container']['region_description'] = [
        '#markup' => '<div class="description">' . $module_info['description'] . '</div>',
      ];
    }

    // Load existing configuration for this region
    $local_laws_config = $config->get('local_laws') ?? [];
    $region_config = $local_laws_config[$region] ?? [];

    // Add fields from module
    foreach ($fields as $field_name => $field_config) {
      $field_key = "local_laws_{$region}_{$field_name}";

      // Set default value from config
      $default_value = $region_config[$field_name] ?? $field_config['#default_value'] ?? NULL;

      $field_config['#default_value'] = $default_value;
      $field_config['#name'] = "local_laws[$region][$field_name]";

      $form['local_laws']['fields_container'][$field_key] = $field_config;
    }

    // Add disclaimer/note
    $form['local_laws']['fields_container']['disclaimer'] = [
      '#markup' => '<div class="messages messages--warning">' .
        $this->t('<strong>Legal Disclaimer:</strong> This configuration is for informational purposes only. Consult with a legal professional for compliance with local laws in %region.',
          ['%region' => $region_name]) .
        '</div>',
      '#weight' => 100,
    ];

    return $form;
  }

  /**
  * AJAX callback to update local laws fields.
  *
  * @param array $form
  *   Form array.
  * @param \Drupal\Core\Form\FormStateInterface $form_state
  *   Form state.
  *
  * @return array
  *   The fields container element.
  */
  public function updateLocalLawsFields(array &$form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    $region = $triggering_element['#value'] ?? key($this->getAvailableLocalLawModules());

    // Reload configuration
    $config = $this->config('business_identity.settings');

    // Rebuild the fields container
    $form['local_laws']['fields_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'local-laws-fields-wrapper'],
    ];

    $form = $this->loadLocalLawFields($form, $form_state, $region, $config);

    return $form['local_laws']['fields_container'];
  }



  // Add this helper method to process local laws form values:

  /**
  * Process local laws form values for saving.
  *
  * @param array $values
  *   Form values.
  *
  * @return array
  *   Processed local laws configuration.
  */
  protected function processLocalLaws(array $values): array {
    $local_laws = [];

    if (empty($values['local_laws'])) {
      return $local_laws;
    }

    foreach ($values['local_laws'] as $region => $region_values) {
      // Clean up empty values
      $region_values = array_filter($region_values, function($value) {
        return $value !== '' && $value !== NULL;
      });

      if (!empty($region_values)) {
        $local_laws[$region] = $region_values;
      }
    }

    return $local_laws;
  }


}


