<?php

declare(strict_types=1);

namespace Drupal\business_identity\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Business Identity' block.
 *
 * @Block(
 *   id = "business_identity_block",
 *   admin_label = @Translation("Business Identity"),
 *   category = @Translation("Business")
 * )
 */
class BusinessIdentityBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new BusinessIdentityBlock.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $config = $this->configFactory->get('business_identity.settings');
    $opening_hours_config = $this->configFactory->get('business_identity.opening_hours');

    $build = [
      '#theme' => 'block__business_identity',
      '#attributes' => [
        'class' => ['business-identity-block'],
      ],
      '#cache' => [
        'tags' => $config->getCacheTags(),
        'contexts' => ['url'],
      ],
    ];

    // Business information.
    $build['#business'] = [
      'legal_name' => $config->get('legal_name'),
      'commercial_name' => $config->get('commercial_name'),
      'vat_number' => $config->get('vat_number'),
      'address' => $config->get('address'),
      'phone' => $config->get('phone'),
      'email' => $config->get('email'),
      'website' => $config->get('website'),
    ];

    // Social links.
    $social_links = $config->get('social_links') ?? [];
    $build['#social_links'] = [];
    foreach ($social_links as $link) {
      if (!empty($link['platform']) && !empty($link['url'])) {
        $build['#social_links'][] = [
          'platform' => $link['platform'],
          'url' => $link['url'],
          'icon_class' => 'bi-' . $link['platform'],
        ];
      }
    }

    // Opening hours.
    $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
    $build['#opening_hours'] = [];
    foreach ($days as $day) {
      if ($opening_hours_config->get($day . '_open')) {
        $build['#opening_hours'][$day] = [
          'open' => TRUE,
          'start' => $opening_hours_config->get($day . '_start') ?? '09:00',
          'end' => $opening_hours_config->get($day . '_end') ?? '17:00',
          'breaks' => $opening_hours_config->get($day . '_breaks'),
        ];
      }
      else {
        $build['#opening_hours'][$day] = ['open' => FALSE];
      }
    }

    // Add CSS library.
    $build['#attached']['library'][] = 'business_identity/block';

    return $build;
  }

}
