<?php

namespace Drupal\Tests\cache_register\Kernel;

use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Tests the SlotBase class.
 *
 * @coversDefaultClass \Drupal\cache_register\Object\SlotBase
 * @group cache_register
 */
class SlotBaseTest extends CacheRegisterKernelTestBase {

  /**
   * Ensure slot cache persists reopening.
   */
  public function testReopenSlot() {
    /** @var \Drupal\cache_register\Object\SlotBase $slot */
    $slot_1 = $this->createSlotBase('testReopenSlot');
    $cache_data = 'cached_data';
    $this->callMethod(
      $slot_1,
      'doSetCache',
      [$cache_data, -1, []]
    );
    $slot_cache_1 = $slot_1->getCache();

    // Reopen the slot.
    $slot_2 = $this->createSlotBase('testReopenSlot');
    $slot_cache_2 = $slot_2->getCache();
    $this->assertEqual($slot_cache_1, $slot_cache_2);
  }

  /**
   * @covers ::getDrawer
   */
  public function testGetDrawer() {
    $drawer = $this->createDrawer();
    $slot = $this->createSlotBase('slot', $drawer);
    $this->assertEquals($drawer, $slot->getDrawer());
  }

  /**
   * @covers ::getRegister
   * @dataProvider boolProvider
   */
  public function testGetRegister($open_register) {
    $drawer = $this->createDrawer('d', $open_register);
    $slot = $this->createSlotBase('slot', $drawer);

    if (!$open_register) {
      $this->assertEqual(NULL, $slot->getRegister());
    }
    else {
      $this->assertInstanceOf('Drupal\cache_register\Object\Register', $slot->getRegister());
    }
  }

  /**
   * @covers ::getCacheBackend
   */
  public function testGetCacheBackend() {
    $slot = $this->createSlotBase('testGetCacheBackend');
    $this->assertEquals($slot->getDrawer()->getCacheBackend(), $slot->getCacheBackend());
    $this->assertInstanceOf('Drupal\Core\Cache\CacheBackendInterface', $slot->getCacheBackend());
  }

  /**
   * @covers ::doSetCache
   * @dataProvider cacheDataProvider
   *
   * @param int $data
   *   Data value.
   * @param int $expire
   *   Make sure this is a valid timestamp or errors will throw.
   *   Cannot rely on strtotime or associated approaches.
   * @param array $tags
   *   Array value.
   */
  public function testDoSetCache($data, int $expire = CacheBackendInterface::CACHE_PERMANENT, array $tags = []) {
    $slot = $this->createSlotBase('testIsCached');
    $this->commonTestSetCache($slot, $data, $expire, $tags, 'doSetCache');
  }

  /**
   * @covers ::isCached
   */
  public function testIsCached() {
    // Before caching.
    $slot = $this->createSlotBase('testIsCached');
    $this->assertEquals(FALSE, $slot->isCached());

    // After caching.
    $slot->getCacheBackend()->set($slot->id(), 'cache_data');
    $this->assertEquals(TRUE, $slot->isCached());

    // After invalidating the cache.
    $slot->getCacheBackend()->invalidate($slot->id());
    $this->assertEquals(FALSE, $slot->isCached());

    // Repopulate and then delete the cache.
    $slot->getCacheBackend()->set($slot->id(), 'cache_data');
    $slot->getCacheBackend()->delete($slot->id());
    $this->assertEquals(FALSE, $slot->isCached());
  }

  /**
   * @covers ::getCache
   */
  public function testGetCache() {
    // Before caching.
    $slot = $this->createSlotBase('testGetCache');
    $this->assertEquals(NULL, $slot->getCache());
    $this->assertEquals($slot->getCacheBackend()->get($slot->id()), $slot->getCache());

    // After caching.
    $slot->getCacheBackend()->set($slot->id(), 'cache_data');
    $this->assertNotEqual($slot->getCache(), NULL);
    $this->assertEquals($slot->getCacheBackend()->get($slot->id()), $slot->getCache());

    // After invalidating the cache.
    $slot->getCacheBackend()->invalidate($slot->id());
    $this->assertEquals(NULL, $slot->getCache());
    $this->assertNotEqual($slot->getCache(TRUE), NULL);
    $this->assertEquals($slot->getCacheBackend()->get($slot->id()), $slot->getCache());

    // Repopulate and then delete the cache.
    $slot->getCacheBackend()->set($slot->id(), 'cache_data');
    $slot->getCacheBackend()->delete($slot->id());
    $this->assertEquals(NULL, $slot->getCache());
    $this->assertEquals($slot->getCacheBackend()->get($slot->id()), $slot->getCache());
  }

  /**
   * @covers ::doGetCacheData
   */
  public function testDoGetCacheData() {
    $slot = $this->createSlotBase('testIsCached');
    $cid = $slot->id();
    $this->assertEquals(NULL, $slot->getCache());

    // After caching.
    $slot->getCacheBackend()->set($cid, 'cache_data');
    $this->assertNotEqual(
      $this->callMethod($slot, 'doGetCacheData', [FALSE]),
      NULL
    );
    $this->assertEquals(
      $slot->getCache()->data,
      $this->callMethod($slot, 'doGetCacheData', [FALSE])
    );

    // After invalidating the cache.
    $slot->getCacheBackend()->invalidate($slot->id());
    $this->assertEquals(
      NULL,
      $this->callMethod($slot, 'doGetCacheData', [FALSE])
    );
    $this->assertNotEqual(
      $this->callMethod($slot, 'doGetCacheData', [TRUE]),
      NULL
    );
    $this->assertEquals(
      $slot->getCache(TRUE)->data,
      $this->callMethod($slot, 'doGetCacheData', [TRUE])
    );

    // Repopulate and then delete the cache.
    $slot->getCacheBackend()->set($cid, 'cache_data');
    $slot->getCacheBackend()->delete($cid);
    $this->assertEquals(
      NULL,
      $this->callMethod($slot, 'doGetCacheData', [FALSE])
    );
    $this->assertEquals(
      $slot->getCache(),
      $this->callMethod($slot, 'doGetCacheData', [FALSE])
    );
  }

  /**
   * Provides cache tags bool value.
   *
   * @return array
   *   Returns the array values.
   */
  public function cacheTagsProvider() {
    return [
      // Valid.
      ['tag:1', TRUE],
      [['tag:1', 'tag:2'], TRUE],
      // Invalid.
      [new \stdClass(), FALSE],
      [1, FALSE],
      [['tag:1', new \stdClass()], FALSE],
      [['tag:1', 1], FALSE],
    ];
  }

  /**
   * @covers ::addCacheTags
   * @dataProvider cacheTagsProvider
   */
  public function testAddCacheTags($tags, $is_valid) {
    if (!$is_valid) {
      $this->expectException('AssertionError');
    }

    $slot = $this->createSlotBase('testAddCacheTags');

    $this->expectException('Drupal\cache_register\Exception\CacheNotSetError');
    $slot->addCacheTags($tags);

    $slot->setCache('some_data');
    $slot->addCacheTags($tags);

    $cache_entry = $slot->getCache();
    $tags = is_string($tags) ? [$tags] : $tags;
    foreach ($tags as $tag) {
      $this->assertContains($tag, $cache_entry->tags);
    }
  }

  /**
   * Provides data to test id().
   *
   * Leave this and testId() down here at the bottom
   * so we don't need to scroll past all this junk
   * whenever we need to go through this class.
   *
   * @return array[]
   *   Returns the array values.
   */
  public function idProvider(): array {
    return [
      // No register.
      '1 id string, no register' => [
        self::MOCK_DRAWER_ID,
        'id_1',
        self::MOCK_DRAWER_ID . ':id_1',
      ],
      '1 id int, no register' => [
        self::MOCK_DRAWER_ID,
        1,
        self::MOCK_DRAWER_ID . ':1',
      ],
      '1 id array, no register' => [
        self::MOCK_DRAWER_ID,
        ['id_1'],
        self::MOCK_DRAWER_ID . ':id_1',
      ],
      '2 id array, no register' => [
        self::MOCK_DRAWER_ID,
        ['id_1', 'id_2'],
        self::MOCK_DRAWER_ID . ':id_1.id_2',
      ],
      '2 int id array, no register' => [
        self::MOCK_DRAWER_ID,
        [1, 2],
        self::MOCK_DRAWER_ID . ':1.2',
      ],
      '2 mixed id array, no register' => [
        self::MOCK_DRAWER_ID,
        [1, '2'],
        self::MOCK_DRAWER_ID . ':1.2',
      ],
      '3 id array, no register' => [
        self::MOCK_DRAWER_ID,
        ['id_1', 'id_2', 'id_3'],
        self::MOCK_DRAWER_ID . ':id_1.id_2.id_3',
      ],
      // Invalid.
      'object ID, no register' => [
        self::MOCK_DRAWER_ID,
        new \stdClass(),
        NULL,
      ],
      'bool ID, no register' => [
        self::MOCK_DRAWER_ID,
        FALSE,
        NULL,
      ],
      'object array ID, no register' => [
        self::MOCK_DRAWER_ID,
        [new \stdClass()],
        NULL,
      ],
      'mixed invalid array ID, no register' => [
        self::MOCK_DRAWER_ID,
        ['string', new \stdClass()],
        NULL,
      ],
    ];
  }

  /**
   * @covers ::id
   * @dataProvider idProvider
   */
  public function testId($drawer_id, $slot_ids, $expected_id) {
    $drawer = $this->createDrawer($drawer_id);

    if (is_null($expected_id)) {
      $this->expectException('TypeError');
    }

    $slot = $this->createSlotBase($slot_ids, $drawer);
    $this->assertEquals($expected_id, $slot->id());
  }

}
