<?php

namespace Drupal\Tests\cache_register\Kernel;

use Drupal\cache_register\Object\Drawer;
use Drupal\cache_register\Object\DrawerInterface;
use Drupal\cache_register\Object\Register;
use Drupal\cache_register\Object\RegisterInterface;
use Drupal\cache_register\Object\Slot;
use Drupal\cache_register\Object\SlotBaseInterface;
use Drupal\cache_register\Object\SlotInterface;
use Drupal\KernelTests\Core\Cache\DatabaseBackendTest;

/**
 * Defines a base class for cache_register test classes.
 *
 * @group cache_register
 */
abstract class CacheRegisterKernelTestBase extends DatabaseBackendTest {

  protected const MOCK_DRAWER_ID = 'cache_register.test_drawer';

  /**
   * The modules to load to run the test.
   *
   * @var array
   */
  public static $modules = [
    'cache_register',
  ];

  /**
   * Provides the Manager service.
   *
   * In some cases, tests that instantiate our objects directly
   * throw a Database Serialization error. In those cases,
   * the manager should be used instead.
   *
   * @returns \Drupal\cache_register\Manager
   */
  public function getManager() {
    return \Drupal::service('cache_register.manager');
  }

  /**
   * Creates a Test Drawer.
   *
   * @param string $drawer_id
   *   The Drawer ID.
   * @param bool $open_register
   *   Whether or not the open the register.
   *
   * @return \Drupal\cache_register\Object\DrawerInterface
   *   Returns the drawer object.
   */
  public function createDrawer($drawer_id = '', $open_register = FALSE): DrawerInterface {
    $drawer_id = $drawer_id ?: self::MOCK_DRAWER_ID;
    return new Drawer($this->getCacheBackend(), $drawer_id, $open_register);
  }

  /**
   * Creates a SlotBase for testing.
   *
   * @param string $slot_ids
   *   The slot IDs.
   * @param \Drupal\cache_register\Object\DrawerInterface|null $drawer
   *   The drawer.
   *
   * @return \Drupal\cache_register\Object\SlotBaseInterface|object
   *   Returns the slotbase object.
   */
  public function createSlotBase($slot_ids = 'sid', $drawer = NULL): SlotBaseInterface {
    return $this->getMockForAbstractClass(
      'Drupal\cache_register\Object\SlotBase',
      [
        $this->slotTestDrawerValidatorProvider($drawer),
        $slot_ids,
      ]
    );
  }

  /**
   * To help us test protected/private methods.
   *
   * @param int $obj
   *   The object instance.
   * @param int $method_name
   *   The Method name.
   * @param array $args
   *   The args to pass into the method.
   *
   * @return mixed
   *   Returns the mixed.
   *
   * @throws \ReflectionException
   */
  public static function callMethod($obj, $method_name, array $args = []) {
    $class = new \ReflectionClass($obj);
    $method = $class->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($obj, $args);
  }

  /**
   * Creates a Slot for testing.
   *
   * @param int $slot_ids
   *   The slot IDs.
   * @param \Drupal\cache_register\Object\DrawerInterface|null $drawer
   *   The drawer.
   *
   * @return \Drupal\cache_register\Object\SlotInterface
   *   Retirns the slot object.
   */
  public function createSlot($slot_ids, $drawer = NULL): SlotInterface {
    return new Slot($this->slotTestDrawerValidatorProvider($drawer), $slot_ids);
  }

  /**
   * Creates a Register for testing.
   *
   * @param \Drupal\cache_register\Object\DrawerInterface|null $drawer
   *   The drawer.
   *
   * @return \Drupal\cache_register\Object\RegisterInterface
   *   The register.
   */
  public function createRegister($drawer = NULL): RegisterInterface {
    return new Register($this->slotTestDrawerValidatorProvider($drawer));
  }

  /**
   * Provider for simple bool tests.
   */
  public function boolProvider() {
    return [
      [TRUE],
      [FALSE],
    ];
  }

  /**
   * Provides data to test setCache().
   *
   * @return array[]
   *   Returns the array values.
   */
  public function cacheDataProvider() {
    return [
      ['string data'],
      [['array', 'of', 'data']],
      ['string data', strtotime('+1 minute')],
      ['string data', strtotime('+1 month'), ['tag:1', 'tag:2']],
    ];
  }

  /**
   * Common method for testing setCacheData.
   */
  protected function commonTestSetCache($slot, $data, int $expire, array $tags, $method_name = 'setCache') {
    $cid = $slot->id();

    switch ($method_name) {
      case 'doSetCache':
        $this->callMethod(
          $slot,
          'doSetCache',
          [$data, $expire, $tags]
        );
        break;

      case 'setCache':
        $slot->setCache($data, $expire, $tags);
        break;
    }

    $cache_entry = $slot->getCacheBackend()->get($cid);
    $this->assertEquals($cache_entry->data, $data);
    $this->assertEquals($cache_entry->expire, $expire);
    foreach ($tags as $tag) {
      $this->assertContains($tag, $cache_entry->tags);
    }
  }

  /**
   * Common method for testing getCacheData.
   */
  protected function commonTestGetCacheData($slot) {
    $cid = $slot->id();
    $this->assertEquals(NULL, $slot->getCache());

    // After caching.
    $slot->getCacheBackend()->set($cid, 'cache_data');
    $this->assertNotEqual($slot->getCacheData(), NULL);
    $this->assertEquals(
      $slot->getCache()->data,
      $slot->getCacheData()
    );

    // After invalidating the cache.
    $slot->getCacheBackend()->invalidate($slot->id());
    $this->assertEquals(NULL, $slot->getCacheData());
    $this->assertNotEqual($slot->getCacheData(TRUE), NULL);
    $this->assertEquals(
      $slot->getCache(TRUE)->data,
      $slot->getCacheData(TRUE)
    );

    // Repopulate and then delete the cache.
    $slot->getCacheBackend()->set($cid, 'cache_data');
    $slot->getCacheBackend()->delete($cid);
    $this->assertEquals(NULL, $slot->getCache());
    $this->assertEquals(
      $slot->getCache(TRUE),
      $slot->getCacheData(TRUE)
    );
  }

  /**
   * Validates/provides a drawer for test slots.
   *
   * @param \Drupal\cache_register\Object\DrawerInterface|null $drawer
   *   The drawer, if present.
   *
   * @return \Drupal\cache_register\Object\DrawerInterface
   *   The drawer.
   */
  private function slotTestDrawerValidatorProvider(DrawerInterface $drawer = NULL): DrawerInterface {
    if (is_null($drawer)) {
      $drawer = $this->createDrawer();
    }
    elseif (!is_object($drawer)
      || (strpos(get_class($drawer), 'Drawer') === FALSE)) {
      throw new \TypeError('Drawer must be an object of type Drawer. ' . gettype($drawer) . ':' . get_class($drawer) . ' given.');
    }

    return $drawer;
  }

}
