<?php

declare(strict_types=1);

namespace Drupal\cacheviz;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Path\PathMatcherInterface;

/**
 * Service to check if the current path should be excluded from cacheviz.
 */
class PathMatcher {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The current path stack.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  protected CurrentPathStack $currentPath;

  /**
   * The core path matcher.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected PathMatcherInterface $pathMatcher;

  /**
   * Constructs a PathMatcher object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Path\CurrentPathStack $current_path
   *   The current path stack.
   * @param \Drupal\Core\Path\PathMatcherInterface $path_matcher
   *   The core path matcher.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    CurrentPathStack $current_path,
    PathMatcherInterface $path_matcher,
  ) {
    $this->configFactory = $config_factory;
    $this->currentPath = $current_path;
    $this->pathMatcher = $path_matcher;
  }

  /**
   * Checks if the current path is excluded from visualization.
   *
   * @return bool
   *   TRUE if the current path is excluded, FALSE otherwise.
   */
  public function isCurrentPathExcluded(): bool {
    $current_path = $this->currentPath->getPath();
    return $this->isPathExcluded($current_path);
  }

  /**
   * Checks if a given path is excluded from visualization.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if the path is excluded, FALSE otherwise.
   */
  public function isPathExcluded(string $path): bool {
    $config = $this->configFactory->get('cacheviz.settings');
    $excluded_paths = $config->get('excluded_paths') ?? '';

    if (empty($excluded_paths)) {
      return FALSE;
    }

    return $this->pathMatcher->matchPath($path, $excluded_paths);
  }

}
