<?php

declare(strict_types=1);

namespace Drupal\cacheviz;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\Context\CacheContextsManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\Renderer as CoreRenderer;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Decorates core's Renderer to inject cache metadata as HTML comments.
 *
 * Only injects comments when the module is enabled in configuration,
 * user has 'view cacheviz debug' permission, and current path is not excluded.
 */
class Renderer extends CoreRenderer {

  /**
   * The cache context manager.
   *
   * @var \Drupal\Core\Cache\Context\CacheContextsManager|null
   */
  protected ?CacheContextsManager $cacheContextManager = NULL;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|null
   */
  protected ?ConfigFactoryInterface $configFactory = NULL;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface|null
   */
  protected ?AccountProxyInterface $currentUser = NULL;

  /**
   * The path matcher service.
   *
   * @var \Drupal\cacheviz\PathMatcher|null
   */
  protected ?PathMatcher $pathMatcher = NULL;

  /**
   * Whether processing should occur (cached per request).
   *
   * @var bool|null
   */
  protected ?bool $shouldProcess = NULL;

  /**
   * Sets the cache context manager.
   *
   * @param \Drupal\Core\Cache\Context\CacheContextsManager $cache_context_manager
   *   The cache context manager.
   */
  public function setCacheContextManager(CacheContextsManager $cache_context_manager): void {
    $this->cacheContextManager = $cache_context_manager;
  }

  /**
   * Sets the config factory.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function setConfigFactory(ConfigFactoryInterface $config_factory): void {
    $this->configFactory = $config_factory;
  }

  /**
   * Sets the current user.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function setCurrentUser(AccountProxyInterface $current_user): void {
    $this->currentUser = $current_user;
  }

  /**
   * Sets the path matcher.
   *
   * @param \Drupal\cacheviz\PathMatcher $path_matcher
   *   The path matcher.
   */
  public function setPathMatcher(PathMatcher $path_matcher): void {
    $this->pathMatcher = $path_matcher;
  }

  /**
   * {@inheritdoc}
   *
   * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
   */
  protected function doRender(&$elements, $is_root_call = FALSE) {
    // Store original cache metadata before rendering.
    $original_cache = $elements['#cache'] ?? [];

    // Call parent renderer.
    $result = parent::doRender($elements, $is_root_call);

    // Early return if we shouldn't process.
    if (!$this->shouldProcessRequest()) {
      return $result;
    }

    // When there is no output, there is nothing to visualize.
    if ($result === '') {
      return '';
    }

    // Only wrap content that contains HTML markup.
    // HTML comments are only valid within HTML content, not plain text.
    $result_string = (string) $result;
    if ($result_string === \strip_tags($result_string)) {
      return $result;
    }

    // Build pre-bubbling cache metadata with defaults.
    $pre_bubbling_cache = [];
    $pre_bubbling_cache['tags'] = $original_cache['tags'] ?? [];
    $pre_bubbling_cache['max-age'] = $original_cache['max-age'] ?? Cache::PERMANENT;

    // Apply required cache contexts for root calls or cached elements.
    if ($is_root_call || isset($elements['#cache']['keys'])) {
      $required_cache_contexts = $this->rendererConfig['required_cache_contexts'];
      if (isset($original_cache['contexts'])) {
        $pre_bubbling_cache['contexts'] = Cache::mergeContexts(
          $original_cache['contexts'],
          $required_cache_contexts
        );
      }
      else {
        $pre_bubbling_cache['contexts'] = $required_cache_contexts;
      }
    }

    // Build cache metadata for visualization.
    $final_cache = $elements['#cache'] ?? [];
    $interesting_keys = ['keys', 'contexts', 'tags', 'max-age'];
    $has_interesting_data = \array_intersect(\array_keys($final_cache), $interesting_keys)
      || \array_intersect(\array_keys($pre_bubbling_cache), $interesting_keys);

    if ($has_interesting_data && $this->cacheContextManager !== NULL) {
      // Get cache context keys.
      $contexts = $final_cache['contexts'] ?? [];
      $cache_context_keys = [];
      if (!empty($contexts)) {
        $cache_context_keys = [
          'cache-context-keys' => $this->cacheContextManager->convertTokensToKeys($contexts)->getKeys(),
        ];
      }

      $cache_data = [
        'final' => $final_cache,
        'pre_bubbling' => $pre_bubbling_cache,
        'context_keys' => $cache_context_keys,
      ];

      $prefix = '<!--CACHEVIZ_START--><!--' . htmlspecialchars(Json::encode($cache_data), ENT_NOQUOTES) . '-->';
      $suffix = '<!--CACHEVIZ_END-->';

      $wrapped = $prefix . $result_string . $suffix;

      // Update the elements markup and return.
      $elements['#markup'] = Markup::create($wrapped);
      return $elements['#markup'];
    }

    return $result;
  }

  /**
   * Checks if the request should be processed.
   *
   * Caches the result for the duration of the request.
   *
   * @return bool
   *   TRUE if processing should occur, FALSE otherwise.
   */
  protected function shouldProcessRequest(): bool {
    // Return cached result if available.
    if ($this->shouldProcess !== NULL) {
      return $this->shouldProcess;
    }

    // Default to not processing if dependencies aren't set.
    if ($this->configFactory === NULL || $this->currentUser === NULL || $this->pathMatcher === NULL) {
      $this->shouldProcess = FALSE;
      return FALSE;
    }

    $config = $this->configFactory->get('cacheviz.settings');
    if (!$config->get('enabled')) {
      $this->shouldProcess = FALSE;
      return FALSE;
    }

    if (!$this->currentUser->hasPermission('view cacheviz debug')) {
      $this->shouldProcess = FALSE;
      return FALSE;
    }

    if ($this->pathMatcher->isCurrentPathExcluded()) {
      $this->shouldProcess = FALSE;
      return FALSE;
    }

    $this->shouldProcess = TRUE;
    return TRUE;
  }

}
