<?php

declare(strict_types=1);

namespace Drupal\Tests\cacheviz\Kernel;

use Drupal\cacheviz\Renderer;
use Drupal\Core\Cache\Cache;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests for the Cacheviz Renderer decorator.
 *
 * @coversDefaultClass \Drupal\cacheviz\Renderer
 * @group cacheviz
 */
class RendererTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', 'user', 'cacheviz'];

  /**
   * The renderer service.
   *
   * @var \Drupal\cacheviz\Renderer
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['system', 'cacheviz']);
    $this->installEntitySchema('user');

    $this->renderer = $this->container->get('renderer');
  }

  /**
   * Tests that the renderer decorator is properly registered.
   */
  public function testRendererIsDecorated(): void {
    $this->assertInstanceOf(Renderer::class, $this->renderer);
  }

  /**
   * Tests rendering when cacheviz is disabled.
   */
  public function testRenderingWhenDisabled(): void {
    // Disable cacheviz.
    $this->config('cacheviz.settings')
      ->set('enabled', FALSE)
      ->save();

    $render_array = [
      '#markup' => '<div>Test content</div>',
      '#cache' => [
        'tags' => ['test_tag'],
        'contexts' => ['url'],
        'max-age' => Cache::PERMANENT,
      ],
    ];

    $output = (string) $this->renderer->renderRoot($render_array);

    // When disabled, no CACHEVIZ comments should be present.
    $this->assertStringNotContainsString('CACHEVIZ_START', $output);
    $this->assertStringNotContainsString('CACHEVIZ_END', $output);
    $this->assertStringContainsString('Test content', $output);
  }

  /**
   * Tests rendering when user lacks permission.
   */
  public function testRenderingWithoutPermission(): void {
    // Enable cacheviz but don't grant permission.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->save();

    $render_array = [
      '#markup' => '<div>Test content</div>',
      '#cache' => [
        'tags' => ['test_tag'],
        'contexts' => ['url'],
        'max-age' => Cache::PERMANENT,
      ],
    ];

    $output = (string) $this->renderer->renderRoot($render_array);

    // Without permission, no CACHEVIZ comments should be present.
    $this->assertStringNotContainsString('CACHEVIZ_START', $output);
    $this->assertStringNotContainsString('CACHEVIZ_END', $output);
    $this->assertStringContainsString('Test content', $output);
  }

  /**
   * Tests that plain text content is not wrapped.
   */
  public function testPlainTextNotWrapped(): void {
    // Enable cacheviz.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->save();

    // Create a user with permission.
    $user = $this->createUser(['view cacheviz debug']);
    $this->setCurrentUser($user);

    $render_array = [
      '#plain_text' => 'Plain text without HTML',
      '#cache' => [
        'tags' => ['test_tag'],
      ],
    ];

    $output = (string) $this->renderer->renderRoot($render_array);

    // Plain text should not have CACHEVIZ comments.
    $this->assertStringNotContainsString('CACHEVIZ_START', $output);
  }

  /**
   * Creates a test user with given permissions.
   *
   * @param array $permissions
   *   An array of permission strings.
   *
   * @return \Drupal\user\UserInterface
   *   The created user entity.
   */
  protected function createUser(array $permissions = []) {
    /** @var \Drupal\user\RoleInterface $role */
    $role = Role::create([
      'id' => 'test_role',
      'label' => 'Test Role',
    ]);
    foreach ($permissions as $permission) {
      $role->grantPermission($permission);
    }
    $role->save();

    $user = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
      'status' => 1,
      'roles' => ['test_role'],
    ]);
    $user->save();

    return $user;
  }

  /**
   * Sets the current user.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user to set as current.
   */
  protected function setCurrentUser($user): void {
    $this->container->get('current_user')->setAccount($user);
  }

}
