/**
 * @file
 * HTML comment parser for Cacheviz.
 *
 * Parses HTML comments injected by the Cacheviz renderer decorator.
 * Handles nested cache metadata comments using a stack-based approach.
 */

(function (window) {
  'use strict';

  /**
   * Decode HTML entities in a string.
   *
   * @param {string} text The text with HTML entities.
   * @return {string} The decoded text.
   */
  function decodeHtmlEntities(text) {
    const textarea = document.createElement('textarea');
    textarea.innerHTML = text;
    return textarea.value;
  }

  /**
   * Parse Cacheviz HTML comments from the DOM.
   *
   * Uses a stack to handle nested render elements properly.
   *
   * @return {Array} Array of parsed cache data objects with element references.
   */
  function parseCachevizComments() {
    const results = [];
    const stack = [];
    const iterator = document.createNodeIterator(
      document.body,
      NodeFilter.SHOW_COMMENT,
      null,
      false
    );

    let node;

    while ((node = iterator.nextNode())) {
      const text = node.textContent.trim();

      if (text === 'CACHEVIZ_START') {
        // Push new context onto stack.
        stack.push({
          startComment: node,
          data: null
        });
      }
      else if (text === 'CACHEVIZ_END') {
        // Pop from stack and create result.
        if (stack.length > 0) {
          const context = stack.pop();

          if (context.data) {
            // Find the element between start and end comments.
            const element = findElementBetween(context.startComment, node);

            if (element) {
              results.push({
                element: element,
                data: context.data,
                startComment: context.startComment,
                endComment: node
              });
            }
          }
        }
      }
      else if (stack.length > 0 && stack[stack.length - 1].data === null) {
        // This should be the JSON data comment for the current context.
        try {
          const decoded = decodeHtmlEntities(text);
          stack[stack.length - 1].data = JSON.parse(decoded);
        }
        catch (e) {
          // Not valid JSON, skip.
        }
      }
    }

    return results;
  }

  /**
   * Find the main HTML element between two comment nodes.
   *
   * @param {Comment} startComment The start comment node.
   * @param {Comment} endComment The end comment node.
   * @return {Element|null} The found element or null.
   */
  function findElementBetween(startComment, endComment) {
    let node = startComment.nextSibling;

    while (node && node !== endComment) {
      if (node.nodeType === Node.ELEMENT_NODE) {
        return node;
      }
      node = node.nextSibling;
    }

    // If no element found between comments, try parent.
    if (startComment.parentElement) {
      return startComment.parentElement;
    }

    return null;
  }

  /**
   * Generate a CSS selector for an element.
   *
   * @param {Element} element The DOM element.
   * @return {string} A CSS selector string.
   */
  function getElementSelector(element) {
    if (!element) {
      return '';
    }

    // Try ID first.
    if (element.id) {
      return '#' + element.id;
    }

    // Build selector from classes.
    const parts = [];
    parts.push(element.tagName.toLowerCase());

    if (element.className && typeof element.className === 'string') {
      const classes = element.className.split(/\s+/).filter(Boolean).slice(0, 2);
      if (classes.length) {
        parts.push('.' + classes.join('.'));
      }
    }

    return parts.join('');
  }

  // Export functions.
  window.CachevizComments = {
    parse: parseCachevizComments,
    getSelector: getElementSelector
  };

})(window);
