/**
 * @file
 * Cacheviz - Cache Visualization Tool for Drupal.
 *
 * A developer tool for visualizing and debugging Drupal's render cache.
 */

(function (window, document) {
  'use strict';

  /* =========================================================================
     CONSTANTS & CONFIGURATION
     ========================================================================= */

  const CONFIG = {
    storageKey: 'cacheviz_state',
    shortcuts: {
      toggle: { key: 'c', ctrl: true, shift: true },
      highlight: { key: 'h', ctrl: true, shift: true }
    },
    maxAge: {
      permanent: -1,
      uncacheable: 0
    }
  };

  const SEVERITY = {
    critical: { label: 'Critical', color: '#ef4444', icon: '!!' },
    warning: { label: 'Warning', color: '#f59e0b', icon: '!' },
    info: { label: 'Info', color: '#3b82f6', icon: 'i' }
  };

  /* =========================================================================
     UTILITY FUNCTIONS
     ========================================================================= */

  const utils = {
    /**
     * Parse max-age value to number.
     */
    parseMaxAge(value) {
      if (value === undefined || value === null) {
        return undefined;
      }
      const num = parseInt(value, 10);
      return isNaN(num) ? undefined : num;
    },

    /**
     * Convert value to array (handles PHP JSON objects).
     */
    toArray(value) {
      if (!value) {
        return [];
      }
      if (Array.isArray(value)) {
        return value;
      }
      if (typeof value === 'object') {
        return Object.values(value);
      }
      return [];
    },

    /**
     * Escape HTML special characters.
     */
    escapeHtml(text) {
      const div = document.createElement('div');
      div.textContent = String(text);
      return div.innerHTML;
    },

    /**
     * Format max-age for display.
     */
    formatMaxAge(maxAge) {
      if (maxAge === CONFIG.maxAge.uncacheable) {
        return { text: 'Uncacheable', class: 'critical' };
      }
      if (maxAge === CONFIG.maxAge.permanent) {
        return { text: 'Permanent', class: 'success' };
      }
      if (maxAge > 0) {
        if (maxAge >= 86400) {
          return { text: `${Math.floor(maxAge / 86400)}d`, class: 'limited' };
        }
        if (maxAge >= 3600) {
          return { text: `${Math.floor(maxAge / 3600)}h`, class: 'limited' };
        }
        if (maxAge >= 60) {
          return { text: `${Math.floor(maxAge / 60)}m`, class: 'limited' };
        }
        return { text: `${maxAge}s`, class: 'limited' };
      }
      return { text: 'Unknown', class: 'unknown' };
    },

    /**
     * Generate a readable selector for an element.
     */
    getSelector(element) {
      if (!element || !element.tagName) {
        return 'unknown';
      }

      if (element.id) {
        return `#${element.id}`;
      }

      const tag = element.tagName.toLowerCase();
      const classes = element.className && typeof element.className === 'string'
        ? element.className.split(/\s+/).filter(c => c && !c.startsWith('cacheviz')).slice(0, 2)
        : [];

      if (classes.length) {
        return `${tag}.${classes.join('.')}`;
      }

      return tag;
    },

    /**
     * Debounce function calls.
     */
    debounce(fn, delay) {
      let timeout;
      return (...args) => {
        clearTimeout(timeout);
        timeout = setTimeout(() => fn(...args), delay);
      };
    }
  };

  /* =========================================================================
     DATA ANALYSIS
     ========================================================================= */

  class CacheAnalyzer {
    constructor(elements) {
      this.elements = elements;
      this.analyze();
    }

    analyze() {
      this.stats = this.computeStats();
      this.issues = this.detectIssues();
      this.contextGroups = this.groupByContext();
    }

    computeStats() {
      let uncacheable = 0;
      let permanent = 0;
      let limited = 0;
      let hasUserContext = 0;
      let hasSessionContext = 0;
      const allContexts = new Set();
      const allTags = new Set();

      this.elements.forEach(item => {
        const maxAge = this.getEffectiveMaxAge(item);

        if (maxAge === CONFIG.maxAge.uncacheable) {
          uncacheable++;
        }
        else if (maxAge === CONFIG.maxAge.permanent) {
          permanent++;
        }
        else {
          limited++;
        }

        const contexts = this.getAllContexts(item);
        contexts.forEach(c => {
          allContexts.add(c);
          if (c === 'user' || c.startsWith('user.')) {
            hasUserContext++;
          }
          if (c === 'session' || c.startsWith('session.')) {
            hasSessionContext++;
          }
        });

        const tags = utils.toArray(item.data.final?.tags);
        tags.forEach(t => allTags.add(t));
      });

      return {
        total: this.elements.length,
        uncacheable,
        permanent,
        limited,
        hasUserContext,
        hasSessionContext,
        uniqueContexts: allContexts.size,
        uniqueTags: allTags.size,
        allContexts: Array.from(allContexts).sort(),
        allTags: Array.from(allTags).sort()
      };
    }

    detectIssues() {
      const issues = [];
      const groupedIssues = new Map();

      this.elements.forEach(item => {
        const maxAge = this.getEffectiveMaxAge(item);
        const contexts = this.getAllContexts(item);
        const tags = utils.toArray(item.data.final?.tags);
        const selector = utils.getSelector(item.element);

        // Build occurrence data for grouping.
        const occurrence = {
          element: item.element,
          selector,
          data: item.data,
          contexts,
          tags,
          maxAge
        };

        // Uncacheable elements (Critical).
        if (maxAge === CONFIG.maxAge.uncacheable) {
          // Determine WHY it's uncacheable for better grouping.
          const preBubblingMaxAge = utils.parseMaxAge(item.data.pre_bubbling?.['max-age']);
          const finalMaxAge = utils.parseMaxAge(item.data.final?.['max-age']);
          const preBubblingContexts = utils.toArray(item.data.pre_bubbling?.contexts);
          const finalContexts = utils.toArray(item.data.final?.contexts);

          // Identify the likely cause.
          let cause = 'unknown';
          let causeDetail = '';

          // Check if it was uncacheable from the start (pre-bubbling).
          if (preBubblingMaxAge === CONFIG.maxAge.uncacheable) {
            cause = 'element_itself';
            causeDetail = 'Element set max-age: 0 directly';
          }
          else if (finalMaxAge === CONFIG.maxAge.uncacheable && preBubblingMaxAge !== CONFIG.maxAge.uncacheable) {
            cause = 'bubbled_from_child';
            causeDetail = 'Inherited from child element';
          }

          // Check for problematic contexts.
          const problematicContexts = finalContexts.filter(c =>
            c === 'user' || c.startsWith('user.') ||
            c === 'session' || c.startsWith('session.')
          );

          if (problematicContexts.length > 0 && cause === 'unknown') {
            cause = 'context_based';
            causeDetail = problematicContexts.join(', ');
          }

          // Group by cause.
          const groupKey = `critical:uncacheable:${cause}`;
          if (!groupedIssues.has(groupKey)) {
            groupedIssues.set(groupKey, {
              severity: 'critical',
              type: 'uncacheable',
              cause,
              title: 'Uncacheable Element',
              description: causeDetail || 'max-age: 0 - renders on every request',
              occurrences: []
            });
          }
          groupedIssues.get(groupKey).occurrences.push(occurrence);
        }

        // Per-user cache context (Warning).
        const userContexts = contexts.filter(c => c === 'user' || c.startsWith('user.'));
        if (userContexts.length > 0) {
          const contextKey = userContexts.sort().join(',');
          const groupKey = `warning:user:${contextKey}`;
          if (!groupedIssues.has(groupKey)) {
            groupedIssues.set(groupKey, {
              severity: 'warning',
              type: 'user-context',
              title: 'Per-User Cache Variation',
              description: `Varies by: ${userContexts.join(', ')}`,
              occurrences: []
            });
          }
          groupedIssues.get(groupKey).occurrences.push(occurrence);
        }

        // Session context (Warning).
        const sessionContexts = contexts.filter(c => c === 'session' || c.startsWith('session.'));
        if (sessionContexts.length > 0) {
          const contextKey = sessionContexts.sort().join(',');
          const groupKey = `warning:session:${contextKey}`;
          if (!groupedIssues.has(groupKey)) {
            groupedIssues.set(groupKey, {
              severity: 'warning',
              type: 'session-context',
              title: 'Session-Based Cache',
              description: `Varies by: ${sessionContexts.join(', ')}`,
              occurrences: []
            });
          }
          groupedIssues.get(groupKey).occurrences.push(occurrence);
        }

        // Cookie context (Info).
        const cookieContexts = contexts.filter(c => c.startsWith('cookies:'));
        if (cookieContexts.length > 0) {
          const contextKey = cookieContexts.sort().join(',');
          const groupKey = `info:cookie:${contextKey}`;
          if (!groupedIssues.has(groupKey)) {
            groupedIssues.set(groupKey, {
              severity: 'info',
              type: 'cookie-context',
              title: 'Cookie-Based Cache',
              description: `Varies by: ${cookieContexts.join(', ')}`,
              occurrences: []
            });
          }
          groupedIssues.get(groupKey).occurrences.push(occurrence);
        }
      });

      // Convert to array and add first element reference for backward compat.
      groupedIssues.forEach(issue => {
        if (issue.occurrences.length > 0) {
          issue.element = issue.occurrences[0].element;
          issue.selector = issue.occurrences[0].selector;
          issue.data = issue.occurrences[0].data;
          issue.contexts = issue.occurrences[0].contexts;
          issues.push(issue);
        }
      });

      // Sort by severity.
      const severityOrder = { critical: 0, warning: 1, info: 2 };
      issues.sort((a, b) => severityOrder[a.severity] - severityOrder[b.severity]);

      return issues;
    }

    groupByContext() {
      const groups = {};

      this.elements.forEach(item => {
        const contexts = this.getAllContexts(item);
        contexts.forEach(context => {
          if (!groups[context]) {
            groups[context] = [];
          }
          groups[context].push(item);
        });
      });

      return groups;
    }

    getEffectiveMaxAge(item) {
      const finalMaxAge = utils.parseMaxAge(item.data.final?.['max-age']);
      const preMaxAge = utils.parseMaxAge(item.data.pre_bubbling?.['max-age']);

      // If either is uncacheable, the element is uncacheable.
      if (finalMaxAge === CONFIG.maxAge.uncacheable || preMaxAge === CONFIG.maxAge.uncacheable) {
        return CONFIG.maxAge.uncacheable;
      }

      return finalMaxAge !== undefined ? finalMaxAge : preMaxAge;
    }

    /**
     * Build the bubble chain showing how max-age:0 propagates up the DOM.
     *
     * @param {Element} element The element to trace.
     * @return {Array} Array of chain links from root cause to top-level parent.
     */
    buildBubbleChain(element) {
      if (!element) {
        return [];
      }

      const chain = [];
      let current = element;
      let foundRoot = false;

      // Walk up the DOM looking for cacheviz-tracked ancestors.
      while (current && current !== document.body) {
        const item = this.elements.find(i => i.element === current);

        if (item) {
          const preBubblingMaxAge = utils.parseMaxAge(item.data.pre_bubbling?.['max-age']);
          const finalMaxAge = utils.parseMaxAge(item.data.final?.['max-age']);
          const isUncacheable = finalMaxAge === CONFIG.maxAge.uncacheable;
          const wasUncacheableBeforeBubbling = preBubblingMaxAge === CONFIG.maxAge.uncacheable;
          const becameUncacheable = isUncacheable && !wasUncacheableBeforeBubbling;

          // Determine the role in the chain.
          let role = 'inherited';
          if (wasUncacheableBeforeBubbling && !foundRoot) {
            role = 'source';
            foundRoot = true;
          }
          else if (becameUncacheable) {
            role = 'inherited';
          }
          else if (!isUncacheable) {
            // This ancestor is cacheable, stop here.
            break;
          }

          chain.push({
            element: current,
            selector: utils.getSelector(current),
            data: item.data,
            preBubblingMaxAge,
            finalMaxAge,
            role,
            contexts: this.getAllContexts(item),
            tags: utils.toArray(item.data.final?.tags)
          });
        }

        current = current.parentElement;
      }

      // Reverse so root cause is first.
      chain.reverse();

      // Mark the deepest uncacheable element as the root if not found.
      if (!foundRoot && chain.length > 0) {
        chain[0].role = 'source';
      }

      return chain;
    }

    /**
     * Find the root cause of uncacheability for an element.
     *
     * @param {Element} element The uncacheable element.
     * @return {Object|null} The root cause element and reason.
     */
    findRootCause(element) {
      if (!element) {
        return null;
      }

      // Look for descendant elements that are the source of max-age:0.
      const descendants = [];
      const walker = document.createTreeWalker(element, NodeFilter.SHOW_ELEMENT);

      let node;
      while ((node = walker.nextNode())) {
        const item = this.elements.find(i => i.element === node);
        if (item) {
          const preBubblingMaxAge = utils.parseMaxAge(item.data.pre_bubbling?.['max-age']);
          if (preBubblingMaxAge === CONFIG.maxAge.uncacheable) {
            descendants.push({
              element: node,
              selector: utils.getSelector(node),
              data: item.data,
              tags: utils.toArray(item.data.pre_bubbling?.tags),
              contexts: utils.toArray(item.data.pre_bubbling?.contexts)
            });
          }
        }
      }

      // Find the deepest one (most specific cause).
      if (descendants.length > 0) {
        // Sort by DOM depth (deepest first).
        descendants.sort((a, b) => {
          const depthA = this.getElementDepth(a.element);
          const depthB = this.getElementDepth(b.element);
          return depthB - depthA;
        });

        const rootCause = descendants[0];

        // Try to identify why it's uncacheable.
        let reason = 'Unknown';
        const tags = rootCause.tags.join(' ');

        if (tags.includes('CACHE_MISS_IF_UNCACHEABLE_HTTP_METHOD:form')) {
          reason = 'Contains a form (POST requests are uncacheable)';
        }
        else if (tags.includes('webform:')) {
          reason = 'Contains a webform';
        }
        else if (rootCause.contexts.includes('session') || rootCause.contexts.some(c => c.startsWith('session.'))) {
          reason = 'Session-dependent content';
        }
        else if (rootCause.contexts.includes('user') || rootCause.contexts.some(c => c.startsWith('user.'))) {
          reason = 'User-specific content';
        }

        return {
          ...rootCause,
          reason,
          totalDescendants: descendants.length
        };
      }

      return null;
    }

    getElementDepth(element) {
      let depth = 0;
      let current = element;
      while (current.parentElement) {
        depth++;
        current = current.parentElement;
      }
      return depth;
    }

    getAllContexts(item) {
      const final = utils.toArray(item.data.final?.contexts);
      const pre = utils.toArray(item.data.pre_bubbling?.contexts);
      return [...new Set([...final, ...pre])];
    }
  }

  /* =========================================================================
     UI COMPONENTS
     ========================================================================= */

  class CachevizUI {
    constructor(analyzer, pageInfo) {
      this.analyzer = analyzer;
      this.pageInfo = pageInfo;
      this.state = this.loadState();
      this.activeView = 'dashboard';
      this.searchQuery = '';
      this.highlightsActive = false;

      // Store bound event handlers for proper cleanup.
      this._boundHandleElementHover = this.handleElementHover.bind(this);
      this._boundHandleElementLeave = this.handleElementLeave.bind(this);

      this.render();
      this.bindEvents();
      this.bindKeyboardShortcuts();

      // Always highlight critical elements from the start.
      this.highlightCriticalElements();

      if (this.state.autoHighlight) {
        this.toggleHighlights(true);
      }
    }

    /**
     * Highlight all critical (uncacheable) elements with persistent red highlight.
     */
    highlightCriticalElements() {
      const criticalIssues = this.analyzer.issues.filter(i => i.severity === 'critical');

      criticalIssues.forEach(issue => {
        const occurrences = issue.occurrences || [issue];
        occurrences.forEach(occ => {
          if (occ.element) {
            occ.element.classList.add('cv-hl', 'cv-hl--critical', 'cv-hl--always');
          }
        });
      });
    }

    loadState() {
      try {
        const saved = localStorage.getItem(CONFIG.storageKey);
        if (saved) {
          return JSON.parse(saved);
        }
      }
      catch (e) {
        // Ignore.
      }
      return {
        isOpen: true,
        isMinimized: false,
        autoHighlight: true,
        expandedSections: {}
      };
    }

    saveState() {
      try {
        localStorage.setItem(CONFIG.storageKey, JSON.stringify(this.state));
      }
      catch (e) {
        // Ignore.
      }
    }

    render() {
      // Remove existing panel if any.
      const existing = document.getElementById('cacheviz-panel');
      if (existing) {
        existing.remove();
      }

      this.panel = document.createElement('div');
      this.panel.id = 'cacheviz-panel';
      this.panel.className = 'cv-panel';
      if (!this.state.isOpen) {
        this.panel.classList.add('cv-panel--closed');
      }
      if (this.state.isMinimized) {
        this.panel.classList.add('cv-panel--minimized');
      }

      this.panel.innerHTML = this.renderPanel();
      document.body.appendChild(this.panel);

      // Create tooltip.
      this.tooltip = document.createElement('div');
      this.tooltip.className = 'cv-tooltip';
      this.tooltip.style.display = 'none';
      document.body.appendChild(this.tooltip);
    }

    renderPanel() {
      const { stats, issues } = this.analyzer;
      const criticalCount = issues.filter(i => i.severity === 'critical').length;
      const warningCount = issues.filter(i => i.severity === 'warning').length;

      return `
        <div class="cv-header">
          <div class="cv-header__left">
            <div class="cv-logo">
              <span class="cv-logo__icon">◉</span>
              <span class="cv-logo__text">Cacheviz</span>
            </div>
            ${this.renderHealthBadge()}
          </div>
          <div class="cv-header__right">
            <button class="cv-btn cv-btn--icon" data-action="highlight" title="Toggle Highlights (Ctrl+Shift+H)">
              <span class="cv-icon">${this.highlightsActive ? '◉' : '○'}</span>
            </button>
            <button class="cv-btn cv-btn--icon" data-action="minimize" title="Minimize">
              <span class="cv-icon">−</span>
            </button>
            <button class="cv-btn cv-btn--icon" data-action="close" title="Close (Ctrl+Shift+C)">
              <span class="cv-icon">×</span>
            </button>
          </div>
        </div>

        <div class="cv-body">
          <nav class="cv-nav">
            <button class="cv-nav__item ${this.activeView === 'dashboard' ? 'cv-nav__item--active' : ''}" data-view="dashboard">
              Overview
            </button>
            <button class="cv-nav__item ${this.activeView === 'issues' ? 'cv-nav__item--active' : ''}" data-view="issues">
              Issues ${criticalCount + warningCount > 0 ? `<span class="cv-nav__badge">${criticalCount + warningCount}</span>` : ''}
            </button>
            <button class="cv-nav__item ${this.activeView === 'elements' ? 'cv-nav__item--active' : ''}" data-view="elements">
              Elements
            </button>
          </nav>

          <div class="cv-content">
            ${this.renderActiveView()}
          </div>
        </div>

        <div class="cv-footer">
          <span class="cv-shortcut">Ctrl+Shift+C</span> toggle
          <span class="cv-sep">|</span>
          <span class="cv-shortcut">Ctrl+Shift+H</span> highlights
        </div>
      `;
    }

    renderHealthBadge() {
      const { stats, issues } = this.analyzer;
      const criticalCount = issues.filter(i => i.severity === 'critical').length;

      let status = 'good';
      let label = 'Healthy';

      if (criticalCount > 0) {
        status = 'critical';
        label = `${criticalCount} Critical`;
      }
      else if (stats.hasUserContext > 0) {
        status = 'warning';
        label = 'Has Warnings';
      }

      return `<span class="cv-health cv-health--${status}">${label}</span>`;
    }

    renderActiveView() {
      switch (this.activeView) {
        case 'issues':
          return this.renderIssuesView();
        case 'elements':
          return this.renderElementsView();
        default:
          return this.renderDashboardView();
      }
    }

    renderDashboardView() {
      const { stats } = this.analyzer;
      const pageMaxAge = utils.parseMaxAge(this.pageInfo.maxAge);
      const pageMaxAgeFormatted = utils.formatMaxAge(pageMaxAge);

      return `
        <div class="cv-dashboard">
          <!-- Page Status -->
          <div class="cv-card cv-card--page">
            <div class="cv-card__header">
              <span class="cv-card__title">Page Cache Status</span>
            </div>
            <div class="cv-card__body">
              <div class="cv-stat cv-stat--large">
                <span class="cv-stat__value cv-stat__value--${pageMaxAgeFormatted.class}">${pageMaxAgeFormatted.text}</span>
                <span class="cv-stat__label">Max-Age</span>
              </div>
              <div class="cv-stat-row">
                <div class="cv-stat">
                  <span class="cv-stat__value">${utils.toArray(this.pageInfo.contexts).length}</span>
                  <span class="cv-stat__label">Contexts</span>
                </div>
                <div class="cv-stat">
                  <span class="cv-stat__value">${utils.toArray(this.pageInfo.tags).length}</span>
                  <span class="cv-stat__label">Tags</span>
                </div>
              </div>
            </div>
          </div>

          <!-- Elements Overview -->
          <div class="cv-card">
            <div class="cv-card__header">
              <span class="cv-card__title">Elements Breakdown</span>
              <span class="cv-card__subtitle">${stats.total} total</span>
            </div>
            <div class="cv-card__body">
              <div class="cv-breakdown">
                ${this.renderBreakdownBar(stats)}
              </div>
              <div class="cv-breakdown__legend">
                <div class="cv-breakdown__item">
                  <span class="cv-dot cv-dot--critical"></span>
                  <span>Uncacheable: ${stats.uncacheable}</span>
                </div>
                <div class="cv-breakdown__item">
                  <span class="cv-dot cv-dot--limited"></span>
                  <span>Time-limited: ${stats.limited}</span>
                </div>
                <div class="cv-breakdown__item">
                  <span class="cv-dot cv-dot--success"></span>
                  <span>Permanent: ${stats.permanent}</span>
                </div>
              </div>
            </div>
          </div>

          <!-- Quick Stats -->
          <div class="cv-card">
            <div class="cv-card__header">
              <span class="cv-card__title">Cache Variations</span>
            </div>
            <div class="cv-card__body">
              <div class="cv-mini-stats">
                <div class="cv-mini-stat ${stats.hasUserContext > 0 ? 'cv-mini-stat--warning' : ''}">
                  <span class="cv-mini-stat__value">${stats.hasUserContext}</span>
                  <span class="cv-mini-stat__label">Per-User</span>
                </div>
                <div class="cv-mini-stat ${stats.hasSessionContext > 0 ? 'cv-mini-stat--warning' : ''}">
                  <span class="cv-mini-stat__value">${stats.hasSessionContext}</span>
                  <span class="cv-mini-stat__label">Session</span>
                </div>
                <div class="cv-mini-stat">
                  <span class="cv-mini-stat__value">${stats.uniqueContexts}</span>
                  <span class="cv-mini-stat__label">Contexts</span>
                </div>
                <div class="cv-mini-stat">
                  <span class="cv-mini-stat__value">${stats.uniqueTags}</span>
                  <span class="cv-mini-stat__label">Tags</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      `;
    }

    renderBreakdownBar(stats) {
      if (stats.total === 0) {
        return '<div class="cv-bar"><div class="cv-bar__empty">No elements</div></div>';
      }

      const segments = [];
      if (stats.uncacheable > 0) {
        const pct = (stats.uncacheable / stats.total * 100).toFixed(1);
        segments.push(`<div class="cv-bar__segment cv-bar__segment--critical" style="width: ${pct}%" title="Uncacheable: ${stats.uncacheable}"></div>`);
      }
      if (stats.limited > 0) {
        const pct = (stats.limited / stats.total * 100).toFixed(1);
        segments.push(`<div class="cv-bar__segment cv-bar__segment--limited" style="width: ${pct}%" title="Time-limited: ${stats.limited}"></div>`);
      }
      if (stats.permanent > 0) {
        const pct = (stats.permanent / stats.total * 100).toFixed(1);
        segments.push(`<div class="cv-bar__segment cv-bar__segment--success" style="width: ${pct}%" title="Permanent: ${stats.permanent}"></div>`);
      }

      return `<div class="cv-bar">${segments.join('')}</div>`;
    }

    renderIssuesView() {
      const { issues } = this.analyzer;

      if (issues.length === 0) {
        return `
          <div class="cv-empty">
            <div class="cv-empty__icon">✓</div>
            <div class="cv-empty__title">No Issues Found</div>
            <div class="cv-empty__text">All elements have optimal cache configuration.</div>
          </div>
        `;
      }

      // Count total occurrences.
      const totalOccurrences = issues.reduce((sum, i) => sum + (i.occurrences?.length || 1), 0);

      const grouped = {
        critical: issues.filter(i => i.severity === 'critical'),
        warning: issues.filter(i => i.severity === 'warning'),
        info: issues.filter(i => i.severity === 'info')
      };

      let html = `
        <div class="cv-issues">
          <div class="cv-issues__toolbar">
            <span class="cv-issues__summary">${issues.length} issues (${totalOccurrences} elements)</span>
            <button class="cv-btn cv-btn--small" data-action="expand-all">Expand All</button>
            <button class="cv-btn cv-btn--small" data-action="collapse-all">Collapse All</button>
          </div>
      `;

      Object.entries(grouped).forEach(([severity, items]) => {
        if (items.length === 0) {
          return;
        }

        const config = SEVERITY[severity];
        const totalInGroup = items.reduce((sum, i) => sum + (i.occurrences?.length || 1), 0);

        html += `
          <div class="cv-issue-group">
            <div class="cv-issue-group__header">
              <span class="cv-issue-group__icon" style="color: ${config.color}">${config.icon}</span>
              <span class="cv-issue-group__title">${config.label}</span>
              <span class="cv-issue-group__count">${items.length} issues</span>
              <span class="cv-issue-group__total">(${totalInGroup} elements)</span>
            </div>
            <div class="cv-issue-group__list">
              ${items.map((issue, i) => this.renderIssueItem(issue, i, severity)).join('')}
            </div>
          </div>
        `;
      });

      html += '</div>';
      return html;
    }

    renderIssueItem(issue, index, severity) {
      const occurrences = issue.occurrences || [issue];
      const hasMultiple = occurrences.length > 1;
      const isExpanded = this.state.expandedSections[`issue-${severity}-${index}`] || false;

      // Build title with cause info for critical issues.
      let titleExtra = '';
      if (issue.severity === 'critical' && issue.cause) {
        const causeLabels = {
          element_itself: '(Direct)',
          bubbled_from_child: '(Bubbled)',
          context_based: '(Context)',
          unknown: ''
        };
        titleExtra = causeLabels[issue.cause] || '';
      }

      // Build occurrence list HTML.
      let occurrencesHtml = '';
      if (hasMultiple) {
        occurrencesHtml = `
          <div class="cv-issue__occurrences ${isExpanded ? 'cv-issue__occurrences--expanded' : ''}">
            <div class="cv-issue__occurrences-header" data-action="toggle-occurrences" data-issue-key="issue-${severity}-${index}">
              <span class="cv-issue__expand-icon">${isExpanded ? '▼' : '▶'}</span>
              <span>${occurrences.length} affected elements</span>
            </div>
            <div class="cv-issue__occurrences-list" ${isExpanded ? '' : 'style="display: none;"'}>
              ${occurrences.map((occ, occIndex) => this.renderOccurrence(occ, occIndex, severity, index)).join('')}
            </div>
          </div>
        `;
      }

      // For single occurrence, show detailed info inline.
      const singleOcc = occurrences[0];
      const detailsHtml = !hasMultiple ? this.renderOccurrenceDetails(singleOcc) : '';

      return `
        <div class="cv-issue ${hasMultiple ? 'cv-issue--grouped' : ''}" data-issue-index="${index}" data-issue-type="${issue.type}" data-severity="${severity}">
          <div class="cv-issue__header">
            <div class="cv-issue__title-row">
              <span class="cv-issue__title">${utils.escapeHtml(issue.title)} ${titleExtra}</span>
              ${hasMultiple ? `<span class="cv-issue__badge">${occurrences.length}</span>` : ''}
            </div>
            <div class="cv-issue__desc">${utils.escapeHtml(issue.description)}</div>
          </div>
          ${detailsHtml}
          ${!hasMultiple ? `
            <div class="cv-issue__footer">
              <span class="cv-issue__selector">${utils.escapeHtml(issue.selector)}</span>
              <div class="cv-issue__actions">
                <button class="cv-btn cv-btn--small" data-action="locate" data-occ-index="0">Locate</button>
                <button class="cv-btn cv-btn--small" data-action="show-details" data-severity="${severity}" data-issue-index="${index}" data-occ-index="0">Details</button>
              </div>
            </div>
          ` : ''}
          ${occurrencesHtml}
        </div>
      `;
    }

    renderOccurrence(occurrence, occIndex, severity, issueIndex) {
      const maxAgeFormatted = utils.formatMaxAge(occurrence.maxAge);
      const isVisible = this.isElementInViewport(occurrence.element);
      const isHidden = occurrence.element ? this.isElementHidden(occurrence.element) : true;

      let visibilityIcon = '';
      let visibilityTitle = '';
      if (isHidden) {
        visibilityIcon = '<span class="cv-occurrence__visibility cv-occurrence__visibility--hidden" title="Element is hidden (display:none or visibility:hidden)">⊘</span>';
        visibilityTitle = 'Hidden';
      }
      else if (!isVisible) {
        visibilityIcon = '<span class="cv-occurrence__visibility cv-occurrence__visibility--offscreen" title="Element is off-screen (scroll to view)">↕</span>';
        visibilityTitle = 'Off-screen';
      }

      return `
        <div class="cv-occurrence" data-occ-index="${occIndex}" data-severity="${severity}" data-issue-index="${issueIndex}" data-hoverable="true">
          <div class="cv-occurrence__row">
            <div class="cv-occurrence__main">
              ${visibilityIcon}
              <span class="cv-occurrence__selector">${utils.escapeHtml(occurrence.selector)}</span>
              <span class="cv-occurrence__maxage cv-occurrence__maxage--${maxAgeFormatted.class}">${maxAgeFormatted.text}</span>
            </div>
            <div class="cv-occurrence__meta">
              ${occurrence.contexts.length > 0 ? `<span class="cv-occurrence__info">${occurrence.contexts.length} ctx</span>` : ''}
              ${occurrence.tags.length > 0 ? `<span class="cv-occurrence__info">${occurrence.tags.length} tags</span>` : ''}
            </div>
          </div>
          <div class="cv-occurrence__actions">
            <button class="cv-btn cv-btn--tiny" data-action="locate-occurrence" data-severity="${severity}" data-issue-index="${issueIndex}" data-occ-index="${occIndex}">Locate</button>
            <button class="cv-btn cv-btn--tiny" data-action="show-details" data-severity="${severity}" data-issue-index="${issueIndex}" data-occ-index="${occIndex}">Details</button>
          </div>
        </div>
      `;
    }

    isElementInViewport(element) {
      if (!element) {
        return false;
      }
      const rect = element.getBoundingClientRect();
      return (
        rect.top < window.innerHeight &&
        rect.bottom > 0 &&
        rect.left < window.innerWidth &&
        rect.right > 0
      );
    }

    isElementHidden(element) {
      if (!element) {
        return true;
      }
      const style = window.getComputedStyle(element);
      return style.display === 'none' ||
             style.visibility === 'hidden' ||
             style.opacity === '0' ||
             element.offsetParent === null;
    }

    renderOccurrenceDetails(occurrence) {
      const contexts = occurrence.contexts || [];
      const tags = occurrence.tags || [];
      const preBubbling = occurrence.data?.pre_bubbling || {};
      const final = occurrence.data?.final || {};

      const preMaxAge = utils.parseMaxAge(preBubbling['max-age']);
      const finalMaxAge = utils.parseMaxAge(final['max-age']);
      const preContexts = utils.toArray(preBubbling.contexts);
      const finalContexts = utils.toArray(final.contexts);

      return `
        <div class="cv-issue__details">
          <div class="cv-detail-section">
            <div class="cv-detail-section__title">Cache Metadata</div>
            <div class="cv-detail-row">
              <span class="cv-detail-label">Pre-bubble max-age:</span>
              <span class="cv-detail-value cv-detail-value--${utils.formatMaxAge(preMaxAge).class}">${utils.formatMaxAge(preMaxAge).text}</span>
            </div>
            <div class="cv-detail-row">
              <span class="cv-detail-label">Final max-age:</span>
              <span class="cv-detail-value cv-detail-value--${utils.formatMaxAge(finalMaxAge).class}">${utils.formatMaxAge(finalMaxAge).text}</span>
            </div>
            ${preMaxAge !== finalMaxAge ? `
              <div class="cv-detail-note cv-detail-note--warning">
                ⚠ max-age changed during bubbling
              </div>
            ` : ''}
          </div>

          ${contexts.length > 0 ? `
            <div class="cv-detail-section">
              <div class="cv-detail-section__title">Contexts (${contexts.length})</div>
              <div class="cv-detail-tags">
                ${contexts.slice(0, 6).map(c => `<span class="cv-detail-tag ${c.startsWith('user') ? 'cv-detail-tag--warning' : ''}">${utils.escapeHtml(c)}</span>`).join('')}
                ${contexts.length > 6 ? `<span class="cv-detail-more">+${contexts.length - 6} more</span>` : ''}
              </div>
              ${preContexts.length !== finalContexts.length ? `
                <div class="cv-detail-note">
                  Pre-bubble: ${preContexts.length} → Final: ${finalContexts.length} contexts
                </div>
              ` : ''}
            </div>
          ` : ''}

          ${tags.length > 0 ? `
            <div class="cv-detail-section">
              <div class="cv-detail-section__title">Tags (${tags.length})</div>
              <div class="cv-detail-tags">
                ${tags.slice(0, 4).map(t => `<span class="cv-detail-tag">${utils.escapeHtml(t)}</span>`).join('')}
                ${tags.length > 4 ? `<span class="cv-detail-more">+${tags.length - 4} more</span>` : ''}
              </div>
            </div>
          ` : ''}
        </div>
      `;
    }

    renderElementsView() {
      const { elements } = this.analyzer;

      return `
        <div class="cv-elements">
          <div class="cv-search">
            <input type="text" class="cv-search__input" placeholder="Filter elements..." value="${utils.escapeHtml(this.searchQuery)}">
          </div>
          <div class="cv-elements__list">
            ${this.renderFilteredElements()}
          </div>
        </div>
      `;
    }

    renderFilteredElements() {
      let filtered = this.analyzer.elements;

      if (this.searchQuery) {
        const query = this.searchQuery.toLowerCase();
        filtered = filtered.filter(item => {
          const selector = utils.getSelector(item.element).toLowerCase();
          const contexts = this.analyzer.getAllContexts(item).join(' ').toLowerCase();
          const tags = utils.toArray(item.data.final?.tags).join(' ').toLowerCase();
          return selector.includes(query) || contexts.includes(query) || tags.includes(query);
        });
      }

      if (filtered.length === 0) {
        return '<div class="cv-empty cv-empty--small">No matching elements</div>';
      }

      // Limit display for performance.
      const displayItems = filtered.slice(0, 50);

      let html = displayItems.map((item, i) => {
        const maxAge = this.analyzer.getEffectiveMaxAge(item);
        const maxAgeFormatted = utils.formatMaxAge(maxAge);
        const contexts = this.analyzer.getAllContexts(item);

        return `
          <div class="cv-element" data-element-index="${i}">
            <div class="cv-element__main">
              <span class="cv-element__selector">${utils.escapeHtml(utils.getSelector(item.element))}</span>
              <span class="cv-element__maxage cv-element__maxage--${maxAgeFormatted.class}">${maxAgeFormatted.text}</span>
            </div>
            <div class="cv-element__meta">
              ${contexts.length > 0 ? `<span class="cv-element__contexts">${contexts.length} contexts</span>` : ''}
            </div>
          </div>
        `;
      }).join('');

      if (filtered.length > 50) {
        html += `<div class="cv-more">Showing 50 of ${filtered.length} elements</div>`;
      }

      return html;
    }

    bindEvents() {
      // Header actions and general clicks.
      this.panel.addEventListener('click', (e) => {
        const actionEl = e.target.closest('[data-action]');
        if (!actionEl) {
          return;
        }

        const action = actionEl.dataset.action;

        switch (action) {
          case 'close':
            this.close();
            break;

          case 'minimize':
            this.toggleMinimize();
            break;

          case 'highlight':
            this.toggleHighlights();
            break;

          case 'locate':
            const issueEl = e.target.closest('.cv-issue');
            if (issueEl) {
              const severity = issueEl.dataset.severity;
              const issueIndex = parseInt(issueEl.dataset.issueIndex, 10);
              const occIndex = parseInt(actionEl.dataset.occIndex || '0', 10);
              this.locateIssueOccurrence(severity, issueIndex, occIndex);
            }
            break;

          case 'locate-occurrence':
            this.locateIssueOccurrence(
              actionEl.dataset.severity,
              parseInt(actionEl.dataset.issueIndex, 10),
              parseInt(actionEl.dataset.occIndex, 10)
            );
            break;

          case 'show-details':
            this.showOccurrenceDetails(
              actionEl.dataset.severity,
              parseInt(actionEl.dataset.issueIndex, 10),
              parseInt(actionEl.dataset.occIndex, 10)
            );
            break;

          case 'toggle-occurrences':
            this.toggleOccurrences(actionEl.dataset.issueKey);
            break;

          case 'expand-all':
            this.expandAllIssues();
            break;

          case 'collapse-all':
            this.collapseAllIssues();
            break;
        }
      });

      // Navigation.
      this.panel.addEventListener('click', (e) => {
        const navItem = e.target.closest('.cv-nav__item');
        if (navItem && navItem.dataset.view) {
          this.switchView(navItem.dataset.view);
        }
      });

      // Element clicks.
      this.panel.addEventListener('click', (e) => {
        const elementEl = e.target.closest('.cv-element');
        if (elementEl) {
          const index = parseInt(elementEl.dataset.elementIndex, 10);
          this.focusElement(index);
        }
      });

      // Search input.
      this.panel.addEventListener('input', utils.debounce((e) => {
        if (e.target.classList.contains('cv-search__input')) {
          this.searchQuery = e.target.value;
          this.updateElementsList();
        }
      }, 200));

      // Reopen from closed state.
      this.panel.addEventListener('click', (e) => {
        if (this.panel.classList.contains('cv-panel--closed')) {
          this.open();
        }
      });

      // Hover highlight for occurrences and issues.
      this.panel.addEventListener('mouseenter', (e) => {
        const hoverable = e.target.closest('[data-hoverable="true"]');
        if (hoverable) {
          this.highlightOccurrenceElement(hoverable);
        }
      }, true);

      this.panel.addEventListener('mouseleave', (e) => {
        const hoverable = e.target.closest('[data-hoverable="true"]');
        if (hoverable) {
          this.clearHoverHighlight();
        }
      }, true);
    }

    highlightOccurrenceElement(hoverableEl) {
      const severity = hoverableEl.dataset.severity;
      const issueIndex = parseInt(hoverableEl.dataset.issueIndex, 10);
      const occIndex = parseInt(hoverableEl.dataset.occIndex, 10);

      const issues = this.analyzer.issues.filter(i => i.severity === severity);
      const issue = issues[issueIndex];

      if (!issue) {
        return;
      }

      const occurrences = issue.occurrences || [issue];
      const occurrence = occurrences[occIndex];

      if (!occurrence?.element) {
        return;
      }

      // Clear any previous hover highlight.
      this.clearHoverHighlight();

      // Add hover highlight class based on severity.
      const hoverClass = severity === 'critical' ? 'cv-hl-hover--critical' : 'cv-hl-hover';
      occurrence.element.classList.add(hoverClass);
      this._hoverHighlightedElement = occurrence.element;
      this._hoverHighlightClass = hoverClass;
    }

    clearHoverHighlight() {
      if (this._hoverHighlightedElement && this._hoverHighlightClass) {
        this._hoverHighlightedElement.classList.remove(this._hoverHighlightClass);
        this._hoverHighlightedElement = null;
        this._hoverHighlightClass = null;
      }
      // Also clear any stray hover highlights.
      document.querySelectorAll('.cv-hl-hover, .cv-hl-hover--critical').forEach(el => {
        el.classList.remove('cv-hl-hover', 'cv-hl-hover--critical');
      });
    }

    toggleOccurrences(issueKey) {
      this.state.expandedSections[issueKey] = !this.state.expandedSections[issueKey];
      this.saveState();
      this.updateContent();
    }

    expandAllIssues() {
      const { issues } = this.analyzer;
      const grouped = {
        critical: issues.filter(i => i.severity === 'critical'),
        warning: issues.filter(i => i.severity === 'warning'),
        info: issues.filter(i => i.severity === 'info')
      };

      Object.entries(grouped).forEach(([severity, items]) => {
        items.forEach((issue, index) => {
          if (issue.occurrences && issue.occurrences.length > 1) {
            this.state.expandedSections[`issue-${severity}-${index}`] = true;
          }
        });
      });

      this.saveState();
      this.updateContent();
    }

    collapseAllIssues() {
      Object.keys(this.state.expandedSections).forEach(key => {
        if (key.startsWith('issue-')) {
          this.state.expandedSections[key] = false;
        }
      });
      this.saveState();
      this.updateContent();
    }

    locateIssueOccurrence(severity, issueIndex, occIndex) {
      const issues = this.analyzer.issues.filter(i => i.severity === severity);
      const issue = issues[issueIndex];

      if (!issue) {
        return;
      }

      const occurrences = issue.occurrences || [issue];
      const occurrence = occurrences[occIndex];

      if (occurrence?.element) {
        this.focusOnElement(occurrence.element);
      }
    }

    showOccurrenceDetails(severity, issueIndex, occIndex) {
      const issues = this.analyzer.issues.filter(i => i.severity === severity);
      const issue = issues[issueIndex];

      if (!issue) {
        return;
      }

      const occurrences = issue.occurrences || [issue];
      const occurrence = occurrences[occIndex];

      if (!occurrence) {
        return;
      }

      // Show details in a modal/popup.
      this.showDetailsModal(occurrence);
    }

    showDetailsModal(occurrence) {
      // Remove existing modal.
      const existing = document.querySelector('.cv-modal');
      if (existing) {
        existing.remove();
      }

      // Check if this is a critical element and build bubble chain.
      const isCritical = occurrence.maxAge === CONFIG.maxAge.uncacheable;
      let bubbleChainHtml = '';
      let rootCauseHtml = '';

      if (isCritical && occurrence.element) {
        const bubbleChain = this.analyzer.buildBubbleChain(occurrence.element);
        const rootCause = this.analyzer.findRootCause(occurrence.element);

        if (bubbleChain.length > 1) {
          bubbleChainHtml = this.renderBubbleChain(bubbleChain);
        }

        if (rootCause) {
          rootCauseHtml = this.renderRootCause(rootCause);
        }
      }

      const modal = document.createElement('div');
      modal.className = 'cv-modal';
      modal.innerHTML = `
        <div class="cv-modal__backdrop" data-action="close-modal"></div>
        <div class="cv-modal__content">
          <div class="cv-modal__header">
            <span class="cv-modal__title">Element Details</span>
            <button class="cv-btn cv-btn--icon" data-action="close-modal">×</button>
          </div>
          <div class="cv-modal__body">
            <div class="cv-modal__selector">${utils.escapeHtml(occurrence.selector)}</div>
            ${this.renderOccurrenceDetails(occurrence)}

            ${rootCauseHtml}
            ${bubbleChainHtml}

            <div class="cv-detail-section">
              <div class="cv-detail-section__title">All Contexts</div>
              <div class="cv-detail-tags cv-detail-tags--wrap">
                ${(occurrence.contexts || []).map(c => `<span class="cv-detail-tag ${c.startsWith('user') ? 'cv-detail-tag--warning' : ''}">${utils.escapeHtml(c)}</span>`).join('')}
              </div>
            </div>

            <div class="cv-detail-section">
              <div class="cv-detail-section__title">All Tags</div>
              <div class="cv-detail-tags cv-detail-tags--wrap">
                ${(occurrence.tags || []).map(t => `<span class="cv-detail-tag">${utils.escapeHtml(t)}</span>`).join('')}
              </div>
            </div>

            <div class="cv-detail-section">
              <div class="cv-detail-section__title">Raw Data</div>
              <pre class="cv-detail-raw">${utils.escapeHtml(JSON.stringify(occurrence.data, null, 2))}</pre>
            </div>
          </div>
          <div class="cv-modal__footer">
            <button class="cv-btn" data-action="locate-from-modal">Locate Element</button>
            <button class="cv-btn cv-btn--secondary" data-action="close-modal">Close</button>
          </div>
        </div>
      `;

      // Store occurrence reference for locate action.
      modal._occurrence = occurrence;
      modal._analyzer = this.analyzer;

      modal.addEventListener('click', (e) => {
        const action = e.target.closest('[data-action]')?.dataset.action;
        if (action === 'close-modal') {
          modal.remove();
        }
        else if (action === 'locate-from-modal') {
          if (modal._occurrence?.element) {
            this.focusOnElement(modal._occurrence.element);
          }
          modal.remove();
        }
        else if (action === 'locate-chain-element') {
          const index = parseInt(e.target.closest('[data-chain-index]').dataset.chainIndex, 10);
          const bubbleChain = modal._analyzer.buildBubbleChain(modal._occurrence.element);
          if (bubbleChain[index]?.element) {
            this.focusOnElement(bubbleChain[index].element);
          }
        }
      });

      document.body.appendChild(modal);
    }

    renderRootCause(rootCause) {
      return `
        <div class="cv-detail-section cv-root-cause">
          <div class="cv-detail-section__title">
            <span class="cv-root-cause__icon">⚠</span>
            Root Cause
          </div>
          <div class="cv-root-cause__content">
            <div class="cv-root-cause__reason">${utils.escapeHtml(rootCause.reason)}</div>
            <div class="cv-root-cause__selector">${utils.escapeHtml(rootCause.selector)}</div>
            ${rootCause.totalDescendants > 1 ? `
              <div class="cv-root-cause__note">
                Found ${rootCause.totalDescendants} uncacheable descendants
              </div>
            ` : ''}
          </div>
        </div>
      `;
    }

    renderBubbleChain(chain) {
      if (chain.length === 0) {
        return '';
      }

      const chainItems = chain.map((item, index) => {
        const isSource = item.role === 'source';
        const roleIcon = isSource ? '🔴' : '↑';
        const roleLabel = isSource ? 'Source' : 'Inherited';
        const roleClass = isSource ? 'cv-chain__item--source' : 'cv-chain__item--inherited';

        return `
          <div class="cv-chain__item ${roleClass}" data-chain-index="${index}">
            <div class="cv-chain__connector">
              ${index < chain.length - 1 ? '<div class="cv-chain__line"></div>' : ''}
            </div>
            <div class="cv-chain__node">
              <span class="cv-chain__icon">${roleIcon}</span>
            </div>
            <div class="cv-chain__content">
              <div class="cv-chain__header">
                <span class="cv-chain__selector">${utils.escapeHtml(item.selector)}</span>
                <span class="cv-chain__role">${roleLabel}</span>
              </div>
              <div class="cv-chain__meta">
                <span>pre: ${utils.formatMaxAge(item.preBubblingMaxAge).text}</span>
                <span>→</span>
                <span>final: ${utils.formatMaxAge(item.finalMaxAge).text}</span>
              </div>
              <button class="cv-btn cv-btn--tiny cv-chain__locate" data-action="locate-chain-element" data-chain-index="${index}">
                Locate
              </button>
            </div>
          </div>
        `;
      }).join('');

      return `
        <div class="cv-detail-section cv-bubble-chain">
          <div class="cv-detail-section__title">
            <span class="cv-bubble-chain__icon">🔗</span>
            Bubble Chain
            <span class="cv-bubble-chain__count">${chain.length} elements</span>
          </div>
          <div class="cv-detail-section__desc">
            Shows how max-age: 0 propagates up the DOM tree
          </div>
          <div class="cv-chain">
            ${chainItems}
          </div>
        </div>
      `;
    }

    bindKeyboardShortcuts() {
      document.addEventListener('keydown', (e) => {
        // Toggle panel.
        if (e.key.toLowerCase() === CONFIG.shortcuts.toggle.key &&
            e.ctrlKey === CONFIG.shortcuts.toggle.ctrl &&
            e.shiftKey === CONFIG.shortcuts.toggle.shift) {
          e.preventDefault();
          this.toggle();
        }

        // Toggle highlights.
        if (e.key.toLowerCase() === CONFIG.shortcuts.highlight.key &&
            e.ctrlKey === CONFIG.shortcuts.highlight.ctrl &&
            e.shiftKey === CONFIG.shortcuts.highlight.shift) {
          e.preventDefault();
          this.toggleHighlights();
        }
      });
    }

    switchView(view) {
      this.activeView = view;
      this.updateContent();
      this.updateNav();
    }

    updateContent() {
      const content = this.panel.querySelector('.cv-content');
      if (content) {
        content.innerHTML = this.renderActiveView();
      }
    }

    updateNav() {
      this.panel.querySelectorAll('.cv-nav__item').forEach(item => {
        item.classList.toggle('cv-nav__item--active', item.dataset.view === this.activeView);
      });
    }

    updateElementsList() {
      const list = this.panel.querySelector('.cv-elements__list');
      if (list) {
        list.innerHTML = this.renderFilteredElements();
      }
    }

    toggle() {
      if (this.state.isOpen) {
        this.close();
      }
      else {
        this.open();
      }
    }

    open() {
      this.state.isOpen = true;
      this.state.isMinimized = false;
      this.panel.classList.remove('cv-panel--closed', 'cv-panel--minimized');
      this.saveState();
    }

    close() {
      this.state.isOpen = false;
      this.panel.classList.add('cv-panel--closed');
      this.saveState();
    }

    toggleMinimize() {
      this.state.isMinimized = !this.state.isMinimized;
      this.panel.classList.toggle('cv-panel--minimized', this.state.isMinimized);
      this.saveState();
    }

    toggleHighlights(force) {
      this.highlightsActive = force !== undefined ? force : !this.highlightsActive;

      if (this.highlightsActive) {
        this.showHighlights();
      }
      else {
        this.clearHighlights();
      }

      // Update button state.
      const btn = this.panel.querySelector('[data-action="highlight"] .cv-icon');
      if (btn) {
        btn.textContent = this.highlightsActive ? '◉' : '○';
      }

      this.state.autoHighlight = this.highlightsActive;
      this.saveState();
    }

    showHighlights() {
      this.analyzer.elements.forEach(item => {
        if (!item.element) {
          return;
        }

        const maxAge = this.analyzer.getEffectiveMaxAge(item);
        let highlightClass = 'cv-hl--permanent';

        if (maxAge === CONFIG.maxAge.uncacheable) {
          highlightClass = 'cv-hl--critical';
        }
        else if (maxAge !== CONFIG.maxAge.permanent) {
          highlightClass = 'cv-hl--limited';
        }

        // Check for user/session contexts.
        const contexts = this.analyzer.getAllContexts(item);
        if (contexts.some(c => c === 'user' || c.startsWith('user.') || c === 'session' || c.startsWith('session.'))) {
          highlightClass = 'cv-hl--warning';
        }

        // Critical takes precedence.
        if (maxAge === CONFIG.maxAge.uncacheable) {
          highlightClass = 'cv-hl--critical';
        }

        item.element.classList.add('cv-hl', highlightClass);
        item.element.addEventListener('mouseenter', this._boundHandleElementHover);
        item.element.addEventListener('mouseleave', this._boundHandleElementLeave);
      });
    }

    clearHighlights() {
      // Remove event listeners to prevent memory leaks.
      this.analyzer.elements.forEach(item => {
        if (item.element) {
          item.element.removeEventListener('mouseenter', this._boundHandleElementHover);
          item.element.removeEventListener('mouseleave', this._boundHandleElementLeave);
        }
      });

      document.querySelectorAll('.cv-hl').forEach(el => {
        // Preserve critical always-on highlights.
        if (el.classList.contains('cv-hl--always')) {
          el.classList.remove('cv-hl--warning', 'cv-hl--limited', 'cv-hl--permanent', 'cv-hl--focus');
        }
        else {
          el.classList.remove('cv-hl', 'cv-hl--critical', 'cv-hl--warning', 'cv-hl--limited', 'cv-hl--permanent', 'cv-hl--focus');
        }
      });
    }

    handleElementHover(e) {
      const element = e.currentTarget;
      const item = this.analyzer.elements.find(i => i.element === element);
      if (!item) {
        return;
      }

      const maxAge = this.analyzer.getEffectiveMaxAge(item);
      const maxAgeFormatted = utils.formatMaxAge(maxAge);
      const contexts = this.analyzer.getAllContexts(item);
      const tags = utils.toArray(item.data.final?.tags);

      this.tooltip.innerHTML = `
        <div class="cv-tooltip__header">
          <span class="cv-tooltip__maxage cv-tooltip__maxage--${maxAgeFormatted.class}">${maxAgeFormatted.text}</span>
        </div>
        <div class="cv-tooltip__row">
          <span class="cv-tooltip__label">Contexts</span>
          <span class="cv-tooltip__value">${contexts.length}</span>
        </div>
        <div class="cv-tooltip__row">
          <span class="cv-tooltip__label">Tags</span>
          <span class="cv-tooltip__value">${tags.length}</span>
        </div>
        ${contexts.length > 0 ? `
          <div class="cv-tooltip__tags">
            ${contexts.slice(0, 4).map(c => `<span class="cv-tooltip__tag">${utils.escapeHtml(c)}</span>`).join('')}
            ${contexts.length > 4 ? `<span class="cv-tooltip__more">+${contexts.length - 4}</span>` : ''}
          </div>
        ` : ''}
      `;

      this.tooltip.style.display = 'block';
      this.positionTooltip(e);
    }

    handleElementLeave() {
      this.tooltip.style.display = 'none';
    }

    positionTooltip(e) {
      const offset = 12;
      let left = e.clientX + offset;
      let top = e.clientY + offset;

      this.tooltip.style.left = left + 'px';
      this.tooltip.style.top = top + 'px';

      // Adjust if off-screen.
      const rect = this.tooltip.getBoundingClientRect();
      if (rect.right > window.innerWidth) {
        this.tooltip.style.left = (e.clientX - rect.width - offset) + 'px';
      }
      if (rect.bottom > window.innerHeight) {
        this.tooltip.style.top = (e.clientY - rect.height - offset) + 'px';
      }
    }

    focusElement(index) {
      let filtered = this.analyzer.elements;
      if (this.searchQuery) {
        const query = this.searchQuery.toLowerCase();
        filtered = filtered.filter(item => {
          const selector = utils.getSelector(item.element).toLowerCase();
          return selector.includes(query);
        });
      }

      const item = filtered[index];
      if (item?.element) {
        this.focusOnElement(item.element);
      }
    }

    focusOnElement(element) {
      // Clear previous focus.
      document.querySelectorAll('.cv-hl--focus').forEach(el => {
        el.classList.remove('cv-hl--focus');
      });

      // Add highlight if not already highlighted.
      if (!element.classList.contains('cv-hl')) {
        element.classList.add('cv-hl', 'cv-hl--warning');
      }

      element.classList.add('cv-hl--focus');
      element.scrollIntoView({ behavior: 'smooth', block: 'center' });

      // Remove focus after animation.
      setTimeout(() => {
        element.classList.remove('cv-hl--focus');
      }, 2000);
    }

    /**
     * Expose API for console usage.
     */
    exposeAPI() {
      window.cacheviz = {
        elements: this.analyzer.elements,
        issues: this.analyzer.issues,
        stats: this.analyzer.stats,
        highlight: () => this.toggleHighlights(true),
        clear: () => this.toggleHighlights(false),
        query: (type, value) => {
          return this.analyzer.elements.filter(item => {
            const data = item.data.final || {};
            if (type === 'maxAge') {
              return utils.parseMaxAge(data['max-age']) === value;
            }
            if (type === 'context') {
              return this.analyzer.getAllContexts(item).some(c => c.includes(value));
            }
            if (type === 'tag') {
              return utils.toArray(data.tags).some(t => t.includes(value));
            }
            return false;
          });
        }
      };
    }
  }

  /* =========================================================================
     INITIALIZATION
     ========================================================================= */

  function init() {
    // Parse elements from HTML comments.
    if (typeof window.CachevizComments === 'undefined') {
      console.warn('Cacheviz: Comment parser not loaded');
      return;
    }

    const elements = window.CachevizComments.parse();

    // Get page-level cache info.
    let pageInfo = { maxAge: -1, contexts: [], tags: [] };
    const settingsEl = document.querySelector('[data-drupal-selector="drupal-settings-json-cacheviz"]');
    if (settingsEl) {
      try {
        const data = JSON.parse(settingsEl.textContent);
        pageInfo = data.cacheviz?.page || pageInfo;
      }
      catch (e) {
        // Ignore.
      }
    }

    // Analyze and create UI.
    const analyzer = new CacheAnalyzer(elements);
    const ui = new CachevizUI(analyzer, pageInfo);
    ui.exposeAPI();

    // Log summary.
    console.log(
      '%c◉ Cacheviz %c' + analyzer.stats.total + ' elements | ' +
      analyzer.issues.filter(i => i.severity === 'critical').length + ' critical | ' +
      analyzer.issues.filter(i => i.severity === 'warning').length + ' warnings',
      'color: #3b82f6; font-weight: bold;',
      'color: #888;'
    );
  }

  // Run when DOM is ready.
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  }
  else {
    init();
  }

})(window, document);
