<?php

declare(strict_types=1);

namespace Drupal\Tests\cacheviz\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests for the Cacheviz settings form.
 *
 * @coversDefaultClass \Drupal\cacheviz\Form\SettingsForm
 * @group cacheviz
 */
class SettingsFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['cacheviz'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests that the settings page is accessible with permission.
   */
  public function testSettingsPageAccess(): void {
    // Test anonymous user cannot access.
    $this->drupalGet('admin/config/development/cacheviz');
    $this->assertSession()->statusCodeEquals(403);

    // Test user with permission can access.
    $admin = $this->drupalCreateUser(['administer cacheviz']);
    $this->drupalLogin($admin);
    $this->drupalGet('admin/config/development/cacheviz');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Cacheviz Settings');
  }

  /**
   * Tests that the settings form can be submitted.
   */
  public function testSettingsFormSubmission(): void {
    $admin = $this->drupalCreateUser(['administer cacheviz']);
    $this->drupalLogin($admin);

    $this->drupalGet('admin/config/development/cacheviz');

    // Verify form fields are present.
    $this->assertSession()->fieldExists('enabled');
    $this->assertSession()->fieldExists('auto_highlight_problems');
    $this->assertSession()->fieldExists('excluded_paths');

    // Submit the form with new values.
    $this->submitForm([
      'enabled' => TRUE,
      'auto_highlight_problems' => FALSE,
      'excluded_paths' => "/admin/*\n/user/*\n/custom/path",
    ], 'Save configuration');

    // Verify the configuration was saved.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    $config = $this->config('cacheviz.settings');
    $this->assertTrue($config->get('enabled'));
    $this->assertFalse($config->get('auto_highlight_problems'));
    $this->assertStringContainsString('/custom/path', $config->get('excluded_paths'));
  }

  /**
   * Tests the menu link appears in development configuration.
   */
  public function testMenuLinkExists(): void {
    $admin = $this->drupalCreateUser([
      'administer cacheviz',
      'access administration pages',
      'administer site configuration',
    ]);
    $this->drupalLogin($admin);

    $this->drupalGet('admin/config/development');
    $this->assertSession()->linkExists('Cacheviz');
  }

  /**
   * Tests that enabling/disabling works correctly.
   */
  public function testEnableDisableToggle(): void {
    $admin = $this->drupalCreateUser(['administer cacheviz']);
    $this->drupalLogin($admin);

    // First, disable cacheviz.
    $this->drupalGet('admin/config/development/cacheviz');
    $this->submitForm(['enabled' => FALSE], 'Save configuration');

    $config = $this->config('cacheviz.settings');
    $this->assertFalse($config->get('enabled'));

    // Then, enable cacheviz.
    $this->drupalGet('admin/config/development/cacheviz');
    $this->submitForm(['enabled' => TRUE], 'Save configuration');

    $config = $this->config('cacheviz.settings');
    $this->assertTrue($config->get('enabled'));
  }

}
