<?php

declare(strict_types=1);

namespace Drupal\Tests\cacheviz\Unit;

use Drupal\cacheviz\PathMatcher;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for the PathMatcher service.
 *
 * @coversDefaultClass \Drupal\cacheviz\PathMatcher
 * @group cacheviz
 */
class PathMatcherTest extends UnitTestCase {

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The current path stack mock.
   *
   * @var \Drupal\Core\Path\CurrentPathStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentPath;

  /**
   * The core path matcher mock.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $corePathMatcher;

  /**
   * The config mock.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->currentPath = $this->createMock(CurrentPathStack::class);
    $this->corePathMatcher = $this->createMock(PathMatcherInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);

    $this->configFactory->method('get')
      ->with('cacheviz.settings')
      ->willReturn($this->config);
  }

  /**
   * Tests isPathExcluded() with empty exclusion patterns.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWithEmptyPatterns(): void {
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn('');

    $pathMatcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertFalse($pathMatcher->isPathExcluded('/any/path'));
  }

  /**
   * Tests isPathExcluded() with null exclusion patterns.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWithNullPatterns(): void {
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn(NULL);

    $pathMatcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertFalse($pathMatcher->isPathExcluded('/any/path'));
  }

  /**
   * Tests isPathExcluded() when path matches exclusion pattern.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWhenMatched(): void {
    $excludedPaths = "/admin\n/admin/*\n/user/login";
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn($excludedPaths);

    $this->corePathMatcher->method('matchPath')
      ->with('/admin/config', $excludedPaths)
      ->willReturn(TRUE);

    $pathMatcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertTrue($pathMatcher->isPathExcluded('/admin/config'));
  }

  /**
   * Tests isPathExcluded() when path does not match exclusion pattern.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWhenNotMatched(): void {
    $excludedPaths = "/admin\n/admin/*";
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn($excludedPaths);

    $this->corePathMatcher->method('matchPath')
      ->with('/node/1', $excludedPaths)
      ->willReturn(FALSE);

    $pathMatcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertFalse($pathMatcher->isPathExcluded('/node/1'));
  }

  /**
   * Tests isCurrentPathExcluded() delegates to isPathExcluded().
   *
   * @covers ::isCurrentPathExcluded
   */
  public function testIsCurrentPathExcluded(): void {
    $excludedPaths = "/admin/*";
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn($excludedPaths);

    $this->currentPath->method('getPath')
      ->willReturn('/admin/structure');

    $this->corePathMatcher->method('matchPath')
      ->with('/admin/structure', $excludedPaths)
      ->willReturn(TRUE);

    $pathMatcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertTrue($pathMatcher->isCurrentPathExcluded());
  }

}
