<?php

declare(strict_types=1);

namespace Drupal\cacheviz\EventSubscriber;

use Drupal\cacheviz\PathMatcher;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Injects page-level cache information into drupalSettings.
 */
final readonly class ResponseSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a ResponseSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\cacheviz\PathMatcher $pathMatcher
   *   The path matcher service.
   * @param \Drupal\Core\Extension\ExtensionPathResolver $extensionPathResolver
   *   The extension path resolver.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file URL generator.
   * @param string $appRoot
   *   The app root path.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected AccountProxyInterface $currentUser,
    protected PathMatcher $pathMatcher,
    protected ExtensionPathResolver $extensionPathResolver,
    protected FileUrlGeneratorInterface $fileUrlGenerator,
    protected string $appRoot,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run after most other response subscribers but before compression.
    return [
      KernelEvents::RESPONSE => ['onResponse', -100],
    ];
  }

  /**
   * Injects cache info and library attachment into the response.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    if (!$this->shouldProcess()) {
      return;
    }

    $response = $event->getResponse();
    $content = $response->getContent();

    if ($content === FALSE || $content === '') {
      return;
    }

    // Only process HTML responses.
    $content_type = $response->headers->get('Content-Type', '');
    if (!str_contains($content_type, 'text/html')) {
      return;
    }

    // Get page-level cache metadata from response headers.
    $cache_tags = $response->headers->get('X-Drupal-Cache-Tags', '');
    $cache_contexts = $response->headers->get('X-Drupal-Cache-Contexts', '');
    $cache_max_age_header = $response->headers->get('X-Drupal-Cache-Max-Age', '');

    // Parse max-age - header may contain "0 (Uncacheable)" or just a number.
    $cache_max_age = -1;
    if ($cache_max_age_header !== '') {
      // Extract numeric part from header like "0 (Uncacheable)".
      if (preg_match('/^(-?\d+)/', $cache_max_age_header, $matches)) {
        $cache_max_age = (int) $matches[1];
      }
    }

    $config = $this->configFactory->get('cacheviz.settings');
    $auto_highlight = $config->get('auto_highlight_problems') ?? TRUE;

    // Build the JavaScript settings.
    $settings = [
      'cacheviz' => [
        'page' => [
          'tags' => $cache_tags !== '' ? explode(' ', $cache_tags) : [],
          'contexts' => $cache_contexts !== '' ? explode(' ', $cache_contexts) : [],
          'maxAge' => $cache_max_age,
        ],
        'autoHighlight' => (bool) $auto_highlight,
      ],
    ];

    // Inject drupalSettings if not already present.
    $settings_json = json_encode($settings, JSON_THROW_ON_ERROR);
    $script = '<script type="application/json" data-drupal-selector="drupal-settings-json-cacheviz">'
      . $settings_json . '</script>';

    // Attach CSS and JS with proper URL generation and cache busting.
    $module_path = $this->extensionPathResolver->getPath('module', 'cacheviz');
    $assets = [
      'css' => [
        $module_path . '/css/cacheviz.panel.css',
        $module_path . '/css/cacheviz.highlight.css',
      ],
      'js' => [
        $module_path . '/js/cacheviz.comments.js',
        $module_path . '/js/cacheviz.js',
      ],
    ];

    $library_link = '';
    foreach ($assets['css'] as $css_path) {
      $url = $this->generateAssetUrl($css_path);
      $library_link .= '<link rel="stylesheet" media="all" href="' . $url . '" />';
    }

    $library_script = '';
    foreach ($assets['js'] as $js_path) {
      $url = $this->generateAssetUrl($js_path);
      $library_script .= '<script src="' . $url . '"></script>';
    }

    // Inject before closing </head> tag.
    if (str_contains($content, '</head>')) {
      $content = str_replace('</head>', $script . $library_link . '</head>', $content);
    }

    // Inject before closing </body> tag.
    if (str_contains($content, '</body>')) {
      $content = str_replace('</body>', $library_script . '</body>', $content);
    }

    $response->setContent($content);
  }

  /**
   * Checks if the response should be processed.
   *
   * @return bool
   *   TRUE if processing should occur, FALSE otherwise.
   */
  protected function shouldProcess(): bool {
    $config = $this->configFactory->get('cacheviz.settings');
    if (!$config->get('enabled')) {
      return FALSE;
    }

    if (!$this->currentUser->hasPermission('view cacheviz debug')) {
      return FALSE;
    }

    if ($this->pathMatcher->isCurrentPathExcluded()) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Generates an asset URL with cache busting query string.
   *
   * @param string $relative_path
   *   The relative path to the asset file.
   *
   * @return string
   *   The absolute URL with cache busting query parameter.
   */
  protected function generateAssetUrl(string $relative_path): string {
    // Generate the base URL using Drupal's file URL generator.
    $url = $this->fileUrlGenerator->generateString($relative_path);

    // Add cache busting based on file modification time.
    $full_path = $this->appRoot . '/' . $relative_path;
    if (file_exists($full_path)) {
      $mtime = filemtime($full_path);
      $url .= '?v=' . $mtime;
    }

    return $url;
  }

}
