<?php

declare(strict_types=1);

namespace Drupal\Tests\cacheviz\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests for the Cacheviz response subscriber.
 *
 * @coversDefaultClass \Drupal\cacheviz\EventSubscriber\ResponseSubscriber
 * @group cacheviz
 */
final class ResponseSubscriberTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['cacheviz', 'node'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests that CSS and JS are not injected for anonymous users.
   */
  public function testAssetsNotInjectedForAnonymous(): void {
    // Enable cacheviz.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->save();

    $this->drupalGet('<front>');

    // Anonymous user should not see cacheviz assets.
    $this->assertSession()->responseNotContains('cacheviz.panel.css');
    $this->assertSession()->responseNotContains('cacheviz.js');
    $this->assertSession()->responseNotContains('drupal-settings-json-cacheviz');
  }

  /**
   * Tests that CSS and JS are injected for users with permission.
   */
  public function testAssetsInjectedForAuthorizedUser(): void {
    // Enable cacheviz.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->save();

    // Create and login user with permission.
    $user = $this->drupalCreateUser(['view cacheviz debug']);
    $this->drupalLogin($user);

    $this->drupalGet('<front>');

    // Authorized user should see cacheviz assets.
    $this->assertSession()->responseContains('cacheviz.panel.css');
    $this->assertSession()->responseContains('cacheviz.js');
    $this->assertSession()->responseContains('drupal-settings-json-cacheviz');
  }

  /**
   * Tests that assets are not injected when module is disabled.
   */
  public function testAssetsNotInjectedWhenDisabled(): void {
    // Disable cacheviz.
    $this->config('cacheviz.settings')
      ->set('enabled', FALSE)
      ->save();

    // Create and login user with permission.
    $user = $this->drupalCreateUser(['view cacheviz debug']);
    $this->drupalLogin($user);

    $this->drupalGet('<front>');

    // Assets should not be injected when disabled.
    $this->assertSession()->responseNotContains('cacheviz.panel.css');
    $this->assertSession()->responseNotContains('cacheviz.js');
  }

  /**
   * Tests that excluded paths do not get assets injected.
   */
  public function testExcludedPathsNotInjected(): void {
    // Enable cacheviz with excluded paths.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->set('excluded_paths', "/user/*\n/admin/*")
      ->save();

    // Create and login user with permission.
    $user = $this->drupalCreateUser([
      'view cacheviz debug',
      'access user profiles',
    ]);
    $this->drupalLogin($user);

    // Visit an excluded path.
    $this->drupalGet('user/' . $user->id());

    // Assets should not be on excluded paths.
    $this->assertSession()->responseNotContains('cacheviz.panel.css');
  }

  /**
   * Tests that page cache metadata is included in settings.
   */
  public function testPageCacheMetadataIncluded(): void {
    // Enable cacheviz.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->save();

    // Create and login user with permission.
    $user = $this->drupalCreateUser(['view cacheviz debug']);
    $this->drupalLogin($user);

    $this->drupalGet('<front>');

    // Check that the settings JSON contains expected structure.
    $this->assertSession()->responseContains('drupal-settings-json-cacheviz');
    $this->assertSession()->responseContains('"cacheviz"');
    $this->assertSession()->responseContains('"page"');
  }

  /**
   * Tests that auto_highlight_problems setting is passed to frontend.
   */
  public function testAutoHighlightSettingPassed(): void {
    // Enable cacheviz with auto highlight enabled.
    $this->config('cacheviz.settings')
      ->set('enabled', TRUE)
      ->set('auto_highlight_problems', TRUE)
      ->save();

    // Create and login user with permission.
    $user = $this->drupalCreateUser(['view cacheviz debug']);
    $this->drupalLogin($user);

    $this->drupalGet('<front>');

    // Check that autoHighlight is true.
    $this->assertSession()->responseContains('"autoHighlight":true');

    // Now disable auto highlight.
    $this->config('cacheviz.settings')
      ->set('auto_highlight_problems', FALSE)
      ->save();

    $this->drupalGet('<front>');

    // Check that autoHighlight is false.
    $this->assertSession()->responseContains('"autoHighlight":false');
  }

}
