<?php

declare(strict_types=1);

namespace Drupal\Tests\cacheviz\Unit;

use Drupal\cacheviz\PathMatcher;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for the PathMatcher service.
 *
 * @coversDefaultClass \Drupal\cacheviz\PathMatcher
 * @group cacheviz
 */
final class PathMatcherTest extends UnitTestCase {

  /**
   * The config factory mock.
   */
  protected ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The current path stack mock.
   */
  protected CurrentPathStack&MockObject $currentPath;

  /**
   * The core path matcher mock.
   */
  protected PathMatcherInterface&MockObject $corePathMatcher;

  /**
   * The config mock.
   */
  protected ImmutableConfig&MockObject $config;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->currentPath = $this->createMock(CurrentPathStack::class);
    $this->corePathMatcher = $this->createMock(PathMatcherInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);

    $this->configFactory->method('get')
      ->with('cacheviz.settings')
      ->willReturn($this->config);
  }

  /**
   * Tests isPathExcluded() with empty exclusion patterns.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWithEmptyPatterns(): void {
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn('');

    $path_matcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertFalse($path_matcher->isPathExcluded('/any/path'));
  }

  /**
   * Tests isPathExcluded() with null exclusion patterns.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWithNullPatterns(): void {
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn(NULL);

    $path_matcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertFalse($path_matcher->isPathExcluded('/any/path'));
  }

  /**
   * Tests isPathExcluded() when path matches exclusion pattern.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWhenMatched(): void {
    $excluded_paths = "/admin\n/admin/*\n/user/login";
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn($excluded_paths);

    $this->corePathMatcher->method('matchPath')
      ->with('/admin/config', $excluded_paths)
      ->willReturn(TRUE);

    $path_matcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertTrue($path_matcher->isPathExcluded('/admin/config'));
  }

  /**
   * Tests isPathExcluded() when path does not match exclusion pattern.
   *
   * @covers ::isPathExcluded
   */
  public function testIsPathExcludedWhenNotMatched(): void {
    $excluded_paths = "/admin\n/admin/*";
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn($excluded_paths);

    $this->corePathMatcher->method('matchPath')
      ->with('/node/1', $excluded_paths)
      ->willReturn(FALSE);

    $path_matcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertFalse($path_matcher->isPathExcluded('/node/1'));
  }

  /**
   * Tests isCurrentPathExcluded() delegates to isPathExcluded().
   *
   * @covers ::isCurrentPathExcluded
   */
  public function testIsCurrentPathExcluded(): void {
    $excluded_paths = "/admin/*";
    $this->config->method('get')
      ->with('excluded_paths')
      ->willReturn($excluded_paths);

    $this->currentPath->method('getPath')
      ->willReturn('/admin/structure');

    $this->corePathMatcher->method('matchPath')
      ->with('/admin/structure', $excluded_paths)
      ->willReturn(TRUE);

    $path_matcher = new PathMatcher(
      $this->configFactory,
      $this->currentPath,
      $this->corePathMatcher
    );

    $this->assertTrue($path_matcher->isCurrentPathExcluded());
  }

}
