<?php

declare(strict_types=1);

namespace Drupal\cacheviz;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Path\PathMatcherInterface;

/**
 * Service to check if the current path should be excluded from cacheviz.
 */
final readonly class PathMatcher {

  /**
   * Constructs a PathMatcher object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Path\CurrentPathStack $currentPath
   *   The current path stack.
   * @param \Drupal\Core\Path\PathMatcherInterface $pathMatcher
   *   The core path matcher.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected CurrentPathStack $currentPath,
    protected PathMatcherInterface $pathMatcher,
  ) {}

  /**
   * Checks if the current path is excluded from visualization.
   *
   * @return bool
   *   TRUE if the current path is excluded, FALSE otherwise.
   */
  public function isCurrentPathExcluded(): bool {
    return $this->isPathExcluded($this->currentPath->getPath());
  }

  /**
   * Checks if a given path is excluded from visualization.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if the path is excluded, FALSE otherwise.
   */
  public function isPathExcluded(string $path): bool {
    $config = $this->configFactory->get('cacheviz.settings');
    $excluded_paths = $config->get('excluded_paths') ?? '';

    if ($excluded_paths === '') {
      return FALSE;
    }

    return $this->pathMatcher->matchPath($path, $excluded_paths);
  }

}
