<?php

namespace Drupal\calculator_field\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'calculator_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "calculator_formatter",
 *   label = @Translation("Calculator Formatter"),
 *   field_types = {"calculator_field"}
 * )
 */
class CalculatorFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * Service to handle token extraction and replacement.
   *
   * @var \Drupal\calculator_field\Service\TokenExtractor
   */
  protected $tokenExtractor;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    /** @var static $instance */
    $instance = parent::create(
      $container,
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
    $instance->tokenExtractor = $container->get('calculator_field.token_extractor');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_formula' => FALSE,
      'show_formula_values' => FALSE,
      'custom_classes' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['show_formula'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show formula'),
      '#default_value' => $this->getSetting('show_formula'),
      '#description' => $this->t('Display the formula expression (as entered) alongside the result.'),
    ];
    $form['show_formula_values'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show formula with values'),
      '#default_value' => $this->getSetting('show_formula_values'),
      '#description' => $this->t('Display the formula with field values during view (e.g. "10 * 2").'),
    ];
    $form['custom_classes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom CSS classes'),
      '#default_value' => $this->getSetting('custom_classes'),
      '#description' => $this->t('Additional CSS classes to add to the output container.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->getSetting('show_formula') ? $this->t('Shows formula: Yes') : $this->t('Shows formula: No');
    $summary[] = $this->getSetting('show_formula_values') ? $this->t('Shows values in formula: Yes') : $this->t('Shows values in formula: No');
    if ($this->getSetting('custom_classes')) {
      $summary[] = $this->t('Custom classes: @c', ['@c' => $this->getSetting('custom_classes')]);
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   *
   * Build elements for rendering calculator field values, computing on the fly.
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $entity = $items->getEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    // Read the formula from the widget settings for this field.
    $field_name = $this->fieldDefinition->getName();
    $form_display = \Drupal::service('entity_display.repository')
      ->getFormDisplay($entity_type_id, $bundle, 'default');
    $component = $form_display ? $form_display->getComponent($field_name) : NULL;
    $formula = ($component['settings']['formula'] ?? '') ?: '';

    $show_formula = $this->getSetting('show_formula');
    $show_formula_values = $this->getSetting('show_formula_values');
    $custom_classes = $this->getSetting('custom_classes');

    if ($formula) {
      // Replace tokens with runtime values from the entity.
      $calc_string = $this->tokenExtractor->replaceTokensWithValues($formula, $entity);
      // Sanitize to allow only digits, operators, parentheses, dot and whitespace.
      $safe = preg_replace('/[^0-9+\-*/().\s]/', '', $calc_string);
    }

    foreach ($items as $delta => $item) {
      $value = $item->value;
      if (!empty($safe)) {
        try {
          // phpcs:ignore DrupalPractice.CodeAnalysis.EvalUse
          // The use of eval() is discouraged. Here, input is validated to contain
          // only math characters. Consider using a proper math parser for greater
          // safety in production.
          $computed = eval('return ' . $safe . ';');
          if (is_numeric($computed)) {
            $value = $computed;
          }
        }
        catch (\Throwable $e) {
          // Fallback to stored value on error.
        }
      }

      // Build nice formula/field value output.
      $out = [];
      if ($show_formula) {
        $out[] = [
          '#type' => 'inline_template',
          '#template' => '<div class="calculator-formula">{% trans %}Formula:{% endtrans %} <code>{{ formula }}</code></div>',
          '#context' => ['formula' => $formula],
        ];
      }
      if ($show_formula_values && !empty($formula)) {
        $eval_display = $formula;
        preg_match_all('/\[([^\]]+)\]/', $formula, $matches);
        if (!empty($matches[1])) {
          foreach ($matches[1] as $fname) {
            $val = '';
            if ($entity->hasField($fname) && !$entity->get($fname)->isEmpty()) {
              $val = $entity->get($fname)->value;
            }
            $eval_display = str_replace('[' . $fname . ']', $val, $eval_display);
          }
        }
        $out[] = [
          '#type' => 'inline_template',
          '#template' => '<div class="calculator-formula-values">Evaluated: <code>{{ evaluated }}</code></div>',
          '#context' => ['evaluated' => $eval_display],
        ];
        // List all tokens and actual values.
        preg_match_all('/\[([^\]]+)\]/', $formula, $matches);
        if (!empty($matches[1])) {
          $field_definitions = $entity->getFieldDefinitions();
          $list = [];
          foreach (array_unique($matches[1]) as $fname) {
            $val = '';
            if ($entity->hasField($fname) && !$entity->get($fname)->isEmpty()) {
              $val = $entity->get($fname)->value;
            }
            $label = isset($field_definitions[$fname]) ? $field_definitions[$fname]->getLabel() : $fname;
            $list[] = '<li><strong>' . htmlspecialchars($label) . '</strong>: ' . htmlspecialchars($val) . '</li>';
          }
          $out[] = [
            '#markup' => '<ul class="calculator-formula-field-list">' . implode('', $list) . '</ul>',
          ];
        }
      }
      // Result itself.
      $out[] = [
        '#markup' => '<div class="calculator-result"><strong>'
        . $this->t('Result:') . '</strong> ' . number_format((float) $value, 2) . '</div>',
      ];

      $elements[$delta] = [
        '#type' => 'container',
        '#attributes' => ['class' => array_filter(['calculator-field-output', $custom_classes])],
        'content' => $out,
      ];
    }
    return $elements;
  }

}
