<?php

namespace Drupal\calculator_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'calculator_widget' widget.
 *
 * @FieldWidget(
 *   id = "calculator_widget",
 *   label = @Translation("Calculator Widget"),
 *   field_types = {
 *     "calculator_field"
 *   }
 * )
 */
class CalculatorWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * Token extractor service.
   *
   * @var \Drupal\calculator_field\Service\TokenExtractor
   */
  protected $tokenExtractor;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->tokenExtractor = $container->get('calculator_field.token_extractor');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'formula' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    // Derive entity type and bundle robustly for settings UI.
    $entity_type_id = method_exists($this->fieldDefinition, 'getTargetEntityTypeId') ? $this->fieldDefinition->getTargetEntityTypeId() : 'node';
    $bundle = $form_state->get('bundle');
    if (!$bundle && method_exists($this->fieldDefinition, 'getTargetBundle')) {
      $bundle = $this->fieldDefinition->getTargetBundle();
    }
    $tokens = [];
    if ($bundle) {
      $tokens = $this->tokenExtractor->getFieldTokens($entity_type_id, $bundle);
    }

    $form['formula'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Formula'),
      '#description' => $this->t('Use field tokens like @tokens', [
        '@tokens' => $tokens ? implode(', ', $tokens) : $this->t('no fields found for this bundle'),
      ]),
      '#default_value' => $this->getSetting('formula'),
      '#rows' => 2,
      '#attributes' => ['class' => ['calculator-formula-input']],
    ];

    // Add clickable token links to help build the formula without creating form values.
    if (!empty($tokens)) {
      $links = [];
      foreach ($tokens as $token) {
        $links[] = '<a href="#" class="calculator-token-link" data-token="' . $token . '">' . $token . '</a>';
      }
      $form['token_selector'] = [
        '#type' => 'item',
        '#title' => $this->t('Insert token'),
        '#markup' => '<div class="calculator-token-list">' . implode(' ', $links) . '</div>',
        '#parents' => [],
      ];
      $form['#attached']['library'][] = 'calculator_field/calculator_token_selector';
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Formula: @formula', ['@formula' => $this->getSetting('formula') ?: 'None']);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $entity = $items->getEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    $tokens = $this->tokenExtractor->getFieldTokens($entity_type_id, $bundle);
    $formula = $this->getSetting('formula') ?: '';

    $element['value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Calculated Result'),
      '#default_value' => $items[$delta]->value ?? '',
      '#attributes' => ['readonly' => 'readonly', 'class' => ['calculator-field-result']],
      '#attached' => [
        'library' => ['calculator_field/calculator'],
        'drupalSettings' => [
          'calculator_field' => [
            'formula' => $formula,
            'tokens' => $tokens,
          ],
        ],
      ],
    ];

    return $element;
  }

  /**
   * Load all field tokens for the current bundle.
   */
  protected function getAvailableFieldTokens($entity_type_id, $bundle) {
    return $this->tokenExtractor->getFieldTokens($entity_type_id, $bundle);
  }

}
