<?php

namespace Drupal\calculator_field\Service;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Extracts tokens and replaces placeholders with entity field values.
 */
class TokenExtractor {

  /**
   * Provides field management operations.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Entity type manager service for entity operations.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs the token extractor service.
   *
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   Entity field manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager service.
   */
  public function __construct(EntityFieldManagerInterface $entity_field_manager, EntityTypeManagerInterface $entity_type_manager) {
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Get available field tokens for a given entity and bundle.
   *
   * @param string $entity_type_id
   *   The entity type (e.g., 'node').
   * @param string $bundle
   *   The bundle (e.g., 'article').
   *
   * @return array
   *   An array of available tokens (e.g., ['[field_price]', '[field_weight]']).
   */
  public function getFieldTokens($entity_type_id, $bundle) {
    $definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);

    $tokens = [];
    foreach ($definitions as $field_name => $definition) {
      // Skip common system/base fields.
      $excluded_names = [
        'nid',
        'vid',
        'uuid',
        'uid',
        'tid',
        'rid',
        'langcode',
        'default_langcode',
        'revision_id',
        'revision_translation_affected',
        'status',
        'created',
        'changed',
        'promote',
        'sticky',
        'title',
        'name',
      ];
      if (in_array($field_name, $excluded_names, TRUE)) {
        continue;
      }

      // Only allow numeric fields, exclude this calculator field itself.
      $type = $definition->getType();
      if ($type === 'calculator_field') {
        continue;
      }

      if (in_array($type, ['integer', 'float', 'decimal'], TRUE)) {
        $tokens[] = '[' . $field_name . ']';
      }
    }

    return $tokens;
  }

  /**
   * Replace tokens in a formula with actual values from the entity.
   *
   * @param string $formula
   *   Formula string like "[field_price] * [field_quantity]".
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity instance.
   *
   * @return string
   *   Formula with numeric values substituted.
   */
  public function replaceTokensWithValues($formula, ContentEntityInterface $entity) {
    $replaced = $formula;

    preg_match_all('/\[([^\]]+)\]/', $formula, $matches);
    foreach ($matches[1] as $field_name) {
      if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
        $value = $entity->get($field_name)->value;
        $replaced = str_replace('[' . $field_name . ']', $value, $replaced);
      }
      else {
        $replaced = str_replace(
          '[' . $field_name . ']',
          0,
          $replaced
              );
      }
    }

    return $replaced;
  }

}
