<?php

declare(strict_types=1);

namespace Drupal\Tests\canvas_extjs\Kernel;

use Drupal\canvas\Entity\Component as ComponentEntity;
use Drupal\canvas\Entity\VersionedConfigEntityBase;
use Drupal\canvas_extjs\Plugin\Canvas\ComponentSource\ExternalJavaScriptComponent;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests rendering output of external JavaScript components.
 *
 * @group canvas_extjs
 */
class ComponentRenderingTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'canvas',
    'canvas_extjs',
    'custom_elements',
    'field',
    'text',
    'user',
    'options',
    'node',
    'file',
    'image',
    'media',
    'datetime',
    'link',
    'path',
    'serialization',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installConfig(['field', 'node', 'system', 'file', 'image', 'media']);

    // Register test components.
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';
    $manager->register($fixtureUrl, [
      'javascript' => ['/modules/contrib/canvas_extjs/tests/fixtures/test-components.js'],
    ]);

    // Create node type with component tree field.
    $node_type = \Drupal::entityTypeManager()
      ->getStorage('node_type')
      ->create([
        'type' => 'page',
        'name' => 'Page',
      ]);
    $node_type->save();

    // Create field storage and instance.
    $field_storage = \Drupal::entityTypeManager()
      ->getStorage('field_storage_config')
      ->create([
        'field_name' => 'field_component_tree',
        'entity_type' => 'node',
        'type' => 'component_tree',
      ]);
    $field_storage->save();

    $field = \Drupal::entityTypeManager()
      ->getStorage('field_config')
      ->create([
        'field_storage' => $field_storage,
        'bundle' => 'page',
        'label' => 'Component Tree',
      ]);
    $field->save();
  }

  /**
   * Test JavaScript variant rendering via Canvas field pipeline.
   *
   * Tests a component tree with nested slot component rendered through
   * Canvas field rendering logic. Verifies Canvas annotations in preview mode.
   */
  public function testJavaScriptVariantRendering(): void {
    $renderer = \Drupal::service('renderer');

    // Create component tree: TwoColumnLayout with TestButton in column-one.
    $buttonUuid = 'nested-button-uuid';
    $layoutUuid = 'parent-layout-uuid';

    $componentTree = [
      // TwoColumnLayout as root component.
      [
        'uuid' => $layoutUuid,
        'component_id' => 'extjs.two_column_layout',
        'component_version' => 'active',
        'inputs' => [
          'width' => [
            'sourceType' => 'static:field_item:integer',
            'value' => 66,
            'expression' => 'ℹ︎integer␟value',
          ],
        ],
      ],
      // TestButton in column-one slot.
      [
        'uuid' => $buttonUuid,
        'component_id' => 'extjs.test_button',
        'component_version' => 'active',
        'parent_uuid' => $layoutUuid,
        'slot' => 'column-one',
        'inputs' => [
          'label' => [
            'sourceType' => 'static:field_item:string',
            'value' => 'Click Me',
            'expression' => 'ℹ︎string␟value',
          ],
          'variant' => [
            'sourceType' => 'static:field_item:string',
            'value' => 'primary',
            'expression' => 'ℹ︎string␟value',
          ],
        ],
      ],
    ];

    $node = \Drupal::entityTypeManager()
      ->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => 'Test Page',
        'field_component_tree' => $componentTree,
      ]);
    $node->save();

    // Render in preview mode via Canvas field pipeline.
    /** @var \Drupal\canvas\Plugin\Field\FieldType\ComponentTreeItemList $field_items */
    $field_items = $node->get('field_component_tree');
    $renderable = $field_items->toRenderable($node, TRUE);
    $html = (string) $renderer->renderInIsolation($renderable);

    // Verify layout component wrapper comments.
    $this->assertStringContainsString('<!-- canvas-start-' . $layoutUuid . ' -->', $html);
    $this->assertStringContainsString('<!-- canvas-end-' . $layoutUuid . ' -->', $html);

    // Extract slot data.
    preg_match('/data-component-slots="([^"]*)"/', $html, $matches);
    $this->assertNotEmpty($matches);
    $slotDataJson = html_entity_decode($matches[1], ENT_QUOTES | ENT_HTML5);
    $slotData = json_decode($slotDataJson, TRUE);
    $this->assertArrayHasKey('column-one', $slotData);

    // Verify slot marker comments wrap slot content.
    $this->assertStringContainsString('<!-- canvas-slot-start-' . $layoutUuid . '/column-one -->', $slotData['column-one']);
    $this->assertStringContainsString('<!-- canvas-slot-end-' . $layoutUuid . '/column-one -->', $slotData['column-one']);

    // Verify nested button wrapper comments inside slot.
    $this->assertStringContainsString('<!-- canvas-start-' . $buttonUuid . ' -->', $slotData['column-one']);
    $this->assertStringContainsString('<!-- canvas-end-' . $buttonUuid . ' -->', $slotData['column-one']);

    // Verify correct nesting order.
    $this->assertMatchesRegularExpression(
      '/<!-- canvas-slot-start-' . preg_quote($layoutUuid) . '\/column-one -->.*<!-- canvas-start-' . preg_quote($buttonUuid) . ' -->.*<!-- canvas-end-' . preg_quote($buttonUuid) . ' -->.*<!-- canvas-slot-end-' . preg_quote($layoutUuid) . '\/column-one -->/s',
      $slotData['column-one']
    );

    // Verify non-preview mode has no annotations.
    $renderableNonPreview = $field_items->toRenderable($node, FALSE);
    $htmlNonPreview = (string) $renderer->renderInIsolation($renderableNonPreview);
    $this->assertStringNotContainsString('canvas-start-', $htmlNonPreview);
    $this->assertStringNotContainsString('canvas-slot-start-', $htmlNonPreview);
  }

  /**
   * Test Custom Elements variant rendering via Canvas field pipeline.
   *
   * Tests a component tree with nested slot component rendered through
   * Canvas field rendering logic using Custom Elements variant with nuxt
   * preview provider.
   */
  public function testCustomElementsVariantRendering(): void {
    // Register components with custom elements preview variant.
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';
    $manager->register($fixtureUrl, [
      'custom_elements_preview' => [
        'preview_provider' => 'nuxt',
        'base_url' => 'http://localhost:3000',
      ],
    ]);

    $renderer = \Drupal::service('renderer');

    // Create component tree: TwoColumnLayout with TestButton in column-one.
    $buttonUuid = 'ce-nested-button';
    $layoutUuid = 'ce-parent-layout';

    $componentTree = [
      // TwoColumnLayout as root component.
      [
        'uuid' => $layoutUuid,
        'component_id' => 'extjs.two_column_layout',
        'component_version' => 'active',
        'inputs' => [
          'width' => [
            'sourceType' => 'static:field_item:integer',
            'value' => 75,
            'expression' => 'ℹ︎integer␟value',
          ],
        ],
      ],
      // TestButton in column-one slot.
      [
        'uuid' => $buttonUuid,
        'component_id' => 'extjs.test_button',
        'component_version' => 'active',
        'parent_uuid' => $layoutUuid,
        'slot' => 'column-one',
        'inputs' => [
          'label' => [
            'sourceType' => 'static:field_item:string',
            'value' => 'CE Nested',
            'expression' => 'ℹ︎string␟value',
          ],
        ],
      ],
    ];

    $node = \Drupal::entityTypeManager()
      ->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => 'CE Test Page',
        'field_component_tree' => $componentTree,
      ]);
    $node->save();

    // Render in preview mode via Canvas field pipeline.
    /** @var \Drupal\canvas\Plugin\Field\FieldType\ComponentTreeItemList $field_items */
    $field_items = $node->get('field_component_tree');
    $renderable = $field_items->toRenderable($node, TRUE);
    $html = (string) $renderer->renderInIsolation($renderable);

    // Verify layout component wrapper comments.
    $this->assertStringContainsString('<!-- canvas-start-' . $layoutUuid . ' -->', $html);
    $this->assertStringContainsString('<!-- canvas-end-' . $layoutUuid . ' -->', $html);

    // Extract slot data from data-component-slots attribute.
    preg_match('/data-component-slots="([^"]*)"/', $html, $matches);
    $this->assertNotEmpty($matches);
    $slotDataJson = html_entity_decode($matches[1], ENT_QUOTES | ENT_HTML5);
    $slotData = json_decode($slotDataJson, TRUE);
    $this->assertArrayHasKey('column-one', $slotData);

    // Verify slot marker comments wrap slot content.
    $this->assertStringContainsString('<!-- canvas-slot-start-' . $layoutUuid . '/column-one -->', $slotData['column-one']);
    $this->assertStringContainsString('<!-- canvas-slot-end-' . $layoutUuid . '/column-one -->', $slotData['column-one']);

    // Verify nested button wrapper comments inside slot.
    $this->assertStringContainsString('<!-- canvas-start-' . $buttonUuid . ' -->', $slotData['column-one']);
    $this->assertStringContainsString('<!-- canvas-end-' . $buttonUuid . ' -->', $slotData['column-one']);

    // Verify correct nesting order.
    $this->assertMatchesRegularExpression(
      '/<!-- canvas-slot-start-' . preg_quote($layoutUuid) . '\/column-one -->.*<!-- canvas-start-' . preg_quote($buttonUuid) . ' -->.*<!-- canvas-end-' . preg_quote($buttonUuid) . ' -->.*<!-- canvas-slot-end-' . preg_quote($layoutUuid) . '\/column-one -->/s',
      $slotData['column-one']
    );

    // Verify non-preview mode has no annotations.
    $renderableNonPreview = $field_items->toRenderable($node, FALSE);
    $htmlNonPreview = (string) $renderer->renderInIsolation($renderableNonPreview);
    $this->assertStringNotContainsString('canvas-start-', $htmlNonPreview);
    $this->assertStringNotContainsString('canvas-slot-start-', $htmlNonPreview);
  }

  /**
   * Tests rendering with auto provider detection.
   */
  public function testAutoProviderRendering(): void {
    // Create a component with auto provider.
    $component = ComponentEntity::create([
      'id' => 'extjs.test_auto_button',
      'label' => 'Test Auto Button',
      'category' => 'Test',
      'source' => 'extjs',
      'provider' => NULL,
      'source_local_id' => 'TestAutoButton',
      'active_version' => 'v1',
      'versioned_properties' => [
        VersionedConfigEntityBase::ACTIVE_VERSION => [
          'settings' => [
            'label' => 'Test Auto Button',
            'local_source_id' => 'TestAutoButton',
            'props' => ['type' => 'object', 'properties' => []],
            'slots' => [],
            'prop_field_definitions' => [],
            'custom_elements_preview' => [
              'preview_provider' => 'auto',
            ],
          ],
        ],
      ],
      'status' => TRUE,
    ]);
    // Ensure prop field definitions.
    ExternalJavaScriptComponent::ensurePropFieldDefinitions($component);
    $component->save();

    $settings = $component->getSettings();
    $this->assertEquals('auto', $settings['custom_elements_preview']['preview_provider']);

    // Render the component.
    $source = $component->getComponentSource();
    $build = $source->renderComponent(['props' => []], [], 'test-uuid', TRUE);

    $this->assertEquals('auto', $build['#extjs_preview_config']['preview_provider']);

    // In test environment, auto detection falls back to 'markup' provider.
    $renderer = \Drupal::service('renderer');
    $html = (string) $renderer->renderRoot($build);

    // Should use markup provider.
    $this->assertStringContainsString('custom-elements-preview--markup', $html);
  }

  /**
   * Tests rendering with explicit nuxt provider and base URL.
   */
  public function testNuxtProviderWithBaseUrl(): void {
    // Create a component with nuxt provider.
    $component = ComponentEntity::create([
      'id' => 'extjs.test_nuxt_card',
      'label' => 'Test Nuxt Card',
      'category' => 'Test',
      'source' => 'extjs',
      'provider' => NULL,
      'source_local_id' => 'TestNuxtCard',
      'active_version' => 'v1',
      'versioned_properties' => [
        VersionedConfigEntityBase::ACTIVE_VERSION => [
          'settings' => [
            'label' => 'Test Nuxt Card',
            'local_source_id' => 'TestNuxtCard',
            'props' => ['type' => 'object', 'properties' => []],
            'slots' => [],
            'prop_field_definitions' => [],
            'custom_elements_preview' => [
              'preview_provider' => 'nuxt',
              'base_url' => 'http://localhost:3000',
            ],
          ],
        ],
      ],
      'status' => TRUE,
    ]);
    // Ensure prop field definitions.
    ExternalJavaScriptComponent::ensurePropFieldDefinitions($component);
    $component->save();

    $settings = $component->getSettings();
    $this->assertEquals('nuxt', $settings['custom_elements_preview']['preview_provider']);
    $this->assertEquals('http://localhost:3000', $settings['custom_elements_preview']['base_url']);

    // Render the component.
    $source = $component->getComponentSource();
    $build = $source->renderComponent(['props' => []], [], 'test-uuid-2', TRUE);

    $this->assertEquals('nuxt', $build['#extjs_preview_config']['preview_provider']);
    $this->assertEquals('http://localhost:3000', $build['#extjs_preview_config']['base_url']);

    // Render to verify nuxt provider is used.
    $renderer = \Drupal::service('renderer');
    $html = (string) $renderer->renderRoot($build);

    // Should use nuxt provider (which creates nuxt-preview-container).
    $this->assertStringContainsString('nuxt-preview-container', $html);
    // The base URL is used internally by the provider but isn't output in HTML.
    // Just verify the nuxt provider was instantiated and used.
  }

}
