<?php

declare(strict_types=1);

namespace Drupal\canvas_extjs;

use Drupal\canvas\Entity\VersionedConfigEntityBase;
use Drupal\canvas_extjs\Plugin\Canvas\ComponentSource\ExternalJavaScriptComponent;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use JsonSchema\Constraints\Constraint;
use JsonSchema\Validator;

/**
 * Service for managing component index operations.
 *
 * Handles validation, fetching, and registration of external JavaScript
 * components from component index files.
 */
class ComponentIndexManager {

  use CamelCaseSnakeCaseConverterTrait;

  /**
   * The JSON schema validator.
   *
   * @var \JsonSchema\Validator
   */
  protected Validator $validator;

  /**
   * Path to the component index schema file.
   *
   * @var string
   */
  protected string $schemaPath;

  /**
   * Constructs a ComponentIndexManager.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param string|null $schemaPath
   *   Optional path to component index schema file.
   *   Defaults to the bundled schema.
   */
  public function __construct(
    protected ClientInterface $httpClient,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    ?string $schemaPath = NULL,
  ) {
    $this->validator = new Validator();
    $this->schemaPath = $schemaPath ?? __DIR__ . '/../schema/component-index.schema.json';
  }

  /**
   * Validates component index data against the JSON schema.
   *
   * @param array $data
   *   The component index data to validate.
   *
   * @throws \RuntimeException
   *   If validation fails.
   */
  public function validate(array $data): void {
    // Load the schema.
    if (!file_exists($this->schemaPath)) {
      throw new \RuntimeException(
        sprintf('Component index schema file not found at: %s', $this->schemaPath)
      );
    }

    $schemaContent = file_get_contents($this->schemaPath);

    try {
      $schema = json_decode($schemaContent, flags: JSON_THROW_ON_ERROR);
    }
    catch (\JsonException $e) {
      throw new \RuntimeException(
        sprintf('Invalid schema JSON: %s', $e->getMessage()),
        0,
        $e
      );
    }

    // Convert array to object (JSON Schema validator expects objects).
    $dataObject = Validator::arrayToObjectRecursive($data);

    // Fix empty properties arrays that should be objects.
    // (like Drupal core does).
    foreach ($dataObject->components ?? [] as $component) {
      if (isset($component->props->properties)
        && is_array($component->props->properties)
        && $component->props->properties === []) {
        $component->props->properties = new \stdClass();
      }
    }

    // Validate.
    $this->validator->validate(
      $dataObject,
      $schema,
      Constraint::CHECK_MODE_COERCE_TYPES
    );

    // Throw exception if validation failed.
    if (!$this->validator->isValid()) {
      $errorMessages = array_map(
        fn($error) => sprintf('[%s] %s', $error['property'], $error['message']),
        $this->validator->getErrors()
      );
      throw new \RuntimeException(
        sprintf("Validation failed:\n%s", implode("\n", $errorMessages))
      );
    }
  }

  /**
   * Fetches component index data from a URL or local file path.
   *
   * @param string $source
   *   URL (http/https) or local file path to the component index JSON.
   *
   * @return array
   *   The parsed and validated component index data.
   *
   * @throws \RuntimeException
   *   If the source cannot be fetched, JSON is invalid, or validation fails.
   */
  public function fetch(string $source): array {
    // Fetch the content.
    if (UrlHelper::isExternal($source)) {
      $content = $this->fetchFromUrl($source);
    }
    else {
      $content = $this->fetchFromFile($source);
    }

    // Parse JSON.
    try {
      $data = json_decode($content, associative: TRUE, flags: JSON_THROW_ON_ERROR);
    }
    catch (\JsonException $e) {
      throw new \RuntimeException(
        sprintf('Invalid JSON in component index: %s', $e->getMessage()),
        0,
        $e
      );
    }

    $this->validate($data);

    return $data;
  }

  /**
   * Fetches content from a URL.
   *
   * @param string $url
   *   The URL to fetch.
   *
   * @return string
   *   The fetched content.
   *
   * @throws \RuntimeException
   *   If the URL cannot be fetched.
   */
  protected function fetchFromUrl(string $url): string {
    try {
      $response = $this->httpClient->request('GET', $url);
      return (string) $response->getBody();
    }
    catch (GuzzleException $e) {
      throw new \RuntimeException(
        sprintf('Failed to fetch component index from URL %s: %s', $url, $e->getMessage()),
        0,
        $e
      );
    }
  }

  /**
   * Fetches content from a local file.
   *
   * @param string $path
   *   The file path.
   *
   * @return string
   *   The file contents.
   *
   * @throws \RuntimeException
   *   If the file cannot be read.
   */
  protected function fetchFromFile(string $path): string {
    if (!file_exists($path)) {
      throw new \RuntimeException(
        sprintf('Component index file not found: %s', $path)
      );
    }

    $content = file_get_contents($path);
    if ($content === FALSE) {
      throw new \RuntimeException(
        sprintf('Failed to read component index file: %s', $path)
      );
    }

    return $content;
  }

  /**
   * Registers components from a component index source.
   *
   * Creates or updates ComponentEntity instances for each component in the
   * index. Stores source URL in settings for later re-sync or removal.
   *
   * @param string $source
   *   URL or file path to the component index.
   * @param array $options
   *   Registration options:
   *   - custom_elements_preview: array with:
   *     - preview_provider: string ('auto' or ID of a specific provider)
   *     - base_url: string|null (optional) The base URL to use with a specific
   *       provider. Required when a specific provider is given.
   *   - javascript: string[] URLs.
   *   When no options are specified, the "auto" preview-provider of custom
   *   elements is enabled by default.
   *
   * @return array
   *   Summary of registration operation with keys:
   *   - created: Array of component IDs that were created.
   *   - updated: Array of component IDs that were updated.
   *   - errors: Array of error messages for components that failed.
   *
   * @throws \RuntimeException
   *   If the source cannot be fetched or is invalid.
   */
  public function register(string $source, array $options = []): array {
    $data = $this->fetch($source);

    // Set default if no rendering config provided.
    if (empty($options)) {
      $options = ['custom_elements_preview' => ['preview_provider' => 'auto']];
    }

    // Validate rendering config.
    $this->validateRenderingConfig($options);

    // Check custom_elements module requirement.
    if (isset($options['custom_elements_preview'])) {
      if (!$this->moduleHandler->moduleExists('custom_elements')) {
        throw new \RuntimeException(
          'custom_elements module is required for custom_elements_preview rendering'
        );
      }
    }

    $result = [
      'created' => [],
      'updated' => [],
      'errors' => [],
    ];

    $storage = $this->entityTypeManager->getStorage('component');
    // Reset cache to ensure we get latest data.
    $storage->resetCache();

    foreach ($data['components'] as $componentData) {
      try {
        $componentId = 'extjs.' . $this->convertToSnakeCase($componentData['id']);

        // Load existing or create new.
        /** @var \Drupal\canvas\Entity\ComponentInterface|null $component */
        $component = $storage->load($componentId);
        $isNew = $component === NULL;

        // Prepare settings.
        $settings = [
          'label' => $componentData['name'],
          'local_source_id' => $componentData['id'],
          'props' => $componentData['props'] ?? ['type' => 'object', 'properties' => []],
          'slots' => $componentData['slots'] ?? [],
          'source_url' => $source,
          'prop_field_definitions' => [],
        ];

        // Add rendering config.
        $settings = array_merge($settings, $options);

        // Common logic for both create and update: generate version hash.
        // Hash is based on settings AFTER prop field definitions are added.
        /** @var \Drupal\canvas\Entity\ComponentInterface $tempComponent */
        $tempComponent = $storage->create([
          'id' => '__temp_' . $componentId,
          'label' => $componentData['name'],
          'category' => $componentData['category'],
          'source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID,
          'source_local_id' => $componentData['id'],
          'active_version' => VersionedConfigEntityBase::ACTIVE_VERSION,
          'versioned_properties' => [
            VersionedConfigEntityBase::ACTIVE_VERSION => ['settings' => $settings],
          ],
        ]);
        ExternalJavaScriptComponent::ensurePropFieldDefinitions($tempComponent);
        $settingsWithFieldDefs = $tempComponent->getSettings();

        // Generate version hash.
        $componentSource = $tempComponent->getComponentSource();
        $version = $componentSource->generateVersionHash();

        if ($isNew) {
          $component = $storage->create([
            'id' => $componentId,
            'label' => $componentData['name'],
            'category' => $componentData['category'],
            'source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID,
            'provider' => NULL,
            'source_local_id' => $componentData['id'],
            'active_version' => $version,
            'versioned_properties' => [
              VersionedConfigEntityBase::ACTIVE_VERSION => ['settings' => $settingsWithFieldDefs],
            ],
            'status' => TRUE,
          ]);
          $component->save();
        }
        else {
          $component
            ->set('label', $componentData['name'])
            ->set('category', $componentData['category'])
            ->createVersion($version)
            ->setSettings($settingsWithFieldDefs);
          $component->save();
        }

        if ($isNew) {
          $result['created'][] = $componentId;
        }
        else {
          $result['updated'][] = $componentId;
        }
      }
      catch (\Exception $e) {
        $result['errors'][] = sprintf(
          'Failed to register component %s: %s',
          $componentData['id'] ?? 'unknown',
          $e->getMessage()
        );
      }
    }

    return $result;
  }

  /**
   * Validates rendering configuration.
   *
   * Ensures only one variant is configured and validates requirements.
   *
   * @param array $config
   *   Configuration to validate.
   *
   * @throws \InvalidArgumentException
   *   If configuration is invalid.
   */
  protected function validateRenderingConfig(array $config): void {
    $hasCustomElementsPreview = isset($config['custom_elements_preview']);
    $hasJavaScript = isset($config['javascript']);

    // Ensure exactly one rendering variant is configured.
    if ($hasCustomElementsPreview && $hasJavaScript) {
      throw new \InvalidArgumentException(
        'Only one rendering variant allowed per component'
      );
    }

    if (!$hasCustomElementsPreview && !$hasJavaScript) {
      throw new \InvalidArgumentException(
        'A rendering variant must be configured: custom_elements_preview or javascript'
      );
    }

    // Validate base_url requirement for non-auto providers.
    if ($hasCustomElementsPreview && is_array($config['custom_elements_preview'])) {
      $provider = $config['custom_elements_preview']['preview_provider'] ?? NULL;
      $baseUrl = $config['custom_elements_preview']['base_url'] ?? NULL;
      if ($provider !== 'auto' && empty($baseUrl)) {
        throw new \InvalidArgumentException(
          'base_url is required when using a specific preview_provider (non-auto)'
        );
      }
    }
  }

  /**
   * Unregisters all components from a given source.
   *
   * Deletes all ComponentEntity instances that were registered from the
   * specified source URL or file path.
   *
   * @param string $source
   *   The source URL or file path that was used during registration.
   *
   * @return array
   *   Summary with keys:
   *   - deleted: Array of component IDs that were deleted.
   *   - errors: Array of error messages.
   */
  public function unregister(string $source): array {
    $result = [
      'deleted' => [],
      'errors' => [],
    ];

    $storage = $this->entityTypeManager->getStorage('component');

    // Find all extjs components.
    $components = $storage->loadByProperties(['source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID]);

    foreach ($components as $component) {
      try {
        /** @var \Drupal\canvas\Entity\ComponentInterface $component */
        $component->loadVersion($component->getActiveVersion());
        $settings = $component->getSettings();

        // Check if this component was registered from the given source.
        if (isset($settings['source_url']) && $settings['source_url'] === $source) {
          $componentId = $component->id();
          $component->delete();
          $result['deleted'][] = $componentId;
        }
      }
      catch (\Exception $e) {
        $result['errors'][] = sprintf(
          'Failed to unregister component %s: %s',
          $component->id(),
          $e->getMessage()
        );
      }
    }

    return $result;
  }

}
