/**
 * @file
 * Test component bundle for Canvas ExtJS JavaScript rendering variant.
 *
 * This bundle provides simplified implementations of TestButton and
 * TwoColumnLayout for testing the JavaScript rendering pipeline.
 *
 * Contract: Must export a render() function that:
 * - Accepts (container, componentName, props, slots)
 * - Returns container element on success, null if component not found
 * - Clears container and renders component
 */

/**
 * Renders a test button component.
 *
 * @param {HTMLElement} container
 *   The container element.
 * @param {Object} props
 *   Component prop values (e.g., {label: 'Click me', variant: 'primary'}).
 * @param {Object.<string, string>} slots
 *   Component slots as HTML markup strings (e.g., {default: '<p>Content</p>'}).
 *
 * @return {HTMLElement}
 *   The container element.
 */
function renderTestButton(container, props, slots) {
  container.innerHTML = '';

  const button = document.createElement('button');
  button.className = `test-button test-button--${props.variant || 'primary'}`;
  button.textContent = props.label || 'Click me';
  button.disabled = props.disabled || false;

  if (props.size) {
    button.classList.add(`test-button--${props.size}`);
  }

  if (slots.default) {
    button.innerHTML = slots.default;
  }

  container.appendChild(button);
  return container;
}

/**
 * Renders a two-column layout component.
 *
 * @param {HTMLElement} container
 *   The container element.
 * @param {Object} props
 *   Component prop values (e.g., {width: 33}).
 * @param {Object.<string, string>} slots
 *   Component slots as HTML markup strings (e.g., {'column-one': '<p>...</p>'}).
 *
 * @return {HTMLElement}
 *   The container element.
 */
function renderTwoColumnLayout(container, props, slots) {
  container.innerHTML = '';

  const layout = document.createElement('div');
  layout.className = 'two-column-layout';
  layout.style.display = 'flex';

  const col1 = document.createElement('div');
  col1.className = 'column column-one';
  col1.style.width = `${props.width || 50}%`;
  col1.innerHTML = slots['column-one'] || '';

  const col2 = document.createElement('div');
  col2.className = 'column column-two';
  col2.style.width = `${100 - (props.width || 50)}%`;
  col2.innerHTML = slots['column-two'] || '';

  layout.appendChild(col1);
  layout.appendChild(col2);
  container.appendChild(layout);
  return container;
}

const renderers = {
  renderTestButton,
  renderTwoColumnLayout,
};

/**
 * Main render function.
 *
 * @param {HTMLElement} container
 *   The container element to render into.
 * @param {string} componentName
 *   The component name (PascalCase).
 * @param {Object} props
 *   Component prop values as key-value pairs.
 * @param {Object.<string, string>} slots
 *   Component slots as HTML markup strings.
 *
 * @return {HTMLElement|null}
 *   The container element on success, null if component not found.
 */
export default async function render(container, componentName, props, slots) {
  return renderers[`render${componentName}`]?.(container, props, slots) ?? null;
}
